/*******************************************************************************
 * frame.h
 *
 * This header file is included by all C modules in POV-Ray. It defines all
 * globally-accessible types and constants.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2009 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/frame.h $
 * $Revision: 1.3 $
 * $Change: 4715 $
 * $DateTime: 2009/02/21 10:00:39 $
 * $Author: christian $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *      unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#ifndef FRAME_H
#define FRAME_H

// Generic header for all modules 

#include <new>

#include <stdio.h>
#include <string.h>
#include <limits.h>

#include <vector>
#include <stack>

#ifndef MATH_H_INCLUDED
#include <math.h>
#endif

#include "base/configbase.h"
#include "base/types.h"
#include "base/fileinputoutput.h"

#include "backend/configbackend.h"
#include "backend/support/simplevector.h"
#include "backend/support/randomsequences.h"
#include "backend/control/messagefactory.h"

#include "pov_mem.h"

namespace pov
{

using namespace std;

/*****************************************************************************
 *
 * Typedefs that need to be known here.
 *
 *****************************************************************************/

class ObjectBase;
typedef ObjectBase * ObjectPtr;

class CompoundObject;


/*****************************************************************************
 *
 * Scalar, color and vector stuff.
 *
 *****************************************************************************/

typedef DBL UV_VECT[2];
typedef DBL VECTOR[3];
typedef DBL VECTOR_4D[4];
typedef DBL MATRIX[4][4];
typedef DBL EXPRESS[5];
typedef COLC COLOUR[5];
typedef COLC RGB[3];
typedef SNGL SNGL_VECT[3];

// Vector array elements. 
enum
{
	U = 0,
	V = 1
};

enum
{
	X = 0,
	Y = 1,
	Z = 2,
	T = 3,
	W = 3
};

// Color array elements. 
enum
{
	pRED    = 0,
	pGREEN  = 1,
	pBLUE   = 2,
	pFILTER = 3,
	pTRANSM = 4
};

// Macros to manipulate scalars, vectors, and colors. 

inline void Assign_Vector(VECTOR d, const VECTOR s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_Vector(VECTOR d, const SNGL_VECT s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_Vector(SNGL_VECT d, const VECTOR s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_Vector(SNGL_VECT d, const SNGL_VECT s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_UV_Vect(UV_VECT d, UV_VECT s)
{
	d[X] = s[X];
	d[Y] = s[Y];
}

inline void Assign_Vector_4D(VECTOR_4D d, VECTOR_4D s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
	d[T] = s[T];
}

inline void Assign_Colour(COLOUR d, const COLOUR s)
{
	d[pRED] = s[pRED];
	d[pGREEN] = s[pGREEN];
	d[pBLUE] = s[pBLUE];
	d[pFILTER] = s[pFILTER];
	d[pTRANSM] = s[pTRANSM];
}

inline void Assign_RGB(RGB d, const RGB s)
{
	d[pRED] = s[pRED];
	d[pGREEN] = s[pGREEN];
	d[pBLUE] = s[pBLUE];
}

inline void Assign_Colour_Express(COLOUR d, EXPRESS s)
{
	d[pRED] = s[pRED];
	d[pGREEN] = s[pGREEN];
	d[pBLUE] = s[pBLUE];
	d[pFILTER] = s[pFILTER];
	d[pTRANSM] = s[pTRANSM];
}

inline void Assign_Express(EXPRESS d, EXPRESS s)
{
	d[pRED] = s[pRED];
	d[pGREEN] = s[pGREEN];
	d[pBLUE] = s[pBLUE];
	d[pFILTER] = s[pFILTER];
	d[pTRANSM] = s[pTRANSM];
}

inline void Make_Colour(COLOUR c, COLC r, COLC g, COLC b)
{
	c[pRED] = r;
	c[pGREEN] = g;
	c[pBLUE] = b;
	c[pFILTER] = 0.0;
	c[pTRANSM] = 0.0;
}

inline void Make_ColourA(COLOUR c, COLC r, COLC g, COLC b, COLC a, COLC t)
{
	c[pRED] = r;
	c[pGREEN] = g;
	c[pBLUE] = b;
	c[pFILTER] = a;
	c[pTRANSM] = t;
}

inline void Make_Vector(VECTOR v, DBL a, DBL b, DBL c)
{
	v[X] = a;
	v[Y] = b;
	v[Z] = c;
}

inline void Make_Vector(SNGL_VECT v, SNGL a, SNGL b, SNGL c)
{
	v[X] = a;
	v[Y] = b;
	v[Z] = c;
}

inline void Make_UV_Vector(UV_VECT v, DBL a, DBL b)
{
	v[U] = a;
	v[V] = b;
}

inline void Make_RGB(RGB c, COLC r, COLC g, COLC b)
{
	c[pRED] = r;
	c[pGREEN] = g;
	c[pBLUE] = b;
}

inline void Destroy_Float(DBL *x)
{
	if(x != NULL)
		POV_FREE(x);
}

inline void Destroy_Vector(VECTOR *x)
{
	if(x != NULL)
		POV_FREE(x);
}

inline void Destroy_UV_Vect(UV_VECT *x)
{
	if(x != NULL)
		POV_FREE(x);
}

inline void Destroy_Vector_4D(VECTOR_4D *x)
{
	if(x != NULL)
		POV_FREE(x);
}

inline void Destroy_Colour(COLOUR *x)
{
	if(x != NULL)
		POV_FREE(x);
}

#if 0
#pragma mark * Vector2d
#endif

class Vector2d
{
	public:
		Vector2d()
		{
			vect[X] = 0.0;
			vect[Y] = 0.0;
		}

		explicit Vector2d(DBL d)
		{
			vect[X] = d;
			vect[Y] = d;
		}

		Vector2d(DBL x, DBL y)
		{
			vect[X] = x;
			vect[Y] = y;
		}

		explicit Vector2d(const UV_VECT vi)
		{
			vect[X] = vi[X];
			vect[Y] = vi[Y];
		}

		explicit Vector2d(const SNGL_VECT vi)
		{
			vect[X] = DBL(vi[X]);
			vect[Y] = DBL(vi[Y]);
		}

		Vector2d(const Vector2d& b)
		{
			vect[X] = b[X];
			vect[Y] = b[Y];
		}

		Vector2d& operator=(const Vector2d& b)
		{
			vect[X] = b[X];
			vect[Y] = b[Y];
			return *this;
		}

		DBL operator[](int idx) const { return vect[idx]; }
		DBL& operator[](int idx) { return vect[idx]; }

		Vector2d operator+(const Vector2d& b) const
		{
			return Vector2d(vect[X] + b[X], vect[Y] + b[Y]);
		}

		Vector2d operator-(const Vector2d& b) const
		{
			return Vector2d(vect[X] - b[X], vect[Y] - b[Y]);
		}

		Vector2d operator*(const Vector2d& b) const
		{
			return Vector2d(vect[X] * b[X], vect[Y] * b[Y]);
		}

		Vector2d operator/(const Vector2d& b) const
		{
			return Vector2d(vect[X] / b[X], vect[Y] / b[Y]);
		}

		Vector2d& operator+=(const Vector2d& b)
		{
			vect[X] += b[X];
			vect[Y] += b[Y];
			return *this;
		}

		Vector2d& operator-=(const Vector2d& b)
		{
			vect[X] -= b[X];
			vect[Y] -= b[Y];
			return *this;
		}

		Vector2d& operator*=(const Vector2d& b)
		{
			vect[X] *= b[X];
			vect[Y] *= b[Y];
			return *this;
		}

		Vector2d& operator/=(const Vector2d& b)
		{
			vect[X] /= b[X];
			vect[Y] /= b[Y];
			return *this;
		}

		Vector2d operator-() const
		{
			return Vector2d(-vect[X], -vect[Y]);
		}

		Vector2d operator+(DBL b) const
		{
			return Vector2d(vect[X] + b, vect[Y] + b);
		}

		Vector2d operator-(DBL b) const
		{
			return Vector2d(vect[X] - b, vect[Y] - b);
		}

		Vector2d operator*(DBL b) const
		{
			return Vector2d(vect[X] * b, vect[Y] * b);
		}

		Vector2d operator/(DBL b) const
		{
			return Vector2d(vect[X] / b, vect[Y] / b);
		}

		Vector2d& operator+=(DBL b)
		{
			vect[X] += b;
			vect[Y] += b;
			return *this;
		}

		Vector2d& operator-=(DBL b)
		{
			vect[X] -= b;
			vect[Y] -= b;
			return *this;
		}

		Vector2d& operator*=(DBL b)
		{
			vect[X] *= b;
			vect[Y] *= b;
			return *this;
		}

		Vector2d& operator/=(DBL b)
		{
			vect[X] /= b;
			vect[Y] /= b;
			return *this;
		}

		const UV_VECT& operator*() const { return vect; }
		UV_VECT& operator*() { return vect; }

		DBL x() const { return vect[X]; }
		DBL& x() { return vect[X]; }

		DBL y() const { return vect[Y]; }
		DBL& y() { return vect[Y]; }

		DBL u() const { return vect[X]; }
		DBL& u() { return vect[X]; }

		DBL v() const { return vect[Y]; }
		DBL& v() { return vect[Y]; }

		DBL length() const
		{
			return sqrt(vect[X] * vect[X] + vect[Y] * vect[Y]);
		}
		DBL lengthSqr() const
		{
			return vect[X] * vect[X] + vect[Y] * vect[Y];
		}
		Vector2d normalized() const
		{
			DBL l = length();
			if (l != 0)
				return *this / l;
			else
				return *this;
		}
		void normalize()
		{
			DBL l = length();
			if (l != 0)
				*this /= l;
			// no else
		}

	private:
		DBL vect[2];
};

#if 0
#pragma mark * Vector3d
#endif

class Vector3d
{
	public:

		Vector3d()
		{
			vect[X] = 0.0;
			vect[Y] = 0.0;
			vect[Z] = 0.0;
		}

		explicit Vector3d(DBL d)
		{
			vect[X] = d;
			vect[Y] = d;
			vect[Z] = d;
		}

		Vector3d(DBL x, DBL y, DBL z)
		{
			vect[X] = x;
			vect[Y] = y;
			vect[Z] = z;
		}

		explicit Vector3d(const VECTOR vi)
		{
			vect[X] = vi[X];
			vect[Y] = vi[Y];
			vect[Z] = vi[Z];
		}

		explicit Vector3d(const SNGL_VECT vi)
		{
			vect[X] = DBL(vi[X]);
			vect[Y] = DBL(vi[Y]);
			vect[Z] = DBL(vi[Z]);
		}

		Vector3d(const Vector3d& b)
		{
			vect[X] = b[X];
			vect[Y] = b[Y];
			vect[Z] = b[Z];
		}

		Vector3d& operator=(const Vector3d& b)
		{
			vect[X] = b[X];
			vect[Y] = b[Y];
			vect[Z] = b[Z];
			return *this;
		}

		DBL operator[](int idx) const { return vect[idx]; }
		DBL& operator[](int idx) { return vect[idx]; }

		Vector3d operator+(const Vector3d& b) const
		{
			return Vector3d(vect[X] + b[X], vect[Y] + b[Y], vect[Z] + b[Z]);
		}

		Vector3d operator-(const Vector3d& b) const
		{
			return Vector3d(vect[X] - b[X], vect[Y] - b[Y], vect[Z] - b[Z]);
		}

		Vector3d operator*(const Vector3d& b) const
		{
			return Vector3d(vect[X] * b[X], vect[Y] * b[Y], vect[Z] * b[Z]);
		}

		Vector3d operator/(const Vector3d& b) const
		{
			return Vector3d(vect[X] / b[X], vect[Y] / b[Y], vect[Z] / b[Z]);
		}

		Vector3d& operator+=(const Vector3d& b)
		{
			vect[X] += b[X];
			vect[Y] += b[Y];
			vect[Z] += b[Z];
			return *this;
		}

		Vector3d& operator-=(const Vector3d& b)
		{
			vect[X] -= b[X];
			vect[Y] -= b[Y];
			vect[Z] -= b[Z];
			return *this;
		}

		Vector3d& operator*=(const Vector3d& b)
		{
			vect[X] *= b[X];
			vect[Y] *= b[Y];
			vect[Z] *= b[Z];
			return *this;
		}

		Vector3d& operator/=(const Vector3d& b)
		{
			vect[X] /= b[X];
			vect[Y] /= b[Y];
			vect[Z] /= b[Z];
			return *this;
		}

		Vector3d operator-() const
		{
			return Vector3d(-vect[X], -vect[Y], -vect[Z]);
		}

		Vector3d operator+(DBL b) const
		{
			return Vector3d(vect[X] + b, vect[Y] + b, vect[Z] + b);
		}

		Vector3d operator-(DBL b) const
		{
			return Vector3d(vect[X] - b, vect[Y] - b, vect[Z] - b);
		}

		Vector3d operator*(DBL b) const
		{
			return Vector3d(vect[X] * b, vect[Y] * b, vect[Z] * b);
		}

		Vector3d operator/(DBL b) const
		{
			return Vector3d(vect[X] / b, vect[Y] / b, vect[Z] / b);
		}

		Vector3d& operator+=(DBL b)
		{
			vect[X] += b;
			vect[Y] += b;
			vect[Z] += b;
			return *this;
		}

		Vector3d& operator-=(DBL b)
		{
			vect[X] -= b;
			vect[Y] -= b;
			vect[Z] -= b;
			return *this;
		}

		Vector3d& operator*=(DBL b)
		{
			vect[X] *= b;
			vect[Y] *= b;
			vect[Z] *= b;
			return *this;
		}

		Vector3d& operator/=(DBL b)
		{
			vect[X] /= b;
			vect[Y] /= b;
			vect[Z] /= b;
			return *this;
		}

		const VECTOR& operator*() const { return vect; }
		VECTOR& operator*() { return vect; }

		DBL x() const { return vect[X]; }
		DBL& x() { return vect[X]; }

		DBL y() const { return vect[Y]; }
		DBL& y() { return vect[Y]; }

		DBL z() const { return vect[Z]; }
		DBL& z() { return vect[Z]; }

		DBL length() const
		{
			return sqrt(vect[X] * vect[X] + vect[Y] * vect[Y] + vect[Z] * vect[Z]);
		}
		DBL lengthSqr() const
		{
			return vect[X] * vect[X] + vect[Y] * vect[Y] + vect[Z] * vect[Z];
		}
		Vector3d normalized() const
		{
			DBL l = length();
			if (l != 0)
				return *this / l;
			else
				return *this;
		}
		void normalize()
		{
			DBL l = length();
			if (l != 0)
				*this /= l;
			// no else
		}

	private:
		DBL vect[3];
};

inline DBL dot(const Vector2d& a, const Vector2d& b)
{
	return (a.x() * b.x()) + (a.y() * b.y());
}

inline DBL dot(const Vector3d& a, const Vector3d& b)
{
	return ((a.x() * b.x()) + (a.y() * b.y()) + (a.z() * b.z()));
}

inline Vector3d cross(const Vector3d& a, const Vector3d& b)
{
	return Vector3d(((a.y() * b.z()) - (a.z() * b.y())), ((a.z() * b.x()) - (a.x() * b.z())), ((a.x() * b.y()) - (a.y() * b.x())));
}

inline bool similar(const Vector3d& a, const Vector3d& b)
{
	return ( fabs(a.x()-b.x()) + fabs(a.y()-b.y()) + fabs(a.z()-b.z()) < EPSILON );
}

typedef pov_base::Colour Colour;

/*****************************************************************************
 *
 * Bounding box stuff (see also BOUND.H).
 *
 *****************************************************************************/

#if 0
#pragma mark * Bounding
#endif

typedef SNGL BBOX_VAL;

typedef BBOX_VAL BBOX_VECT[3];

typedef struct Bounding_Box_Struct BBOX;

struct Bounding_Box_Struct
{
	union
	{
		BBOX_VECT Lower_Left;
		BBOX_VECT pmin;
		BBOX_VECT lowerleft;
	};
	union
	{
		BBOX_VECT Lengths;
		BBOX_VECT pmax;
		BBOX_VECT length;
	};

	SNGL GetMinX() { return Lower_Left[X]; }
	SNGL GetMinY() { return Lower_Left[Y]; }
	SNGL GetMinZ() { return Lower_Left[Z]; }

	SNGL GetMaxX() { return Lower_Left[X] + Lengths[X]; }
	SNGL GetMaxY() { return Lower_Left[Y] + Lengths[Y]; }
	SNGL GetMaxZ() { return Lower_Left[Z] + Lengths[Z]; }
};

inline void Assign_BBox_Vect(BBOX_VECT& d, const BBOX_VECT s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_BBox_Vect(BBOX_VECT& d, const VECTOR s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Assign_BBox_Vect(VECTOR& d, const BBOX_VECT s)
{
	d[X] = s[X];
	d[Y] = s[Y];
	d[Z] = s[Z];
}

inline void Make_BBox(BBOX& BBox, const BBOX_VAL llx, const BBOX_VAL lly, const BBOX_VAL llz, const BBOX_VAL lex, const BBOX_VAL ley, const BBOX_VAL lez)
{
	BBox.Lower_Left[X] = (BBOX_VAL)(llx);
	BBox.Lower_Left[Y] = (BBOX_VAL)(lly);
	BBox.Lower_Left[Z] = (BBOX_VAL)(llz);
	BBox.Lengths[X] = (BBOX_VAL)(lex);
	BBox.Lengths[Y] = (BBOX_VAL)(ley);
	BBox.Lengths[Z] = (BBOX_VAL)(lez);
}

inline void Make_BBox_from_min_max(BBOX& BBox, const BBOX_VECT mins, const BBOX_VECT maxs)
{
	BBox.Lower_Left[X] = (BBOX_VAL)(mins[X]);
	BBox.Lower_Left[Y] = (BBOX_VAL)(mins[Y]);
	BBox.Lower_Left[Z] = (BBOX_VAL)(mins[Z]);
	BBox.Lengths[X] = (BBOX_VAL)(maxs[X]-mins[X]);
	BBox.Lengths[Y] = (BBOX_VAL)(maxs[Y]-mins[Y]);
	BBox.Lengths[Z] = (BBOX_VAL)(maxs[Z]-mins[Z]);
}

inline void Make_BBox_from_min_max(BBOX& BBox, const VECTOR mins, const VECTOR maxs)
{
	BBox.Lower_Left[X] = (BBOX_VAL)(mins[X]);
	BBox.Lower_Left[Y] = (BBOX_VAL)(mins[Y]);
	BBox.Lower_Left[Z] = (BBOX_VAL)(mins[Z]);
	BBox.Lengths[X] = (BBOX_VAL)(maxs[X]-mins[X]);
	BBox.Lengths[Y] = (BBOX_VAL)(maxs[Y]-mins[Y]);
	BBox.Lengths[Z] = (BBOX_VAL)(maxs[Z]-mins[Z]);
}

inline void Make_min_max_from_BBox(BBOX_VECT& mins, BBOX_VECT& maxs, const BBOX& BBox)
{
	mins[X] = BBox.Lower_Left[X];
	mins[Y] = BBox.Lower_Left[Y];
	mins[Z] = BBox.Lower_Left[Z];
	maxs[X] = mins[X] + BBox.Lengths[X];
	maxs[Y] = mins[Y] + BBox.Lengths[Y];
	maxs[Z] = mins[Z] + BBox.Lengths[Z];
}

inline void Make_min_max_from_BBox(VECTOR& mins, VECTOR& maxs, const BBOX& BBox)
{
	mins[X] = BBox.Lower_Left[X];
	mins[Y] = BBox.Lower_Left[Y];
	mins[Z] = BBox.Lower_Left[Z];
	maxs[X] = mins[X] + BBox.Lengths[X];
	maxs[Y] = mins[Y] + BBox.Lengths[Y];
	maxs[Z] = mins[Z] + BBox.Lengths[Z];
}

inline bool Inside_BBox(const VECTOR point, const BBOX& bbox)
{
	if (point[X] < (DBL)bbox.Lower_Left[X])
		return(false);
	if (point[Y] < (DBL)bbox.Lower_Left[Y])
		return(false);
	if (point[Z] < (DBL)bbox.Lower_Left[Z])
		return(false);
	if (point[X] > (DBL)bbox.Lower_Left[X] + (DBL)bbox.Lengths[X])
		return(false);
	if (point[Y] > (DBL)bbox.Lower_Left[Y] + (DBL)bbox.Lengths[Y])
		return(false);
	if (point[Z] > (DBL)bbox.Lower_Left[Z] + (DBL)bbox.Lengths[Z])
		return(false);

	return(true);
}

/*****************************************************************************
 *
 * Transformation stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Transform
#endif

typedef struct Transform_Struct TRANSFORM;

struct Transform_Struct
{
	MATRIX matrix;
	MATRIX inverse;
};



/*****************************************************************************
 *
 * Color map stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Blend Map
#endif

const int MAX_BLEND_MAP_ENTRIES = 256;

typedef struct Blend_Map_Entry BLEND_MAP_ENTRY;
typedef struct Blend_Map_Struct BLEND_MAP;
typedef struct Pattern_Struct TPATTERN;
typedef struct Texture_Struct TEXTURE;
typedef struct Pigment_Struct PIGMENT;
typedef struct Tnormal_Struct TNORMAL;
typedef struct Finish_Struct FINISH;
typedef struct Turb_Struct TURB;
typedef struct Warps_Struct WARP;
typedef struct Spline_Entry SPLINE_ENTRY;
typedef struct Spline_Struct SPLINE;

struct Blend_Map_Entry
{
	SNGL value;
	unsigned char Same;
	union
	{
		COLOUR Colour;
		PIGMENT *Pigment;
		TNORMAL *Tnormal;
		TEXTURE *Texture;
		UV_VECT Point_Slope;
	} Vals;
};

struct Blend_Map_Struct
{
	int Users;
	short Number_Of_Entries;
	char Transparency_Flag, Type;
	BLEND_MAP_ENTRY *Blend_Map_Entries;
};

inline void Make_Blend_Map_Entry(BLEND_MAP_ENTRY& entry, SNGL v, unsigned char s, COLC r, COLC g, COLC b, COLC a, COLC t)
{
	entry.value = v;
	entry.Same = s;
	Make_ColourA(entry.Vals.Colour, r, g, b, a, t);
}


/*****************************************************************************
 *
 * Media and Interior stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Media, Interior
#endif

class Media
{
	public:
		int Type;
		int Intervals;
		int Min_Samples;
		int Max_Samples;
		unsigned Sample_Method : 8;
		bool is_constant : 1;
		bool use_absorption : 1;
		bool use_emission : 1;
		bool use_extinction : 1;
		bool use_scattering : 1;
		bool ignore_photons : 1;
		DBL Jitter;
		DBL Eccentricity;
		DBL sc_ext;
		Colour Absorption;
		Colour Emission;
		Colour Extinction;
		Colour Scattering;

		DBL Ratio;
		DBL Confidence;
		DBL Variance;
		DBL *Sample_Threshold;

		DBL AA_Threshold;
		int AA_Level;

		PIGMENT *Density;

		Media();
		Media(const Media&);
		~Media();

		Media& operator=(const Media&);

		void Transform(TRANSFORM *trans);

		void PostProcess();
};

class Interior
{
	public:
		int References;
		int  hollow, Disp_NElems;
		SNGL IOR, Dispersion;
		SNGL Caustics, Old_Refract;
		SNGL Fade_Distance, Fade_Power;
		Colour Fade_Colour;
		vector<Media> media;

		Interior();
		Interior(const Interior&);
		~Interior();

		void Transform(TRANSFORM *trans);

		void PostProcess();
	private:
		Interior& operator=(const Interior&);
};



/*****************************************************************************
 *
 * Spline stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Spline
#endif

struct Spline_Entry
{
	DBL par;      // Parameter 
	DBL vec[5];   // Value at the parameter 
	DBL coeff[5]; // Interpolating coefficients at the parameter 
};

struct Spline_Struct
{
	int Number_Of_Entries, Type;
	int Max_Entries;
	SPLINE_ENTRY *SplineEntries;
	int Coeffs_Computed;
	int Terms;
	bool Cache_Valid;
	int Cache_Type;
	DBL Cache_Point;
	EXPRESS Cache_Data;
};

typedef struct Spline_Entry SPLINE_ENTRY;


/*****************************************************************************
 *
 * Image stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Image
#endif

// Legal image attributes. 

#define NO_FILE         0x00000000
#define GIF_FILE        0x00000001
#define POT_FILE        0x00000002
#define SYS_FILE        0x00000004
#define IFF_FILE        0x00000008
#define TGA_FILE        0x00000010
#define GRAD_FILE       0x00000020
#define PGM_FILE        0x00000040
#define PPM_FILE        0x00000080
#define PNG_FILE        0x00000100
#define JPEG_FILE       0x00000200
#define TIFF_FILE       0x00000400
#define BMP_FILE        0x00000800
#define EXR_FILE        0x00001000
#define HDR_FILE        0x00002000

#define IMAGE_FILE_MASK 0x0000FFFF

#define IMAGE_FTYPE     0x00010000
#define HF_FTYPE        0x00020000
#define HIST_FTYPE      0x00040000
#define GRAY_FTYPE      0x00080000
#define NORMAL_FTYPE    0x00100000
#define MATERIAL_FTYPE  0x00200000

#define IS16BITIMAGE    0x80000000
#define IS16GRAYIMAGE   0x80000000

// Image types. 

#define IMAGE_FILE    IMAGE_FTYPE+GIF_FILE+SYS_FILE+IFF_FILE+GRAD_FILE+TGA_FILE+PGM_FILE+PPM_FILE+PNG_FILE+JPEG_FILE+TIFF_FILE+BMP_FILE+EXR_FILE+HDR_FILE
#define NORMAL_FILE   NORMAL_FTYPE+GIF_FILE+SYS_FILE+IFF_FILE+GRAD_FILE+TGA_FILE+PGM_FILE+PPM_FILE+PNG_FILE+JPEG_FILE+TIFF_FILE+BMP_FILE+EXR_FILE+HDR_FILE
#define MATERIAL_FILE MATERIAL_FTYPE+GIF_FILE+SYS_FILE+IFF_FILE+GRAD_FILE+TGA_FILE+PGM_FILE+PPM_FILE+PNG_FILE+JPEG_FILE+TIFF_FILE+BMP_FILE+EXR_FILE+HDR_FILE
#define HF_FILE       HF_FTYPE+GIF_FILE+SYS_FILE+POT_FILE+TGA_FILE+PGM_FILE+PPM_FILE+PNG_FILE+JPEG_FILE+TIFF_FILE+BMP_FILE+EXR_FILE+HDR_FILE

#define PIGMENT_TYPE  0
#define NORMAL_TYPE   1
#define PATTERN_TYPE  2
#define TEXTURE_TYPE  4
#define COLOUR_TYPE   5
#define SLOPE_TYPE    6
#define DENSITY_TYPE  7

#define DEFAULT_FRACTAL_EXTERIOR_TYPE 1
#define DEFAULT_FRACTAL_INTERIOR_TYPE 0
#define DEFAULT_FRACTAL_EXTERIOR_FACTOR 1
#define DEFAULT_FRACTAL_INTERIOR_FACTOR 1


/*****************************************************************************
 *
 * Pigment, Tnormal, Finish, Texture & Warps stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Pigment, Normal, Finish, Texture, Warp
#endif

// extract a noise generator from pattern flags
#define GetNoiseGen(p) (((p)->Flags >> 4) & 0x03)

typedef struct Density_file_Struct DENSITY_FILE;
typedef struct Density_file_Data_Struct DENSITY_FILE_DATA;

struct Density_file_Struct
{
	int Interpolation;
	DENSITY_FILE_DATA *Data;
};

struct Density_file_Data_Struct
{
	int References;
	char *Name;
	size_t Sx, Sy, Sz;
	int Type;
	union
	{
		unsigned char *Density8;
		unsigned short *Density16;
		unsigned int *Density32;
	};
};

class ImageData;
class FunctionVM;

struct Crackle_Cache_Struct
{
  VECTOR data [125];
  bool valid [125];
  int lastSeed;
};

struct Pattern_Struct
{
	unsigned short Type, Wave_Type, Flags;
	int References;
	SNGL Frequency, Phase;
	SNGL Exponent;
	WARP *Warps;
	TPATTERN *Next;
	BLEND_MAP *Blend_Map;
	union {
		DENSITY_FILE *Density_File;
		ImageData *image;
		VECTOR Gradient;
		SNGL Agate_Turb_Scale;
		short Num_of_Waves;
		short Iterations;
		short Arms;
		struct { SNGL Mortar; VECTOR Size; } Brick;
		struct { SNGL Control0, Control1; } Quilted;
		struct { DBL Size, UseCoords, Metric; } Facets; 
		struct { VECTOR Form; DBL Metric; DBL Offset; DBL Dim;
		         short IsSolid; } Crackle;
		struct { VECTOR Slope_Vector, Altit_Vector;
		         short Slope_Base, Altit_Base; DBL Slope_Len,
		         Altit_Len; UV_VECT Slope_Mod, Altit_Mod; } Slope;
		struct { UV_VECT Coord; short Iterations, interior_type,
		         exterior_type; DBL efactor, ifactor; int Exponent; } Fractal;
		struct { void *Fn; unsigned int Data; FunctionVM *vm; } Function;
	    struct { VECTOR AOI_origin; unsigned char pt_fixed; } Aoi;

		PIGMENT *Pigment;
		ObjectBase *Object;
	} Vals;
};

struct Pigment_Struct : public Pattern_Struct
{
  COLOUR Colour; 
  COLOUR Quick_Colour;
};

struct Tnormal_Struct : public Pattern_Struct
{
  SNGL Amount;
  SNGL Delta; // NK delta 
};

struct Texture_Struct : public Pattern_Struct
{
  TEXTURE *Next_Material;
  PIGMENT *Pigment;
  TNORMAL *Tnormal;
  FINISH *Finish;
  TEXTURE *Materials;
  int Num_Of_Mats;

};

struct Finish_Struct
{
  SNGL Diffuse, DiffuseBack, Brilliance;
  SNGL Specular, Roughness;
  SNGL Phong, Phong_Size;
  SNGL Irid, Irid_Film_Thickness, Irid_Turb;
  SNGL Temp_Caustics, Temp_IOR, Temp_Dispersion, Temp_Refract, Reflect_Exp;
  SNGL Crand, Metallic;
  RGB Ambient, Reflection_Max, Reflection_Min;  // Changed by MBP 8/27/98 
  RGB SigmaPrimeS, SigmaA;
  SNGL Reflection_Falloff;  // Added by MBP 8/27/98 
  int Reflection_Type;  // Added by MBP 9/5/98 
  SNGL Reflect_Metallic; // MBP 
  int Conserve_Energy;  // added by NK Dec 19 1999 
  bool UseBSSRDF;   // whether to use the BSSRDF lighting model (the infamous Tariq/Ibarria patch)
};

struct Warps_Struct
{
  unsigned short Warp_Type;
  WARP *Prev_Warp;
  WARP *Next_Warp;
};

struct Turb_Struct : public Warps_Struct
{
  VECTOR Turbulence;
  int Octaves;
  SNGL Lambda, Omega;
};

#define Destroy_Finish(x) if ((x)!=NULL) POV_FREE(x)

typedef struct Material_Struct MATERIAL;

struct Material_Struct
{
   TEXTURE *Texture;
   TEXTURE * Interior_Texture;
   Interior *interior;
};

/*****************************************************************************
 *
 * Object stuff (see also OBJECTS.H and primitive include files).
 *
 *****************************************************************************/

#if 0
#pragma mark * Object
#endif

#define INIT_OBJECT_FIELDS(o,t)   \
  o->Type     = t;                \
  o->Texture  = NULL;             \
  o->Interior_Texture = NULL;     \
  o->interior = NULL;             \
  o->Trans    = NULL;             \
  o->UV_Trans = NULL;             \
  o->Ph_Density = 0;              \
  o->Flags    = 0;                \
  Make_BBox(o->BBox, -BOUND_HUGE/2.0, -BOUND_HUGE/2.0, -BOUND_HUGE/2.0, \
    BOUND_HUGE, BOUND_HUGE, BOUND_HUGE);

#ifndef DUMP_OBJECT_DATA
#define DUMP_OBJECT_DATA 0
#endif

// TODO FIXME
class SceneThreadData;
#define TraceThreadData SceneThreadData

// These fields are common to all objects. 
class LightSource;
class Ray;
class Intersection;

template<typename T>
class RefPool
{
	public:
		RefPool() { }
		~RefPool() { for(typename vector<T*>::iterator i(pool.begin()); i != pool.end(); i++) delete *i; pool.clear(); }

		T *alloc() { if(pool.empty()) return new T(); T *ptr(pool.back()); pool.pop_back(); return ptr; }
		void release(T *ptr) { pool.push_back(ptr); }
	private:
		vector<T*> pool;

		RefPool(const RefPool&);
		RefPool& operator=(RefPool&);
};

template<typename T>
struct RefClearDefault
{
	void operator()(T&) { }
};

template<typename T>
struct RefClearContainer
{
	void operator()(T& p) { p.clear(); }
};

template<typename T, class C = RefClearDefault<T> >
class Ref
{
	public:
		Ref(RefPool<T>& p) : pool(p), ptr(p.alloc()) { }
		~Ref() { C c; c(*ptr); pool.release(ptr); }

		T& operator*() { return *ptr; }
		const T& operator*() const { return *ptr; }

		T *operator->() { return ptr; }
		const T *operator->() const { return ptr; }
	private:
		RefPool<T>& pool;
		T *ptr;

		Ref();
		Ref(const Ref&);
		Ref& operator=(Ref&);
};

typedef stack<Intersection, vector<Intersection> > IStackData;
typedef RefPool<IStackData> IStackPool;
typedef Ref<IStackData> IStack;

struct WeightedTexture
{
	COLC weight;
	TEXTURE *texture;

	WeightedTexture(COLC w, TEXTURE *t) :
		weight(w), texture(t) { }
};

// TODO: these sizes will need tweaking.
typedef FixedSimpleVector<WeightedTexture, 64> WeightedTextureVector;
typedef FixedSimpleVector<Interior *, 64> RayInteriorVector;

class ObjectDebugHelper
{
	public:
		int Index;
		bool IsCopy;
		std::string Tag;

		ObjectDebugHelper() { Index = ObjectIndex++; IsCopy = false; }
		ObjectDebugHelper& operator=(const ObjectDebugHelper& src) { Index = ObjectIndex++; IsCopy = true; Tag = src.Tag; return *this; }

		std::string& SimpleDesc (std::string& result);
	private:
		static int ObjectIndex;
		ObjectDebugHelper(const ObjectDebugHelper& src) { Index = ObjectIndex++; IsCopy = true; Tag = src.Tag; }
};

class ObjectBase
{
	public:
		int Type;
		TEXTURE *Texture;
		TEXTURE *Interior_Texture;
		Interior *interior;
		vector<ObjectPtr> Bound;
		vector<ObjectPtr> Clip;
		vector<LightSource *> LLights;
		BBOX BBox;
		TRANSFORM *Trans;
		TRANSFORM *UV_Trans;
		SNGL Ph_Density;
		unsigned int Flags;

		// Only objects with this set to true increment trace level:
		bool incrementsTraceLevel;

#ifdef OBJECT_DEBUG_HELPER
		ObjectDebugHelper Debug;
#endif
		enum BBoxDirection
		{
			BBOX_DIR_X0Y0Z0 = 0,
			BBOX_DIR_X0Y0Z1 = 1,
			BBOX_DIR_X0Y1Z0 = 2,
			BBOX_DIR_X0Y1Z1 = 3,
			BBOX_DIR_X1Y0Z0 = 4,
			BBOX_DIR_X1Y0Z1 = 5,
			BBOX_DIR_X1Y1Z0 = 6,
			BBOX_DIR_X1Y1Z1 = 7
		};

		ObjectBase(): incrementsTraceLevel(true) { Flags = 0; Ph_Density = 0; }
		virtual ~ObjectBase() { }

		virtual ObjectPtr Copy() = 0;

		virtual bool All_Intersections(Ray&, IStack&, TraceThreadData *) = 0;
		virtual bool Inside(VECTOR, TraceThreadData *) = 0;
		virtual void Normal(VECTOR, Intersection *, TraceThreadData *) = 0;
		virtual void UVCoord(UV_VECT, Intersection *, TraceThreadData *);
		virtual void Translate(VECTOR, TRANSFORM *) = 0;
		virtual void Rotate(VECTOR, TRANSFORM *) = 0;
		virtual void Scale(VECTOR, TRANSFORM *) = 0;
		virtual void Transform(TRANSFORM *) = 0;
		virtual void Invert() = 0;
		virtual void Compute_BBox() = 0;
		virtual void Determine_Textures(Intersection *, bool, WeightedTextureVector&, TraceThreadData *Thread);
		virtual bool Intersect_BBox(BBoxDirection, const BBOX_VECT&, const BBOX_VECT&, BBOX_VAL = HUGE_VAL);

		// optional post-render message dispatcher; will be called upon completion
		// of rendering a view. this is the appropriate place to send messages that
		// would traditionally have been sent at the end of a render or at object
		// destruction - e.g. IsoSurface max_gradient warnings. (object destruction
		// isn't the place to do that anymore since a scene may persist between views).
		virtual void DispatchShutdownMessages(MessageFactory& messageFactory) {};

	protected:
		explicit ObjectBase(const ObjectBase&) { }
};

/* This is an abstract structure that is never actually used.
   All other objects are descendents of this primitive type */

class CompoundObject : public ObjectBase
{
	public:
		vector<ObjectPtr> children;
};

typedef struct BBox_Tree_Struct BBOX_TREE;

struct BBox_Tree_Struct
{
	short Infinite;   // Flag if node is infinite            
	short Entries;    // Number of sub-nodes in this node    
	BBOX BBox;        // Bounding box of this node           
	BBOX_TREE **Node; // If node: children; if leaf: element 
};

typedef struct Project_Struct PROJECT;
typedef struct Project_Tree_Node_Struct PROJECT_TREE_NODE;

struct Project_Struct
{
	int x1, y1, x2, y2;
};

/*
 * The following structure represent the bounding box hierarchy in 2d space.
 * Because is_leaf, Object and Project are the first elements in both
 * structures they can be accessed without knowing at which structure
 * a pointer is pointing.
 */

struct Project_Tree_Node_Struct
{
	unsigned short is_leaf;
	BBOX_TREE *Node;
	PROJECT Project;
	unsigned short Entries;
	PROJECT_TREE_NODE **Entry;
};

#if 0
#pragma mark * Light Source
#endif

class LightSource : public CompoundObject
{
	public:
		size_t index;
		COLOUR Colour;
		VECTOR Direction, Center, Points_At, Axis1, Axis2;
		DBL Coeff, Radius, Falloff;
		DBL Fade_Distance, Fade_Power, Fade_Cutoff_Distance_Sqr;
		int Area_Size1, Area_Size2;
		int Adaptive_Level;
//		ObjectBase *Shadow_Cached_Object;
		ObjectBase *Projected_Through_Object;
		BLEND_MAP *blend_map;// NK for dispersion 
//		PROJECT_TREE_NODE *Light_Buffer[6]; // Light buffers for the six general directions in space. [DB 9/94] 

		unsigned Light_Type : 8;
		bool Area_Light : 1;
		bool Use_Full_Area_Lighting : 1; // JN2007: Full area lighting
		bool Jitter : 1;
		bool Orient : 1;
		bool Circular : 1;
//		bool Track : 1;
		bool Parallel : 1;
		bool Photon_Area_Light : 1;
		bool Media_Attenuation : 1;
		bool Media_Interaction : 1;
		bool lightGroupLight : 1;

		LightSource();
		virtual ~LightSource();

		virtual ObjectPtr Copy();

		virtual bool All_Intersections(Ray& , IStack& , TraceThreadData *);
		virtual bool Inside(VECTOR, TraceThreadData *);
		virtual void Normal(VECTOR, Intersection *, TraceThreadData *);
		virtual void UVCoord(UV_VECT, Intersection *, TraceThreadData *);
		virtual void Translate(VECTOR, TRANSFORM *);
		virtual void Rotate(VECTOR, TRANSFORM *);
		virtual void Scale(VECTOR, TRANSFORM *);
		virtual void Transform(TRANSFORM *);
		virtual void Invert();
		virtual void Compute_BBox() { }
};

typedef unsigned short HF_VAL;


/*****************************************************************************
 *
 * Intersection stack stuff.
 *
 *****************************************************************************/

#if 0
#pragma mark * Intersection
#endif

void MInvTransPoint(VECTOR result, const VECTOR vector, const TRANSFORM *transform); // TODO FIXME - needed below, remove code that needs it from frame.h! [trf]

/**
 *  Ray-object intersection data.
 *  This class holds various information on a ray-object intersection.
 */
class Intersection
{
	public:
		/// Distance from the intersecting ray's origin.
		DBL Depth;
		/// Point of the intersection in global coordinate space.
		VECTOR IPoint;
		/// Unpertubed surface normal at the intersection point.
		/// @note This is not necessarily the true geometric surface normal, as it may include fake smoothing.
		/// @note This value is invalid if haveNormal is false.
		/// @todo We should have two distinct vectors: A true geometric one, and a separate one for faked smoothing.
		VECTOR INormal;
		/// Perturbed normal vector (set during texture evaluation).
		VECTOR PNormal;
		/// UV texture coordinate.
		UV_VECT Iuv;
		/// Intersected object.
		ObjectBase *Object;

		/// @name Object-Specific Auxiliary Data
		/// These members hold information specific to particular object types, typically generated during
		/// intersection testing, to be re-used later for normal and/or UV coordinate computation.
		/// @note These values may be invalid or meaningless depending on the type of object intersected.
		//@{
		/// Point of the intersection in local coordinate space (used by Blob)
		/// @note This value is invalid if haveLocalIPoint is false.
		VECTOR LocalIPoint;
		/// Flag to indicate whether INormal was computed during intersection testing (used by HField)
		/// @note Objects either always or never computing INormal during intersection testing don't use this flag.
		bool haveNormal;
		/// Flag to indicate whether LocalIPoint has already been computed.
		bool haveLocalIPoint;
		/// Generic auxiliary integer data #1 (used by Sor, Prism, Lathe, Cones, Boxes)
		int i1;
		/// Generic auxiliary integer data #2 (used by Sor, Prism)
		int i2;
		/// Generic auxiliary float data #1 (used by Prism, Lathe)
		DBL d1;
		/// Generic auxiliary pointer data (used by Mesh)
		void *Pointer;
		//@}

		/// Root-level parent CSG object for cutaway textures.
		ObjectBase *Csg;
		// View ray direction for angle of incidence pattern
		VECTOR RDir; 

		Intersection() { Depth = BOUND_HUGE; Object = NULL; Csg = NULL; }

		Intersection(DBL d, VECTOR& v, ObjectBase *o)
		{
			Depth  = d;
			Object = o;
			Assign_Vector(IPoint, v);
			Assign_UV_Vect(Iuv, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, VECTOR& n, ObjectBase *o)
		{
			Depth  = d;
			Object = o;
			Assign_Vector(IPoint, v);
			Assign_UV_Vect(Iuv, v);
			Assign_Vector(INormal, n);
			haveNormal = true;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, UV_VECT& uv, ObjectBase *o)
		{
			Depth  = d;
			Object = o;
			Assign_Vector(IPoint, v);
			Assign_UV_Vect(Iuv, uv);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, VECTOR& n, UV_VECT& uv, ObjectBase *o)
		{
			Depth  = d;
			Object = o;
			Assign_Vector(IPoint, v);
			Assign_Vector(INormal, n);
			Assign_UV_Vect(Iuv, uv);
			haveNormal = true;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, void *a)
		{
			Depth  = d;
			Object = o;
			Pointer = (void *)(a);
			Assign_Vector(IPoint, v);
			Assign_UV_Vect(Iuv, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, UV_VECT& uv, ObjectBase *o, void *a)
		{
			Depth  = d;
			Object = o;
			Pointer = (void *)(a);
			Assign_Vector(IPoint, v);
			Assign_UV_Vect(Iuv, uv);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, int a)
		{
			Depth  = d;
			Object = o;
			i1 = a;
			Assign_Vector(IPoint, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, DBL a)
		{
			Depth  = d;
			Object = o;
			d1 = a;
			Assign_Vector(IPoint, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, int a, int b)
		{
			Depth  = d;
			Object = o;
			i1 = a;
			i2 = b;
			Assign_Vector(IPoint, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, int a, DBL b)
		{
			Depth  = d;
			Object = o;
			i1 = a;
			d1 = b;
			Assign_Vector(IPoint, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		Intersection(DBL d, VECTOR& v, ObjectBase *o, int a, int b, DBL c)
		{
			Depth  = d;
			Object = o;
			i1 = a;
			i2 = b;
			d1 = c;
			Assign_Vector(IPoint, v);
			haveNormal = false;
			haveLocalIPoint = false;
			Csg = NULL;
		}

		~Intersection() { }
};

/*****************************************************************************
 *
 * Ray stuff (see also RAY.H).
 *
 *****************************************************************************/

#if 0
#pragma mark * Ray
#endif

class Ray
{
	public:
		enum RayType
		{
			OtherRay = 0,
			PrimaryRay = 1,
			ReflectionRay = 2,
			RefractionRay = 3,
		};

		VECTOR Origin;
		VECTOR Direction;

		Ray(RayType rt = PrimaryRay, bool shadowTest = false, bool photon = false, bool radiosity = false, bool monochromatic = false, bool pretrace = false);
		Ray(const VECTOR ov, const VECTOR dv, RayType rt = PrimaryRay, bool shadowTest = false, bool photon = false, bool radiosity = false, bool monochromatic = false, bool pretrace = false);
		~Ray();

		const VECTOR& GetOrigin() { return Origin; }
		const VECTOR& GetDirection() { return Direction; }

		void AppendInterior(Interior *i);
		void AppendInteriors(RayInteriorVector&);
		bool RemoveInterior(Interior *i);
		void ClearInteriors() { interiors.clear(); }

		bool IsInterior(Interior *i);
		const RayInteriorVector& GetInteriors() const { return interiors; }

		void SetDispersionFactor(DBL df);
		DBL GetDispersionFactor();

		void SetFlags(RayType rt, bool shadowTest = false, bool photon = false, bool radiosity = false, bool monochromatic = false, bool pretrace = false);
		void SetFlags(RayType rt, const Ray& other);

		bool IsPrimaryRay() const { return primaryRay; }
		bool IsReflectionRay() const { return reflectionRay; }
		bool IsRefractionRay() const { return refractionRay; }
		bool IsShadowTestRay() const { return shadowTestRay; }
		bool IsPhotonRay() const { return photonRay; }
		bool IsRadiosityRay() const { return radiosityRay; }
		bool IsMonochromaticRay() const { return monochromaticRay; }
		bool IsHollowRay() const { return hollowRay; }
		bool IsPretraceRay() const { return pretraceRay; }

		bool Inside(const BBOX& bbox) const { return Inside_BBox(Origin, bbox); }
	private:
		RayInteriorVector interiors;
		DBL dispersionFactor;

		bool primaryRay : 1;
		bool reflectionRay : 1;
		bool refractionRay : 1;
		bool shadowTestRay : 1;
		bool photonRay : 1;
		bool radiosityRay : 1;
		bool monochromaticRay : 1;
		bool hollowRay : 1;
		bool pretraceRay : 1;
};

struct RayObjectCondition
{
	virtual bool operator()(Ray& ray, ObjectPtr object, DBL data) const = 0;
};

struct TrueRayObjectCondition : public RayObjectCondition
{
	virtual bool operator()(Ray&, ObjectPtr, DBL) const { return true; }
};

struct PointObjectCondition
{
	virtual bool operator()(Vector3d& point, ObjectPtr object) const = 0;
};

struct TruePointObjectCondition : public PointObjectCondition
{
	virtual bool operator()(Vector3d&, ObjectPtr) const { return true; }
};


/*****************************************************************************
 *
 * Frame tracking information
 *
 *****************************************************************************/

enum FRAMETYPE
{
	FT_SINGLE_FRAME,
	FT_MULTIPLE_FRAME
};

#define INT_VALUE_UNSET (-1)
#define DBL_VALUE_UNSET (-1.0)

struct FrameSettings
{
	FRAMETYPE FrameType;
	DBL Clock_Value;      // May change between frames of an animation 
	int FrameNumber;      // May change between frames of an animation 

	int InitialFrame;
	DBL InitialClock;

	int FinalFrame;
	int FrameNumWidth;
	DBL FinalClock;

	int SubsetStartFrame;
	DBL SubsetStartPercent;
	int SubsetEndFrame;
	DBL SubsetEndPercent;

	bool Field_Render_Flag;
	bool Odd_Field_Flag;
};


/*****************************************************************************
 *
 * Miscellaneous stuff.
 *
 *****************************************************************************/

typedef struct complex_block complex;

struct Chunk_Header_Struct
{
	int name;
	int size;
};

struct complex_block
{
	DBL r, c;
};

struct BYTE_XYZ
{
	unsigned char x, y, z;
};

// platform-specific headers should have provided DECLARE_THREAD_LOCAL_VARIABLE.
// if not, rather than generate an error, we will default to useing the common
// __thread-style declaration. if your compiler throws an error on this declaration,
// you'll need to fix your platform-specific config to provide a means to declare
// thread-local variables. support for TLS is required for efficient access by the
// render threads to their stats and cache data.
#ifndef DECLARE_THREAD_LOCAL_PTR
#define DECLARE_THREAD_LOCAL_PTR(ptrType, ptrName)       __thread ptrType *ptrName;
#endif

}

#endif
