/*******************************************************************************
 * ppm.cpp
 *
 * This module contains the code to read and write the PPM file format.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/base/image/ppm.cpp $
 * $Revision: #14 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

/****************************************************************************
*
*  PPM format according to NetPBM specs (http://netpbm.sourceforge.net/doc/):
*
*  This module implements read support for PPM image maps and
*  write support for PPM output.
*
*  For reading both ASCII and binary files are supported ('P3' and 'P6').
*
*  For writing we use binary files. OutputQuality > 8 leads to 16 bit files.
*  Special handling of HF_GRAY_16 -> 16 bit PGM files ('P5')
*  All formats supported for writing can now also be used in continued trace.
*
*****************************************************************************/

#include <vector>
#include <ctype.h>

#include "base/configbase.h"
#include "base/image/image.h"
#include "base/image/pgm.h"
#include "base/image/ppm.h"
#include "base/types.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov_base
{

using namespace std;

namespace Ppm
{

void Write (OStream *file, const Image *image, const Image::WriteOptions& options)
{
  int                   file_type = POV_File_Image_PPM;
  int                   width = image->GetWidth() ;
  int                   height = image->GetHeight() ;
  int                   bpcc = options.bpcc;
  bool                  gray16 = false;
  float                 r;
  float                 g;
  float                 b;
  unsigned int          rval;
  unsigned int          gval;
  unsigned int          bval;
  unsigned int          gray;
  unsigned int          mask;

  // do we want 16 bit grayscale (PGM) output ?
  if (image->GetImageDataType () == Image::Gray_Int16 || image->GetImageDataType () == Image::GrayA_Int16)
  {
    gray16 = true ;
    file->printf("P5\n%d %d\n65535\n", width, height);
  }
  else
    file->printf("P6\n%d %d\n%d\n", width, height, (1 << bpcc) - 1);

  if (bpcc == 0)
    bpcc = image->GetMaxIntValue() == 65535 ? 16 : 8 ;
  else if (bpcc < 5)
    bpcc = 5 ;
  else if (bpcc > 16)
    bpcc = 16 ;

  for (int y = 0 ; y < height ; y++)
  {
    for (int x = 0; x < width; x++)
    {
      if (gray16)
      {
        gray = (int) floor (clip(image->GetGrayValue(x, y), 0.0f, 1.0f) * 65535.0f) ;
        if (!file->Write_Byte((gray >> 8) & 0xFF))
          throw POV_EXCEPTION(kFileDataErr, "Cannot write PPM output data");
        if (!file->Write_Byte(gray & 0xFF))
          throw POV_EXCEPTION(kFileDataErr, "Cannot write PPM output data");
      }
      else
      {
        // otherwise 3*OutputQuality bit pixel coloring written to 8 or 16 bit file
        mask = (1 << bpcc) - 1 ;

        image->GetRGBValue (x, y, r, g, b) ;

        if (bpcc > 8)  // 16 bit per value 
        {
          rval = (unsigned int)floor(clip(powf(r, options.gamma), 0.0f, 1.0f) * 65535.0f * mask) & mask;
          gval = (unsigned int)floor(clip(powf(g, options.gamma), 0.0f, 1.0f) * 65535.0f * mask) & mask;
          bval = (unsigned int)floor(clip(powf(b, options.gamma), 0.0f, 1.0f) * 65535.0f * mask) & mask;
          file->Write_Byte(rval >> 8) ;
          file->Write_Byte(rval & 0xFF) ;
          file->Write_Byte(gval >> 8) ;
          file->Write_Byte(gval & 0xFF) ;
          file->Write_Byte(bval >> 8) ;
          if (!file->Write_Byte(bval & 0xFF))
            throw POV_EXCEPTION(kFileDataErr, "Cannot write PPM output data");
        }
        else
        {
          // 8 bit per value
          rval = (unsigned int)floor(clip(powf(r, options.gamma), 0.0f, 1.0f) * 255.0f);
          gval = (unsigned int)floor(clip(powf(g, options.gamma), 0.0f, 1.0f) * 255.0f);
          bval = (unsigned int)floor(clip(powf(b, options.gamma), 0.0f, 1.0f) * 255.0f);
          file->Write_Byte(rval & 0xFF) ;
          file->Write_Byte(gval & 0xFF) ;
          if (!file->Write_Byte(bval & 0xFF))
            throw POV_EXCEPTION(kFileDataErr, "Cannot write PPM output data");
        }
      }
    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*  Read_PPM_Image
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*    Christoph Hormann
*
* DESCRIPTION
*
*    Reads an PPM image file
*
* CHANGES
*
*    August 2003 - New implementation based on targa/png reading code
*
******************************************************************************/

// TODO: make sure we destroy the image if we throw an exception
Image *Read (IStream *file, const Image::ReadOptions& options)
{
  int                   nbr;
  int                   width;
  int                   height;
  int                   data_hi;
  int                   data_lo;
  char                  line[1024];
  char                  *ptr;
  Image                 *image = NULL;
  unsigned int          depth;
  unsigned int          r ;
  unsigned int          g ;
  unsigned int          b;
  unsigned char         header[2];

  // --- Read Header --- 
  if (!file->read((char *)header, 2))
    throw POV_EXCEPTION(kFileDataErr, "Cannot read header of PPM image");

  if(header[0] != 'P')
    throw POV_EXCEPTION(kFileDataErr, "File is not in PGM format");

  if ((header[1] != '3') && (header[1] != '6'))
    throw POV_EXCEPTION(kFileDataErr, "File is not in PPM format");

  do
  {
    file->getline (line, 1024);
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL)
      *ptr = '\0';  // remove comment 
  } while (line[0]=='\0');  // read until line without comment from beginning 

  // --- First: two numbers: width and height --- 
  if (sscanf(line,"%d %d",&width, &height) != 2)
    throw POV_EXCEPTION(kFileDataErr, "Cannot read width and height from PGM image");

  if (width <= 0 || height <= 0)
    throw POV_EXCEPTION(kFileDataErr, "Invalid width or height read from PGM image");

  do
  {
    file->getline (line, 1024) ;
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL)
      *ptr = '\0';  // remove comment 
  } while (line[0]=='\0');  // read until line without comment from beginning 

  // --- Second: one number: color depth --- 
  if (sscanf(line,"%d",&depth) != 1)
    throw POV_EXCEPTION(kFileDataErr, "Cannot read color depth from PGM image");

  if ((depth > 65535) || (depth < 1))
    throw POV_EXCEPTION(kFileDataErr, "Unsupported number of colors in PGM image");

  if (depth < 256)
  {
    Image::ImageDataType imagetype = options.itype;
    if (imagetype == Image::Undefined)
      imagetype = Image::RGB_Int8;
    image = Image::Create (width, height, imagetype) ;
    bool native = imagetype == Image::RGBA_Int8 || imagetype == Image::RGB_Int8;
    for (int i = 0; i < height; i++)
    {
      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (int x = 0; x < width; x++)
        {
          r = Pgm::Read_ASCII_File_Number(file) * 255 / depth;
          g = Pgm::Read_ASCII_File_Number(file) * 255 / depth;;
          b = Pgm::Read_ASCII_File_Number(file) * 255 / depth;;
          if (native)
            image->SetRGBValue (x, i, r, g, b) ;
          else
            image->SetRGBValue (x, i, r/255.0f, g/255.0f, b/255.0f) ;
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (int x = 0; x < width; x++)
        {
          if ((nbr = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          r = (nbr*255)/depth;

          if ((nbr = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          g = (nbr*255)/depth;

          if ((nbr = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          b = (nbr*255)/depth;

          if (native)
            image->SetRGBValue (x, i, r, g, b) ;
          else
            image->SetRGBValue (x, i, r/255.0f, g/255.0f, b/255.0f) ;
        }
      }
    }
  }
  else // --- 16 bit PPM (binary or ASCII) --- 
  {
    Image::ImageDataType imagetype = options.itype;
    if (imagetype == Image::Undefined)
      imagetype = Image::RGB_Int16;
    image = Image::Create (width, height, imagetype) ;
    bool native = imagetype == Image::RGBA_Int16 || imagetype == Image::RGB_Int16;
    for (int i = 0; i < height; i++)
    {
      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (int x = 0; x < width; x++)
        {
          r = Pgm::Read_ASCII_File_Number(file) * 65535 / depth;
          g = Pgm::Read_ASCII_File_Number(file) * 65535 / depth;;
          b = Pgm::Read_ASCII_File_Number(file) * 65535 / depth;;

          if (native)
            image->SetRGBValue (x, i, r, g, b) ;
          else
            image->SetRGBValue (x, i, r/65535.0f, g/65535.0f, b/65535.0f) ;
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (int x = 0; x < width; x++)
        {
          if ((data_hi = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          if ((data_lo = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          r = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          if ((data_lo = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          g = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          if ((data_lo = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Unexpected EOF in PGM file");
          b = (256*data_hi + data_lo)*65535/depth;

          if (native)
            image->SetRGBValue (x, i, r, g, b) ;
          else
            image->SetRGBValue (x, i, r/65535.0f, g/65535.0f, b/65535.0f) ;
        }
      }
    }
  }

  return (image) ;
}

} // end of namespace Ppm

}

