/*******************************************************************************
 * pvmem.h
 *
 * Definitions for windows-specific memory handling routines.
 *
 * Author: Christopher J. Cason.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/windows/pvmem.h $
 * $Revision: #5 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#ifndef __PVMEM_H__
#define __PVMEM_H__

#undef new
#undef delete

namespace povwin
{

#ifdef _WIN64
  #define InterlockedExchangeAdd InterlockedExchangeAdd64
  #define InterlockedIncrement InterlockedIncrement64
#endif

class WinMemStats
{
  public:
    WinMemStats() { Clear(); }
    ~WinMemStats() {}

    void Clear() { currentAllocated = peakAllocated = callsToAlloc = callsToFree = smallestAlloc = largestAlloc = 0; }
    void * operator new(size_t count) { return (WinMemStats *) HeapAlloc(GetProcessHeap(), 0, count); }
    void operator delete (void *p) { HeapFree(GetProcessHeap(), 0, p); }

    void RecordAlloc(size_t count)
    {
      callsToAlloc++;
      currentAllocated += count;
      if (currentAllocated > 0)
        if (peakAllocated < currentAllocated)
          peakAllocated = currentAllocated;
      if (count > 1)
        if ((smallestAlloc == 0) || (count < smallestAlloc))
          smallestAlloc = count;
      if (count > largestAlloc)
        largestAlloc = count;
    }

    void RecordFree(size_t count)
    {
      callsToFree++;
      currentAllocated -= count;
    }

    void InterlockedRecordAlloc(size_t count)
    {
    }

    void InterlockedRecordFree(size_t count)
    {
    }

    void Report(uint64& allocs, uint64& frees, int64& current, uint64& peak, uint64 &smallest, uint64 &largest)
    {
      allocs = callsToAlloc;
      frees = callsToFree;
      current = currentAllocated;
      peak = peakAllocated;
      smallest = smallestAlloc;
      largest = largestAlloc;
    }

    WinMemStats& operator=(const WinMemStats& rhs)
    {
      // not interlocked, should only be used when thread is not performing memory allocations
      currentAllocated = rhs.currentAllocated;
      peakAllocated = rhs.peakAllocated;
      callsToAlloc = rhs.callsToAlloc;
      callsToFree = rhs.callsToFree;
      smallestAlloc = rhs.smallestAlloc;
      largestAlloc = rhs.largestAlloc;
      return *this;
    }

    WinMemStats operator+(const WinMemStats& rhs) const
    {
      // not interlocked, should only be used when thread is not performing memory allocations
      WinMemStats result = *this;
      result.currentAllocated += rhs.currentAllocated;
      result.callsToAlloc += rhs.callsToAlloc;
      result.callsToFree += rhs.callsToFree;
      result.peakAllocated = 0;
      result.smallestAlloc = 0;
      result.largestAlloc = 0;
      return result;
    }

    WinMemStats& operator+=(const WinMemStats& rhs)
    {
      // not interlocked, should only be used when threads are not performing memory allocations
      currentAllocated += rhs.currentAllocated;
      callsToAlloc += rhs.callsToAlloc;
      callsToFree += rhs.callsToFree;
      peakAllocated = 0;
      smallestAlloc = 0;
      largestAlloc = 0;
      return *this;
    }

  private:
    volatile int64 currentAllocated;
    volatile uint64 peakAllocated;
    volatile uint64 callsToAlloc;
    volatile uint64 callsToFree;
    volatile uint64 smallestAlloc;
    volatile uint64 largestAlloc;
};

typedef WinMemStats PovMemStats;

class WinHeap
{
  public:
    WinHeap()
    {
      InitializeCriticalSectionAndSpinCount(&m_CritSec, 4000);
      m_LowFragHeap = false;

      SYSTEM_INFO si;
      GetSystemInfo(&si);

      // allocate heap
      m_Heap = HeapCreate (HEAP_NO_SERIALIZE, si.dwPageSize, 0);
      if (m_Heap == NULL)
        return;

#if 0
	  // enable low-fragmentation heap. NB this only works on XP or later.
      // TODO: make sure this isn't an unresolved symbol in W2K or earlier.
      ULONG val = 2;
      HeapSetInformation(m_Heap, HeapCompatibilityInformation, &val, sizeof(val));
      if (HeapQueryInformation(m_Heap, HeapCompatibilityInformation, &val, sizeof(val), NULL))
        if (val == 2)
          m_LowFragHeap = true;
#endif
    }
    ~WinHeap()
    {
      if (m_Heap != NULL)
      {
        Lock();
        HeapDestroy(m_Heap);
        Unlock();
      }
      DeleteCriticalSection(&m_CritSec);
    }

    void * operator new(size_t count) { return (WinHeap *) HeapAlloc(GetProcessHeap(), 0, count); }
    bool Validate(const void *p = NULL) const { return m_Heap != NULL && HeapValidate(m_Heap, 0, p); }
    void *Alloc(size_t count) { return HeapAlloc(m_Heap, 0, count ? count : 1); }
    bool Free(void *p) { return HeapFree(m_Heap, 0, p); }
    void *ReAlloc(void *p, size_t count) { return HeapReAlloc(m_Heap, 0, p, count); }
    void Compact() { HeapCompact(m_Heap, 0); }
    void Lock() { EnterCriticalSection(&m_CritSec); }
    void Unlock() { LeaveCriticalSection(&m_CritSec); }
    size_t BlockSize(const void *p) const { return HeapSize(m_Heap, 0, p); }

    void *LockedAlloc(size_t count)
    {
      Lock();
      void *p = HeapAlloc(m_Heap, 0, count ? count : 1);
      Unlock();
      return p;
    }

    void *LockedReAlloc(void *p, size_t count)
    {
      Lock();
      p = HeapReAlloc(m_Heap, 0, p, count);
      Unlock();
      return p;
    }

    bool LockedFree(void *p)
    {
      Lock();
      bool result = HeapFree(m_Heap, 0, p);
      Unlock();
      return result;
    }

    bool LockedValidate(const void *p = NULL)
    {
      if (m_Heap == NULL)
        return false;
      Lock();
      bool result = HeapValidate(m_Heap, 0, p);
      Unlock();
      return result;
    }

  private:
    bool m_LowFragHeap;
    HANDLE m_Heap;
    CRITICAL_SECTION m_CritSec;
};

class HeapLock
{
  public:
    HeapLock(WinHeap *wh) : m_WinHeap(wh) { m_WinHeap->Lock(); }
    ~HeapLock() { m_WinHeap->Unlock(); }
  private:
    WinHeap *m_WinHeap;
};

class WinMemBlock
{
public:
  __declspec(nothrow) WinMemBlock(const void *data, int line)
  {
    m_Sanity = 'PM';
    m_Line = line;
    m_Data = data;
#   if EXTRA_VALIDATION
      m_Hash = ptrdiff_t(m_Sanity) ^ ptrdiff_t(m_Line) ^ ptrdiff_t(m_Data) ^ ptrdiff_t(m_Length) ^ ptrdiff_t(this);
      memcpy((char *) this + m_Length - sizeof(m_Hash), &m_Hash, sizeof(m_Hash));
#   endif
  }

  // must be called with the lock held
  void __declspec(nothrow) * operator new (size_t size, size_t len)
  {
#   if EXTRA_VALIDATION
      WinMemBlock *block = (WinMemBlock *) m_Heap->Alloc(len + size + BlockPadding);
      if (block == NULL)
        return NULL;
      block->m_Length = len + size + BlockPadding;
      return block;
#   else
      return m_Heap->Alloc(len + size);
#   endif
  }

  // must be called with the lock held
  void __declspec(nothrow) operator delete (void *p)
  {
    m_Heap->Free(p);
  }

  bool Validate(bool full = EXTRA_VALIDATION) const
  {
    if (m_Sanity != 'PM')
      return false;
#   if EXTRA_VALIDATION
      ptrdiff_t hash = ptrdiff_t(m_Sanity) ^ ptrdiff_t(m_Line) ^ ptrdiff_t(m_Data) ^ ptrdiff_t(m_Length) ^ ptrdiff_t(this);
      if (m_Hash != hash || memcmp(&m_Hash, (char *) this + m_Length - sizeof(m_Hash), sizeof(m_Hash)) != 0)
        return false;
#   endif
    if (!full)
      return true;
    if (m_Heap->Validate(this) == false)
      return false;
#   if EXTRA_VALIDATION
      if (m_Heap->BlockSize(this) < m_Length)
        return false;
#   endif
    return true;
  }

  size_t Size() const
  {
    size_t len = m_Heap->BlockSize(this);
    if (len == (size_t) -1 || len < sizeof(WinMemBlock) + BlockPadding)
      return (size_t) -1;
#   if EXTRA_VALIDATION
      if (len < m_Length)
        return (size_t) -1;
#   endif
    return len;
  }

  size_t Size(size_t size) const
  {
    return size + sizeof(WinMemBlock) + BlockPadding;
  }

  void *GetMem()
  {
    return this + 1;
  }

  WinMemBlock *Realloc(size_t size, const void *data, int line)
  {
    WinMemBlock *newBlock = (WinMemBlock *) m_Heap->ReAlloc(this, size + sizeof(WinMemBlock) + BlockPadding);
    if (newBlock == NULL)
      return NULL;
    // note that *this is no longer valid here!
    newBlock->m_Line = line;
    newBlock->m_Data = data;
#   if EXTRA_VALIDATION
      newBlock->m_Length = size + sizeof(WinMemBlock) + BlockPadding;
      newBlock->m_Hash = ptrdiff_t(newBlock->m_Sanity) ^ ptrdiff_t(newBlock->m_Line) ^ ptrdiff_t(newBlock->m_Data) ^
                         ptrdiff_t(newBlock->m_Length) ^ ptrdiff_t(newBlock);
      memcpy((char *) newBlock + newBlock->m_Length - sizeof(m_Hash), &newBlock->m_Hash, sizeof(m_Hash));
#   endif
    return newBlock;
  }

  static WinMemBlock *GetBlock(void *p, bool full = EXTRA_VALIDATION)
  {
    WinMemBlock *block = (WinMemBlock *) p - 1;
    if (!block->Validate(full))
      return NULL;
    return block;
  }

  static void SetAllocator(WinHeap *alloc) { m_Heap = alloc; }

private:
  unsigned short m_Sanity;
  short m_Line;
  const void *m_Data;
  static WinHeap *m_Heap;

# if EXTRA_VALIDATION
    size_t m_Length;
    ptrdiff_t m_Hash;
    static const size_t BlockPadding = sizeof(ptrdiff_t);
# else
    static const size_t BlockPadding = 0;
# endif

  WinMemBlock() {} // not available
  void *operator new (size_t len) {} // not available
};

}

#endif
