/*******************************************************************************
 * trace.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/render/trace.cpp $
 * $Revision: #136 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#include <boost/thread.hpp>
#include <boost/bind.hpp>

#include "backend/frame.h"
#include "backend/colour/colour.h"
#include "backend/math/vector.h"
#include "backend/math/matrices.h"
#include "backend/scene/objects.h"
#include "backend/render/ray.h"
#include "backend/pattern/pattern.h"
#include "backend/pattern/warps.h"
#include "backend/support/imageutil.h"
#include "backend/texture/normal.h"
#include "backend/texture/pigment.h"
#include "backend/texture/texture.h"
#include "backend/render/trace.h"
#include "backend/render/tracetask.h"
#include "backend/scene/scene.h"
#include "backend/scene/view.h"
#include "backend/lighting/point.h"
#include "backend/lighting/radiosity.h"
#include "backend/shape/csg.h"
#include "backend/shape/boxes.h"
#include "backend/support/bsptree.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

#define SHADOW_TOLERANCE 1.0e-3

using namespace std;
using namespace boost;

Trace::Trace(shared_ptr<SceneData> sd, TraceThreadData *td, unsigned int mtl, DBL adcb, unsigned int qf,
		     CooperateFunctor& cf, MediaFunctor& mf, RadiosityFunctor& rf) :
	threadData(td),
	sceneData(sd),
	traceLevel(0),
	maxAllowedTraceLevel(mtl),
	maxFoundTraceLevel(0),
	adcBailout(adcb),
	qualityFlags(qf),
	mailbox(0),
	randomNumbers(0.0, 1.0, 32768),
	randomNumberGenerator(&randomNumbers),
	cooperate(cf),
	media(mf),
	radiosity(rf)
{
	lightSourceLevel1ShadowCache.resize(threadData->lightSources.size());
	for(vector<ObjectPtr>::iterator i(lightSourceLevel1ShadowCache.begin()); i != lightSourceLevel1ShadowCache.end(); i++)
		*i = NULL;

	lightSourceOtherShadowCache.resize(threadData->lightSources.size());
	for(vector<ObjectPtr>::iterator i(lightSourceOtherShadowCache.begin()); i != lightSourceOtherShadowCache.end(); i++)
		*i = NULL;

	// TODO: this could be initialised somewhere further up the call tree.
	threadData->qualityFlags = qualityFlags;

	if(sceneData->boundingMethod == 2)
		mailbox = BSPTree::Mailbox(sceneData->numberOfFiniteObjects);
}

Trace::~Trace()
{
}

DBL Trace::TraceRay(Ray& ray, Colour& colour, COLC weight)
{
	Intersection bestisect;
	bool found = false;
	NoSomethingFlagRayObjectCondition precond;
	TrueRayObjectCondition postcond;

	POV_ULONG nrays = threadData->Stats[Number_Of_Rays]++;
	if(((unsigned char) nrays & 0x0f) == 0x00)
		cooperate();

	// Check for max. trace level or ADC bailout.
	if((traceLevel >= maxAllowedTraceLevel) || (weight < adcBailout))
	{
		if(weight < adcBailout)
			threadData->Stats[ADC_Saves]++;

		colour.clear();
		return HUGE_VAL;
	}

	found = FindIntersection(bestisect, ray, precond, postcond);

        const bool traceLevelIncremented =
            !found ||
            bestisect.Object->incrementsTraceLevel ||
            ray.IsRadiosityRay();

        if(traceLevelIncremented)
        {
            // Set highest level traced.
            traceLevel++;
            maxFoundTraceLevel = max(maxFoundTraceLevel, traceLevel);
        }

	if((qualityFlags & Q_VOLUME) && (ray.IsPhotonRay() == true) && (ray.IsHollowRay() == true))
	{
		// Note: this version of ComputeMedia does not deposit photons. This is
		// intentional.  Even though we're processing a photon ray, we don't want
		// to deposit photons in the infinite atmosphere, only in contained
		// media, which is processed later (in ComputeLightedTexture).  [nk]
		media.ComputeMedia(sceneData->atmosphere, ray, bestisect, colour);

		if(sceneData->fog != NULL)
			ComputeFog(ray, bestisect, colour);
	}

	if(found)
		ComputeTextureColour(bestisect, colour, ray, weight, false);
	else
		ComputeSky(ray, colour);

	if((qualityFlags & Q_VOLUME) && (ray.IsPhotonRay() == false) && (ray.IsHollowRay() == true))
	{
		if((sceneData->rainbow != NULL) && (ray.IsShadowTestRay() == false))
			ComputeRainbow(ray, bestisect, colour);

		media.ComputeMedia(sceneData->atmosphere, ray, bestisect, colour);

		if(sceneData->fog != NULL)
			ComputeFog(ray, bestisect, colour);
	}

        if(traceLevelIncremented)
            traceLevel--;

	if(found == false)
		return HUGE_VAL;
	else
		return bestisect.Depth;
}

bool Trace::FindIntersection(Intersection& bestisect, Ray& ray)
{
	switch(sceneData->boundingMethod)
	{
		case 2:
		{
			BSPIntersectFunctor ifn(bestisect, ray, sceneData->objects, threadData);
			bool found = false;

			mailbox.clear();

			found = (*(sceneData->tree))(ray, ifn, mailbox, bestisect.Depth);

			// test infinite objects
			for(vector<ObjectPtr>::iterator it = sceneData->objects.begin() + sceneData->numberOfFiniteObjects; it != sceneData->objects.end(); it++)
			{
				Intersection isect;

				if(FindIntersection(*it, isect, ray) && (isect.Depth < bestisect.Depth))
				{
					bestisect = isect;
					found = true;
				}
			}

			return found;
		}
		case 1:
		{
			if(sceneData->boundingSlabs != NULL)
				return (Intersect_BBox_Tree(priorityQueue, sceneData->boundingSlabs, ray, &bestisect, threadData));
			// drop through
		}
		case 0:
		{
			bool found = false;

			for(vector<ObjectPtr>::iterator it = sceneData->objects.begin(); it != sceneData->objects.end(); it++)
			{
				Intersection isect;

				if(FindIntersection(*it, isect, ray) && (isect.Depth < bestisect.Depth))
				{
					bestisect = isect;
					found = true;
				}
			}

			return found;
		}
	}

	return false;
}

bool Trace::FindIntersection(Intersection& bestisect, Ray& ray, const RayObjectCondition& precondition, const RayObjectCondition& postcondition)
{
	switch(sceneData->boundingMethod)
	{
		case 2:
		{
			BSPIntersectCondFunctor ifn(bestisect, ray, sceneData->objects, threadData, precondition, postcondition);
			bool found = false;

			mailbox.clear();

			found = (*(sceneData->tree))(ray, ifn, mailbox, bestisect.Depth);

			// test infinite objects
			for(vector<ObjectPtr>::iterator it = sceneData->objects.begin() + sceneData->numberOfFiniteObjects; it != sceneData->objects.end(); it++)
			{
				if(precondition(ray, *it) == true)
				{
					Intersection isect;

					if(FindIntersection(*it, isect, ray) && (isect.Depth < bestisect.Depth))
					{
						if(postcondition(ray, *it) == true)
						{
							bestisect = isect;
							found = true;
						}
					}
				}
			}

			return found;
		}
		case 1:
		{
			if(sceneData->boundingSlabs != NULL)
				return (Intersect_BBox_Tree(priorityQueue, sceneData->boundingSlabs, ray, &bestisect, precondition, postcondition, threadData));
			// drop through
		}
		case 0:
		{
			bool found = false;

			for(vector<ObjectPtr>::iterator it = sceneData->objects.begin(); it != sceneData->objects.end(); it++)
			{
				if(precondition(ray, *it) == true)
				{
					Intersection isect;

					if(FindIntersection(*it, isect, ray) && (isect.Depth < bestisect.Depth))
					{
						if(postcondition(ray, *it) == true)
						{
							bestisect = isect;
							found = true;
						}
					}
				}
			}

			return found;
		}
	}

	return false;
}

bool Trace::FindIntersection(ObjectPtr object, Intersection& isect, Ray& ray, DBL closest)
{
	if(object != NULL)
	{
		BBOX_VECT origin;
		BBOX_VECT invdir;
		ObjectBase::BBoxDirection variant;

		Vector3d tmp(1.0 / ray.GetDirection()[X], 1.0 / ray.GetDirection()[Y], 1.0 /ray.GetDirection()[Z]);
		Assign_Vector(origin, ray.Origin);
		Assign_Vector(invdir, *tmp);
		variant = (ObjectBase::BBoxDirection)((int(invdir[X] < 0.0) << 2) | (int(invdir[Y] < 0.0) << 1) | int(invdir[Z] < 0.0));

		if(object->Intersect_BBox(variant, origin, invdir, closest) == false)
			return false;

		if(object->Bound.empty() == false)
		{
			if(Ray_In_Bound(ray, object->Bound, threadData) == false)
				return false;
		}

		IStack depthstack(stackPool);

		if(object->All_Intersections(ray, depthstack, threadData))
		{
			bool found = false;

			while(depthstack->size() > 0)
			{
				if(depthstack->top().Depth < closest)
				{
					isect = depthstack->top();
					closest = depthstack->top().Depth;
					found = true;
				}

				depthstack->pop();
			}

			// TODO FIXME - This was SMALL_TOLERANCE, but that's too rough for some scenes [cjc] need to check what it was in the old code [trf]
			return (found == true) && (closest >= MIN_ISECT_DEPTH);
		}
	}

	return false;
}

unsigned int Trace::GetCurrentTraceLevel()
{
	return traceLevel;
}

unsigned int Trace::GetHighestTraceLevel()
{
	return maxFoundTraceLevel;
}

void Trace::ComputeTextureColour(Intersection& isect, Colour& colour, Ray& ray, COLC weight, bool photonPass)
{
	// NOTE: when called during the photon pass this method is used to deposit photons
	// on the surface and not, per se, to compute texture color.
	WeightedTextureVector wtextures;
	DBL normaldirection;
	Colour c1;
	Vector2d uvcoords;
	Vector3d rawnormal;
	Vector3d ipoint(isect.IPoint);

	// compute the surface normal
	isect.Object->Normal(*rawnormal, &isect, threadData);

	// I added this to flip the normal if the object is inverted (for CSG).
    // However, I subsequently commented it out for speed reasons - it doesn't
	// make a difference (no pun intended). The preexisting flip code below
	// produces a similar (though more extensive) result. [NK]
	// Actually, we should keep this code to guarantee that Normal_Direction
	// is set properly. [NK]
	if(Test_Flag(isect.Object, INVERTED_FLAG))
		rawnormal = -rawnormal;

	// if the surface normal points away, flip its direction
	normaldirection = dot(rawnormal, Vector3d(ray.Direction));
	if(normaldirection > 0.0)
		rawnormal = -rawnormal;

	Assign_Vector(isect.INormal, *rawnormal);
	Assign_Vector(isect.PNormal, *rawnormal);

	if(Test_Flag(isect.Object, UV_FLAG))
	{
		// get the UV vect of the intersection
		isect.Object->UVCoord(*uvcoords, &isect, threadData);
		// save the normal and UV coords into Intersection
		Assign_UV_Vect(isect.Iuv, *uvcoords);
	}

	// now switch to UV mapping if we need to
	if(Test_Flag(isect.Object, UV_FLAG))
		ipoint = Vector3d(uvcoords.u(), uvcoords.v(), 0.0);

	bool isMultiTextured = Test_Flag(isect.Object, MULTITEXTURE_FLAG);

	// Deal with cutaway textures.
	//
	// isect.Csg is the root-level object in this CSG tree if the intersection object
	// is the child of a multi-tex (cutaway_texture) CSG object. If that member points
	// to a valid CSG object, and the intersected object has a NULL texture, we use
	// the texture list provided by the CSG. Otherwise we just use the provided texture
	// of the intersected object (i.e. we skip the call to Determine_Textures()).
	if((isect.Object->Texture != NULL) && (isect.Csg != NULL) && (Test_Flag (isect.Csg, IS_CSG_OBJECT) != false))
		isMultiTextured = false;

	// get textures and weights
	if(isMultiTextured == true)
	{
		isect.Object->Determine_Textures(&isect, normaldirection > 0.0, wtextures, threadData);
	}
	else if(isect.Object->Texture != NULL)
	{
		if((normaldirection > 0.0) && (isect.Object->Interior_Texture != NULL))
			wtextures.push_back(WeightedTexture(1.0, isect.Object->Interior_Texture)); /* Chris Huff: Interior Texture patch */
		else
			wtextures.push_back(WeightedTexture(1.0, isect.Object->Texture));
	}
	else
	{
		// don't need to do anything as the texture list will be empty.
		// TODO: could we perform these tests earlier ? [cjc]
		return;
	}

	// Now, we perform the lighting calculations by stepping through
	// the list of textures and summing the weighted color.

	for(WeightedTextureVector::iterator i(wtextures.begin()); i != wtextures.end(); i++)
	{
		TextureVector warps(texturePool);

		// if the contribution of this texture is neglectable skip ahead
		if((i->weight < adcBailout) || (i->texture == NULL))
			continue;

		if(photonPass == true)
	    {
			// For the photon pass, colour (and thus c1) represents the
			// light energy being transmitted by the photon.  Because of this, we
			// compute the weighted energy value, then pass it to the texture for
			// processing.
			c1.red() = colour.red() * i->weight;
			c1.green() = colour.green() * i->weight;
			c1.blue() = colour.blue() * i->weight;

			// NOTE that ComputeOneTextureColor is being used for a secondary purpose, and
			// that to place photons on the surface and trigger recursive photon shooting
			ComputeOneTextureColour(c1, i->texture, *warps, ipoint, rawnormal, ray, weight, isect, false, photonPass);
		}
		else
		{
			ComputeOneTextureColour(c1, i->texture, *warps, ipoint, rawnormal, ray, weight, isect, false, photonPass);

			colour.red()    += i->weight * c1.red();
			colour.green()  += i->weight * c1.green();
			colour.blue()   += i->weight * c1.blue();
			colour.transm() += i->weight * c1.transm();
		}
	}
}

void Trace::ComputeOneTextureColour(Colour& resultcolour, TEXTURE *texture, vector<TEXTURE *>& warps, Vector3d& ipoint,
                                    Vector3d& rawnormal, Ray& ray, COLC weight, Intersection& isect, bool shadowflag, bool photonPass)
{
	// NOTE: this method is used by the photon pass to deposit photons on the surface
	// (and not, per se, to compute texture color)
	BLEND_MAP *blendmap = texture->Blend_Map;
	BLEND_MAP_ENTRY *prev, *cur;
	DBL value1, value2;
	Vector3d tpoint;
	Vector2d uvcoords;
	Colour c2;

	switch(texture->Type)
	{
		case NO_PATTERN:
		case PLAIN_PATTERN:
			break;
		case AVERAGE_PATTERN:
		case UV_MAP_PATTERN:
		case BITMAP_PATTERN:
		default:
			warps.push_back(texture);
			break;
	}

	// ipoint - interseciton point (and evaluation point)
	// epoint - evaluation point
	// tpoint - turbulated/transformed point

	if(texture->Type <= LAST_SPECIAL_PATTERN)
	{
		switch(texture->Type)
		{
			case NO_PATTERN:
				Make_ColourA(*resultcolour, 1.0, 1.0, 1.0, 1.0, 1.0);
				break;
			case AVERAGE_PATTERN:
#ifdef ZTV_FUNCTION_WARP
        Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)(warps.back()), threadData);
#else
				Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)(warps.back()));
#endif
				ComputeAverageTextureColours(resultcolour, texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
				break;
			case UV_MAP_PATTERN:
				// Don't bother warping, simply get the UV vect of the intersection
				isect.Object->UVCoord(*uvcoords, &isect, threadData);
				tpoint = Vector3d(uvcoords[U], uvcoords[V], 0.0);
				cur = &(texture->Blend_Map->Blend_Map_Entries[0]);
				ComputeOneTextureColour(resultcolour, cur->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
				break;
			case BITMAP_PATTERN:
#ifdef ZTV_FUNCTION_WARP
        Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture, threadData);
#else
				Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture);
#endif
				ComputeOneTextureColour(resultcolour, material_map(*tpoint, texture), warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
				break;
			case PLAIN_PATTERN:
				if(shadowflag == true)
					ComputeShadowTexture(resultcolour, texture, warps, ipoint, rawnormal, ray, isect);
				else
					ComputeLightedTexture(resultcolour, texture, warps, ipoint, rawnormal, ray, weight, isect);
				break;
			default:
				throw POV_EXCEPTION_STRING("Bad texture type in ComputeOneTextureColour");
		}
	}
	else
	{
		// NK 19 Nov 1999 added Warp_EPoint
#ifdef ZTV_FUNCTION_WARP
    Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture, threadData);
#else
		Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture);
#endif
		value1 = Evaluate_TPat((TPATTERN *)texture, *tpoint, &isect, threadData);

		Search_Blend_Map(value1, blendmap, &prev, &cur);

#ifdef ZTV_FUNCTION_WARP
    Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture, threadData);
#else
		Warp_EPoint(*tpoint, *ipoint, (TPATTERN *)texture);
#endif

		// NK phmap
		if(photonPass)
		{
			if(prev == cur)
				ComputeOneTextureColour(resultcolour, cur->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
			else
			{
				value1 = (value1 - prev->value) / (cur->value - prev->value);
				value2 = 1.0 - value1;
				VScale(*c2, *resultcolour, value1);
				ComputeOneTextureColour(c2, cur->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
				VScale(*c2, *resultcolour, value2);
				ComputeOneTextureColour(c2, prev->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
			}
		}
		else
		{
			ComputeOneTextureColour(resultcolour, cur->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);

			if(prev != cur)
			{
				ComputeOneTextureColour(c2, prev->Vals.Texture, warps, tpoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
				value1 = (value1 - prev->value) / (cur->value - prev->value);
				value2 = 1.0 - value1;
				CLinComb2(*resultcolour, value1, *resultcolour, value2, *c2);
			}
		}
	}
}

void Trace::ComputeAverageTextureColours(Colour& resultcolour, TEXTURE *texture, vector<TEXTURE *>& warps, Vector3d& ipoint,
                                            Vector3d& rawnormal, Ray& ray, COLC weight, Intersection& isect, bool shadowflag, bool photonPass)
{
	BLEND_MAP *bmap = texture->Blend_Map;
	SNGL total = 0.0;
	Colour lc;

	if(photonPass == false)
	{
		Make_Colour(*resultcolour, 0.0, 0.0, 0.0);

		for(int i = 0; i < bmap->Number_Of_Entries; i++)
		{
			SNGL val = bmap->Blend_Map_Entries[i].value;

			ComputeOneTextureColour(lc, bmap->Blend_Map_Entries[i].Vals.Texture, warps, ipoint, rawnormal, ray, weight, isect, shadowflag, photonPass);

			resultcolour[pRED]    += lc[pRED]    * val;
			resultcolour[pGREEN]  += lc[pGREEN]  * val;
			resultcolour[pBLUE]   += lc[pBLUE]   * val;
			resultcolour[pFILTER] += lc[pFILTER] * val;
			resultcolour[pTRANSM] += lc[pTRANSM] * val;

			total += val;
		}

		resultcolour[pRED]    /= total;
		resultcolour[pGREEN]  /= total;
		resultcolour[pBLUE]   /= total;
		resultcolour[pFILTER] /= total;
		resultcolour[pTRANSM] /= total;
	}
	else
	{
		for(int i = 0; i < bmap->Number_Of_Entries; i++)
			total += bmap->Blend_Map_Entries[i].value;

		for(int i = 0; i < bmap->Number_Of_Entries; i++)
		{
			VScale(*lc, *resultcolour, bmap->Blend_Map_Entries[i].value / total);

			ComputeOneTextureColour(lc, bmap->Blend_Map_Entries[i].Vals.Texture, warps, ipoint, rawnormal, ray, weight, isect, shadowflag, photonPass);
		}
	}
}

void Trace::ComputeLightedTexture(Colour& resultcolour, TEXTURE *texture, vector<TEXTURE *>& warps, Vector3d& ipoint,
                                  Vector3d& rawnormal, Ray& ray, COLC weight, Intersection& isect)
{
	Interior *interior;
	TEXTURE *layer;
	int i;
	bool radiosity_done, radiosity_needed;
	int layer_number;
	DBL w1;
	DBL new_Weight;
	DBL att, trans, max_Radiosity_Contribution;
	DBL cos_Angle_Incidence;
	Vector3d layNormal, topNormal;
	Colour attCol, layCol, rflCol, rfrCol, filCol;
	Colour tmpCol, tmp;
	RGBColour ambCol; // Note that there is no gathering of filter or transparency
	bool one_colour_found, colour_found;
	bool tir_occured;
	auto_ptr<PhotonGatherer> surfacePhotonGatherer(NULL); // TODO FIXME - auto_ptr why?

	WNRXVector listWNRX(wnrxPool);

	// ResCol builds up the apparent visible color of the point.
	// Only RGB components are significant.  You can't "see" transparency --
	// you see the color of whatever is behind the transparent surface.
	// This color includes the visible appearence of what is behind the
	// transparency so only RGB is needed.
	Make_ColourA(*resultcolour, 0.0, 0.0, 0.0, 0.0, 0.0);

	// FilCol serves two purposes.  It accumulates the filter properties
	// of a multi-layer texture so that if a ray makes it all the way through
	// all layers, the color of object behind is filtered by this object.
	// It also is used to attenuate how much of an underlayer you
	// can see in a layered texture.  Note that when computing the reflective
	// properties of a layered texture, the upper layers don't filter the
	// light from the lower layers -- the layer colors add together (even
	// before we added additive transparency via the "transmit" 5th
	// color channel).  However when computing the transmitted rays, all layers
	// filter the light from any objects behind this object. [CY 1/95]

	// NK layers - switched transmit component to zero
	Make_ColourA(*filCol, 1.0, 1.0, 1.0, 1.0, 0.0);

	trans = 1.0;

	// Add in radiosity (stochastic interreflection-based ambient light) if desired
	radiosity_done = false;

	// This block just sets up radiosity for the code inside the loop, which is first-time-through.
	radiosity_needed = (sceneData->parsedRadiositySettings.Enabled == true) &&
	                   (radiosity.CheckRadiosityTraceLevel() == true) &&
	                   (Test_Flag(isect.Object, NO_RADIOSITY_FLAG) == false);

	// Loop through the layers and compute the ambient, diffuse,
	// phong and specular for these textures.
	one_colour_found = false;

	if(sceneData->photonSettings.photonsEnabled && sceneData->surfacePhotonMap.numPhotons > 0)
		surfacePhotonGatherer.reset(new PhotonGatherer(&sceneData->surfacePhotonMap, sceneData->photonSettings));

	for(layer_number = 0, layer = texture; (layer != NULL) && (trans > adcBailout); layer_number++, layer = (TEXTURE *)layer->Next)
	{
		// Get perturbed surface normal.
		layNormal = rawnormal;

		if((qualityFlags & Q_NORMAL) && (layer->Tnormal != NULL))
		{
			for(vector<TEXTURE *>::iterator i(warps.begin()); i != warps.end(); i++)
				Warp_Normal(*layNormal, *layNormal, (TPATTERN *)(*i), Test_Flag((*i), DONT_SCALE_BUMPS_FLAG));

			Perturb_Normal(*layNormal, layer->Tnormal, *ipoint, &isect, threadData);

			if((Test_Flag(layer->Tnormal, DONT_SCALE_BUMPS_FLAG)))
				VNormalizeEq(*layNormal);

			for(vector<TEXTURE *>::reverse_iterator i(warps.rbegin()); i != warps.rend(); i++)
				UnWarp_Normal(*layNormal, *layNormal, (TPATTERN *)(*i), Test_Flag((*i), DONT_SCALE_BUMPS_FLAG));
		}

		// Store top layer normal.
		if(layer_number == 0)
			topNormal = layNormal;

		// Get surface colour.
		new_Weight = weight * trans;
		colour_found = Compute_Pigment(*layCol, layer->Pigment, *ipoint, &isect, threadData);

		// If a valid color was returned set one_colour_found to true.
		// An invalid color is returned if a surface point is outside
		// an image map used just once.
		one_colour_found = (one_colour_found || colour_found);

		// This section of code used to be the routine Compute_Reflected_Colour.
		// I copied it in here to rearrange some of it more easily and to
		// see if we could eliminate passing a zillion parameters for no
		// good reason. [CY 1/95]

		if(qualityFlags & Q_FULL_AMBIENT)
		{
			// Only use top layer and kill transparency if low quality.
			resultcolour = layCol;
			resultcolour[pFILTER] = 0.0;
			resultcolour[pTRANSM] = 0.0;
		}
		else
		{
			// Store vital information for later reflection.
			listWNRX->push_back(WNRX(new_Weight, layNormal, Colour(), layer->Finish->Reflect_Exp));

			// angle-dependent reflectivity
			VDot(cos_Angle_Incidence, ray.Direction, *layNormal);
			cos_Angle_Incidence *= -1.0;

			if((isect.Object->interior != NULL) || (layer->Finish->Reflection_Type != 1)) 
			{
				ComputeReflectivity(listWNRX->back().weight, listWNRX->back().reflec,
				                    Colour(layer->Finish->Reflection_Max), Colour(layer->Finish->Reflection_Min), 
				                    layer->Finish->Reflection_Type, layer->Finish->Reflection_Falloff, 
				                    cos_Angle_Incidence, ray, isect.Object->interior);
			}
			else
				throw POV_EXCEPTION_STRING("Reflection_Type 1 used with no interior."); // TODO FIXME - wrong place to report this [trf]

			// for metallic reflection, apply the surface color using the fresnel equation
			// (use the same equaltion as "metallic" in phong and specular 
			if(layer->Finish->Reflect_Metallic != 0.0)
			{
				DBL R_M = layer->Finish->Reflect_Metallic;

				DBL x = fabs(acos(cos_Angle_Incidence)) / M_PI_2;
				DBL F = 0.014567225 / Sqr(x - 1.12) - 0.011612903;
				F = min(1.0, max(0.0, F));

				listWNRX->back().reflec[pRED]   *= (1.0 + R_M * (1.0 - F) * (layCol[pRED]   - 1.0));
				listWNRX->back().reflec[pGREEN] *= (1.0 + R_M * (1.0 - F) * (layCol[pGREEN] - 1.0));
				listWNRX->back().reflec[pBLUE]  *= (1.0 + R_M * (1.0 - F) * (layCol[pBLUE]  - 1.0));
			}

			// NK - I think we SHOULD do something like this: (to apply the layer's color) */
			// listWNRX->back().reflec.red()*=FilCol.red();
			// listWNRX->back().reflec.green()*=FilCol.green();
			// listWNRX->back().reflec.blue()*=FilCol.blue();

			att = (1.0 - (layCol[pFILTER] * max3(layCol.red(), layCol[pGREEN], layCol[pBLUE]) + layCol[pTRANSM]));

			// now compute the BRDF contribution
			Make_Colour(*tmpCol, 0.0, 0.0, 0.0);

			// Add ambient contribution.
			if(radiosity_needed)
			{
				// if radiosity calculation needed, but not yet done, do it now
				if(radiosity_done == false)
				{
					// calculate max possible contribution of radiosity, to see if calculating it is worthwhile
					tmp[pRED]   = filCol[pRED]   * att * layCol[pRED]   * layer->Finish->Diffuse;
					tmp[pGREEN] = filCol[pGREEN] * att * layCol[pGREEN] * layer->Finish->Diffuse;
					tmp[pBLUE]  = filCol[pBLUE]  * att * layCol[pBLUE]  * layer->Finish->Diffuse;

					max_Radiosity_Contribution = GREY_SCALE3(tmp[pRED], tmp[pGREEN], tmp[pBLUE]);

					if(max_Radiosity_Contribution > adcBailout * 3.0)
					{
						radiosity.ComputeAmbient(ray, ipoint, rawnormal, layNormal, ambCol, weight * max_Radiosity_Contribution);
						radiosity_done = true;
					}
				}

				tmpCol[pRED]   += filCol[pRED]   * att * layCol[pRED]   * ambCol[pRED]   * layer->Finish->Diffuse;
				tmpCol[pGREEN] += filCol[pGREEN] * att * layCol[pGREEN] * ambCol[pGREEN] * layer->Finish->Diffuse;
				tmpCol[pBLUE]  += filCol[pBLUE]  * att * layCol[pBLUE]  * ambCol[pBLUE]  * layer->Finish->Diffuse;
			}

			tmpCol[pRED]   += filCol[pRED]   * att * layCol[pRED]   * layer->Finish->Ambient[pRED]   * sceneData->ambientLight[pRED];
			tmpCol[pGREEN] += filCol[pGREEN] * att * layCol[pGREEN] * layer->Finish->Ambient[pGREEN] * sceneData->ambientLight[pGREEN];
			tmpCol[pBLUE]  += filCol[pBLUE]  * att * layCol[pBLUE]  * layer->Finish->Ambient[pBLUE]  * sceneData->ambientLight[pBLUE];

			// make sure that radiosity/ambient doesn't get multiplied by FilCol[] twice,
			// so we add it to ResCol NOW and then go on to do the diffuse stuff
			VAddEq(*resultcolour, *tmpCol);
			Make_Colour(*tmpCol, 0.0, 0.0, 0.0);

			// set up the "litObjectIgnoresPhotons" flag (thread variable) so that
			// ComputeShadowColour will know whether or not this lit object is
			// ignoring photons, which affects partial-shadowing (i.e. filter and transmit)
			threadData->litObjectIgnoresPhotons = Test_Flag(isect.Object,PH_IGNORE_PHOTONS_FLAG);

			// Add diffuse, phong, specular, and iridescence contribution.
			Vector3d tmpIPoint(isect.IPoint);
			if((layer->Finish->Diffuse != 0.0) || (layer->Finish->Specular != 0.0) || (layer->Finish->Phong != 0.0))
				ComputeDiffuseLight(layer->Finish, tmpIPoint, ray, layNormal, layCol, tmpCol, att, isect.Object);

			// apply the previous layers' filter color [NK]
			tmpCol[pRED]   *= filCol[pRED];
			tmpCol[pGREEN] *= filCol[pGREEN];
			tmpCol[pBLUE]  *= filCol[pBLUE];

			// now add the temp color to the resulting color
			VAddEq(*resultcolour, *tmpCol);

			if(sceneData->photonSettings.photonsEnabled && sceneData->surfacePhotonMap.numPhotons > 0)
			{
				// NK phmap - now do the same for the photons in the area
				if(!Test_Flag(isect.Object, PH_IGNORE_PHOTONS_FLAG))
				{
					Vector3d tmpIPoint(isect.IPoint);
					ComputePhotonDiffuseLight(layer->Finish, tmpIPoint, ray, layNormal, rawnormal, layCol, tmpCol, att, isect.Object, *surfacePhotonGatherer);
					tmpCol[pRED]   *= filCol[pRED];
					tmpCol[pGREEN] *= filCol[pGREEN];
					tmpCol[pBLUE]  *= filCol[pBLUE];
					VAddEq(*resultcolour, *tmpCol);
				}
			}
		}

		// Get new filter color.
		if(colour_found)
		{
			filCol[pRED]   *= (layCol[pRED]   * layCol[pFILTER] + layCol[pTRANSM]);
			filCol[pGREEN] *= (layCol[pGREEN] * layCol[pFILTER] + layCol[pTRANSM]);
			filCol[pBLUE]  *= (layCol[pBLUE]  * layCol[pFILTER] + layCol[pTRANSM]);
			// note FilCol[pFILTER] stays at 1.0, [pTRANSM] stays at 0.0

			if(layer->Finish->Conserve_Energy)
			{
				// adjust filcol based on reflection
				// this would work so much better with r,g,b,rt,gt,bt
				filCol[pRED]   *= min(1.0, 1.0 - listWNRX->back().reflec[pRED]);
				filCol[pGREEN] *= min(1.0, 1.0 - listWNRX->back().reflec[pGREEN]);
				filCol[pBLUE]  *= min(1.0, 1.0 - listWNRX->back().reflec[pBLUE]);
			}
		}

		// Get new remaining translucency.
		trans = min(1.0, fabs(filCol[pFILTER] * GREY_SCALE(filCol)) + fabs(filCol[pTRANSM])); // NK layers - changed this
	}

	// Calculate transmitted component.
	//
	// If the surface is translucent a transmitted ray is traced
	// and its contribution is added to the total ResCol after
	// filtering it by FilCol.
	tir_occured = false;

	if(((interior = isect.Object->interior) != NULL) && (trans > adcBailout) && (qualityFlags & Q_REFRACT))
	{
		w1 = fabs(filCol[pFILTER]) * max3(fabs(filCol[pRED]), fabs(filCol[pGREEN]), fabs(filCol[pBLUE]));
		new_Weight = weight * w1;

		// Trace refracted ray.
		Vector3d tmpIPoint(isect.IPoint);
		Colour tempcolor;

		tir_occured = ComputeRefraction(interior, tmpIPoint, ray, topNormal, rawnormal, tempcolor, new_Weight);

		if(tir_occured == true)
			rfrCol += tempcolor;
		else
			rfrCol = tempcolor;

		// Get distance based attenuation.
		attCol[pRED] = attCol[pGREEN] = attCol[pBLUE] = interior->Old_Refract;

		if((interior != NULL) && ray.IsInterior(interior) == true)
		{
			if(fabs(interior->Fade_Distance) > EPSILON)
			{
				// NK attenuate
				if(interior->Fade_Power >= 1000)
				{
					attCol[pRED]   *= exp(-(1.0 - interior->Fade_Colour[pRED])   * isect.Depth / interior->Fade_Distance);
					attCol[pGREEN] *= exp(-(1.0 - interior->Fade_Colour[pGREEN]) * isect.Depth / interior->Fade_Distance);
					attCol[pBLUE]  *= exp(-(1.0 - interior->Fade_Colour[pBLUE])  * isect.Depth / interior->Fade_Distance);
				}
				else
				{
					att = 1.0 + pow(isect.Depth / interior->Fade_Distance, (DBL)interior->Fade_Power);
					attCol[pRED]   *= interior->Fade_Colour[pRED]   + (1.0 - interior->Fade_Colour[pRED]) / att;
					attCol[pGREEN] *= interior->Fade_Colour[pGREEN] + (1.0 - interior->Fade_Colour[pGREEN]) / att;
					attCol[pBLUE]  *= interior->Fade_Colour[pBLUE]  + (1.0 - interior->Fade_Colour[pBLUE]) / att;
				}
			}
		}

		// If total internal reflection occured the transmitted light is not filtered.
		if(tir_occured)
		{
			resultcolour[pRED]   += attCol[pRED]   * rfrCol[pRED];
			resultcolour[pGREEN] += attCol[pGREEN] * rfrCol[pGREEN];
			resultcolour[pBLUE]  += attCol[pBLUE]  * rfrCol[pBLUE];
			// NOTE: pTRANSM (alpha channel) stays zero
		}
		else
		{
			if(one_colour_found)
			{
				resultcolour[pRED]   += attCol[pRED]   * rfrCol[pRED]   * (filCol[pRED]   * filCol[pFILTER] + filCol[pTRANSM]);
				resultcolour[pGREEN] += attCol[pGREEN] * rfrCol[pGREEN] * (filCol[pGREEN] * filCol[pFILTER] + filCol[pTRANSM]);
				resultcolour[pBLUE]  += attCol[pBLUE]  * rfrCol[pBLUE]  * (filCol[pBLUE]  * filCol[pFILTER] + filCol[pTRANSM]);
				// We need to know the transmittance value for the alpha channel. [DB]
				resultcolour[pTRANSM] = GREY_SCALE(attCol) * rfrCol[pTRANSM] * trans;
			}
			else
			{
				resultcolour[pRED]   += attCol[pRED]   * rfrCol[pRED];
				resultcolour[pGREEN] += attCol[pGREEN] * rfrCol[pGREEN];
				resultcolour[pBLUE]  += attCol[pBLUE]  * rfrCol[pBLUE];
				// We need to know the transmittance value for the alpha channel. [DB]
				resultcolour[pTRANSM] = GREY_SCALE(attCol) * rfrCol[pTRANSM];
			}
		}
	}

	// Calculate reflected component.
	//
	// If total internal reflection occured all reflections using
	// TopNormal are skipped.
	if(qualityFlags & Q_REFLECT)
	{
		for(i = 0; i < layer_number; i++)
		{
			if((!tir_occured) ||
			   (fabs(topNormal[X]-(*listWNRX)[i].normal[X]) > EPSILON) ||
			   (fabs(topNormal[Y]-(*listWNRX)[i].normal[Y]) > EPSILON) ||
			   (fabs(topNormal[Z]-(*listWNRX)[i].normal[Z]) > EPSILON))
			{
				if(((*listWNRX)[i].reflec[pRED] != 0.0) || ((*listWNRX)[i].reflec[pGREEN] != 0.0) || ((*listWNRX)[i].reflec[pBLUE] != 0.0))
				{
					Vector3d tmpIPoint(isect.IPoint);
					Colour tempCol;

					ComputeReflection(tmpIPoint, ray, (*listWNRX)[i].normal, rawnormal, tempCol, (*listWNRX)[i].weight);
					rflCol += tempCol;

					if((*listWNRX)[i].reflex != 1.0)
					{
						resultcolour[pRED]    += (*listWNRX)[i].reflec[pRED]   * pow(rflCol[pRED],   (*listWNRX)[i].reflex);
						resultcolour[pGREEN]  += (*listWNRX)[i].reflec[pGREEN] * pow(rflCol[pGREEN], (*listWNRX)[i].reflex);
						resultcolour[pBLUE]   += (*listWNRX)[i].reflec[pBLUE]  * pow(rflCol[pBLUE],  (*listWNRX)[i].reflex);
					}
					else
					{
						resultcolour[pRED]   += (*listWNRX)[i].reflec[pRED]   * rflCol[pRED];
						resultcolour[pGREEN] += (*listWNRX)[i].reflec[pGREEN] * rflCol[pGREEN];
						resultcolour[pBLUE]  += (*listWNRX)[i].reflec[pBLUE]  * rflCol[pBLUE];
					}
				}
			}
		}
	}

	// Calculate participating media effects.
	if((qualityFlags & Q_VOLUME) && (!ray.GetInteriors().empty()) && (ray.IsHollowRay() == true))
		media.ComputeMedia(ray.GetInteriors(), ray, isect, resultcolour);
}

void Trace::ComputeShadowTexture(Colour& filtercolour, TEXTURE *texture, vector<TEXTURE *>& warps, Vector3d& ipoint,
                                    Vector3d& rawnormal, Ray& ray, Intersection& isect)
{
	Interior *interior = isect.Object->interior;
	TEXTURE *layer;
	DBL caustics, dotval, k;
	Vector3d layer_Normal;
	Colour refraction, layer_Pigment_Colour;
	bool one_colour_found, colour_found;

	Make_ColourA(*filtercolour, 1.0, 1.0, 1.0, 1.0, 0.0);

	one_colour_found = false;

	for(layer = texture; (layer != NULL) && (fabs(filtercolour[pFILTER]) + fabs(filtercolour[pTRANSM]) > adcBailout); layer = (TEXTURE *)layer->Next)
	{
		colour_found = Compute_Pigment(*layer_Pigment_Colour, layer->Pigment, *ipoint, &isect, threadData);

		if(colour_found)
		{
			one_colour_found = true;

			filtercolour[pRED]    *= layer_Pigment_Colour[pRED]   * layer_Pigment_Colour[pFILTER] + layer_Pigment_Colour[pTRANSM];
			filtercolour[pGREEN]  *= layer_Pigment_Colour[pGREEN] * layer_Pigment_Colour[pFILTER] + layer_Pigment_Colour[pTRANSM];
			filtercolour[pBLUE]   *= layer_Pigment_Colour[pBLUE]  * layer_Pigment_Colour[pFILTER] + layer_Pigment_Colour[pTRANSM];
		}

		// Get normal for faked caustics (will rewrite later to cache).
		if((interior != NULL) && ((caustics = interior->Caustics) != 0.0))
		{
			Assign_Vector(*layer_Normal, *rawnormal);

			if((qualityFlags & Q_NORMAL) && (layer->Tnormal != NULL))
			{
					for(vector<TEXTURE *>::iterator i(warps.begin()); i != warps.end(); i++)
						Warp_Normal(*layer_Normal, *layer_Normal, (TPATTERN *)(*i), Test_Flag((*i), DONT_SCALE_BUMPS_FLAG));

					Perturb_Normal(*layer_Normal, layer->Tnormal, *ipoint, &isect, threadData);

					if((Test_Flag(layer->Tnormal,DONT_SCALE_BUMPS_FLAG)))
						VNormalizeEq(*layer_Normal);

					for(vector<TEXTURE *>::reverse_iterator i(warps.rbegin()); i != warps.rend(); i++)
						UnWarp_Normal(*layer_Normal, *layer_Normal, (TPATTERN *)(*i), Test_Flag((*i), DONT_SCALE_BUMPS_FLAG));
			}

			// Get new filter/transmit values.
			VDot(dotval, *layer_Normal, ray.Direction);

			k = (1.0 + pow(fabs(dotval), caustics));

			filtercolour[pRED]   *= k;
			filtercolour[pGREEN] *= k;
			filtercolour[pBLUE]  *= k;
		}
	}

	// Get distance based attenuation.
	if(interior != NULL)
	{
		Make_Colour(*refraction, 1.0, 1.0, 1.0);

		if(ray.IsInterior(interior) == true)
		{
			if((interior->Fade_Power > 0.0) && (fabs(interior->Fade_Distance) > EPSILON))
			{
				// NK - attenuation
				if(interior->Fade_Power>=1000)
				{
					refraction.red() *= exp(-(1.0 - interior->Fade_Colour.red()) * isect.Depth / interior->Fade_Distance);
					refraction.green() *= exp(-(1.0 - interior->Fade_Colour.green()) * isect.Depth / interior->Fade_Distance);
					refraction.blue() *= exp(-(1.0 - interior->Fade_Colour.blue()) * isect.Depth / interior->Fade_Distance);
				}
				else
				{
					k = 1.0 + pow(isect.Depth / interior->Fade_Distance, (DBL)interior->Fade_Power);
					refraction.red() *= interior->Fade_Colour.red() + (1 - interior->Fade_Colour.red()) / k;
					refraction.green() *= interior->Fade_Colour.green() + (1 - interior->Fade_Colour.green()) / k;
					refraction.blue() *= interior->Fade_Colour.blue() + (1 - interior->Fade_Colour.blue()) / k;
				}
			}
		}
	}
	else
		Make_Colour(*refraction, 1.0, 1.0, 1.0);

	// Get distance based attenuation.
	filtercolour[pRED]    *= refraction.red();
	filtercolour[pGREEN]  *= refraction.green();
	filtercolour[pBLUE]   *= refraction.blue();

	// Calculate participating media effects.
	if((qualityFlags & Q_VOLUME) && (!ray.GetInteriors().empty()) && (ray.IsHollowRay() == true))
		media.ComputeMedia(ray.GetInteriors(), ray, isect, filtercolour);
}

void Trace::ComputeReflection(Vector3d& ipoint, Ray& ray, Vector3d& normal, Vector3d& rawnormal, Colour& colour, COLC weight)
{
	Ray nray(ray);
	DBL n, n2;

	nray.SetFlags(Ray::ReflectionRay, ray);

	// The rest of this is essentally what was originally here, with small changes.
	VDot(n, ray.Direction, *normal);
	n *= -2.0;
	VAddScaled(nray.Direction, ray.Direction, n, *normal);

	// Nathan Kopp & CEY 1998 - Reflection bugfix
	// if the new ray is going the opposite direction as raw normal, we
	// need to fix it.
	VDot(n, nray.Direction, *rawnormal);

	if(n < 0.0)
	{
		// It needs fixing. Which kind?
		VDot(n2, nray.Direction, *normal);

		if(n2 < 0.0)
		{
			// reflected inside rear virtual surface. Reflect Ray using Raw_Normal
			VDot(n, ray.Direction, *rawnormal);
			n *= -2.0;
			VAddScaled(nray.Direction, ray.Direction, n, *rawnormal);
		}
		else
		{
			// Double reflect NRay using Raw_Normal
			// VDot(n,New_Ray.Direction,Jitter_Raw_Normal); - kept the old n around
			n *= -2.0;
			VAddScaledEq(nray.Direction, n, *rawnormal);
		}
	}

	VNormalizeEq(nray.Direction);
	Assign_Vector(nray.Origin, *ipoint);
	threadData->Stats[Reflected_Rays_Traced]++;

	// Trace reflected ray.
	TraceRay(nray, colour, weight);
}

bool Trace::ComputeRefraction(Interior *interior, Vector3d& ipoint, Ray& ray, Vector3d& normal, Vector3d& rawnormal, Colour& colour, COLC weight)
{
	Ray nray(ray);
	Vector3d localnormal;
	DBL n, ior, dispersion;
	unsigned int dispersionelements = interior->Disp_NElems;
	bool havedispersion = (dispersionelements > 0);

	nray.SetFlags(Ray::RefractionRay, ray);

	// Set up new ray.
	Assign_Vector(nray.Origin, *ipoint);

	// Get ratio of iors depending on the interiors the ray is traversing.
	if(ray.GetInteriors().empty())
	{
		// The ray is entering from the atmosphere.
		nray.AppendInterior(interior);

		ior = sceneData->atmosphereIOR / interior->IOR;
		if(havedispersion == true)
			dispersion = sceneData->atmosphereDispersion / interior->Dispersion;
	}
	else
	{
		// The ray is currently inside an object.
		if(nray.RemoveInterior(interior) == true) // The ray is leaving the current object.
		{
			if(nray.GetInteriors().empty())
			{
				// The ray is leaving into the atmosphere
				ior = interior->IOR / sceneData->atmosphereIOR;
				if(havedispersion == true)
					dispersion = interior->Dispersion / sceneData->atmosphereDispersion;
			}
			else
			{
				// The ray is leaving into another object.
				ior = interior->IOR / nray.GetInteriors().back()->IOR;
				if(havedispersion == true)
				{
					dispersion = interior->Dispersion / nray.GetInteriors().back()->Dispersion;
					dispersionelements = max(dispersionelements, (unsigned int)(nray.GetInteriors().back()->Disp_NElems));
				}
			}
		}
		else
		{
			// The ray is entering a new object.
			ior = nray.GetInteriors().back()->IOR / interior->IOR;
			if(havedispersion == true)
				dispersion = nray.GetInteriors().back()->Dispersion / interior->Dispersion;

			nray.AppendInterior(interior);
		}
	}

	// Do the two mediums traversed have the sampe indices of refraction?
	if((fabs(ior - 1.0) < EPSILON) && (fabs(dispersion - 1.0) < EPSILON))
	{
		// Only transmit the ray.
		Assign_Vector(nray.Direction, ray.Direction);
		// Trace a transmitted ray.
		threadData->Stats[Transmitted_Rays_Traced]++;

		colour.clear();
		TraceRay(nray, colour, weight);
	}
	else
	{
		// Refract the ray.
		VDot(n, ray.Direction, *normal);

		if(n <= 0.0)
		{
			localnormal = normal;
			n = -n;
		}
		else
			localnormal = -normal;


		// TODO FIXME: also for first radiosity pass ? (see line 3272 of v3.6 lighting.cpp)
		if((havedispersion == false) || (fabs (dispersion - 1.0) < EPSILON))
			return TraceRefractionRay(ipoint, ray, nray, ior, n, normal, rawnormal, localnormal, colour, weight);
		else if(ray.IsMonochromaticRay() == true)
			return TraceRefractionRay(ipoint, ray, nray, ior * ray.GetDispersionFactor(), n, normal, rawnormal, localnormal, colour, weight);
		else
		{
			RGBColour sumcol, hue;
			DBL ior_mult = pow(dispersion, 1.0 / DBL(dispersionelements - 1));

			ior /= sqrt(dispersion);

			for(unsigned int i = 1; i <= dispersionelements; i++)
			{
				Colour tempcolour;

				// NB setting the dispersion factor also causes the MonochromaticRay flag to be set
				nray.SetDispersionFactor(ComputeDispersionFactor(dispersion, i, dispersionelements));

				(void)TraceRefractionRay(ipoint, ray, nray, ior, n, normal, rawnormal, localnormal, tempcolour, weight);

				ComputeDispersion(hue, i, dispersionelements);
				sumcol += RGBColour(tempcolour) * hue;
				ior *= ior_mult;
			}

			colour = Colour(sumcol / DBL(dispersionelements) * 3.0);
		}
	}

	return false;
}

bool Trace::TraceRefractionRay(Vector3d& ipoint, Ray& ray, Ray& nray, DBL ior, DBL n, Vector3d& normal, Vector3d& rawnormal, Vector3d& localnormal, Colour& colour, COLC weight)
{
	// Compute refrated ray direction using Heckbert's method.
	DBL t = 1.0 + Sqr(ior) * (Sqr(n) - 1.0);

	if(t < 0.0)
	{
		Colour tempcolour;

		// Total internal reflection occures.
		threadData->Stats[Internal_Reflected_Rays_Traced]++;
		ComputeReflection(ipoint, ray, normal, rawnormal, tempcolour, weight);
		colour += tempcolour;

		return true;
	}

	t = ior * n - sqrt(t);

	VLinComb2(nray.Direction, ior, ray.Direction, t, *localnormal);

	// Trace a refracted ray.
	threadData->Stats[Refracted_Rays_Traced]++;

	colour.clear();
	TraceRay(nray, colour, weight);

	return false;
}

void Trace::ComputeDispersion(RGBColour& hue, unsigned int elem, unsigned int nelems)
{
	// Gives color to a dispersion element.
	// 
	// Requirements:
	// * Sum of all hues must add to white (or white*constand)
	//   (white tiles seen through glass should still be white)
	//   
	// * Each hue must be maximally saturated, bright
	//   (The code shown here cheats a little)
	//   [RLP: maximally saturated, anyway.  maximally bright is a
	//    mistake.  And the new code no longer cheats.]
	//   
	// * colors must range from red at elem=1 to violet at elem=nelems
	// 
	// The equations herein were derived by back-of-the-envelope
	// curve fitting to some RGB color-matching function tables 
	// I found on the web somewhere.  I could have just interpolated
	// those tables, but I think this gives results that are as good 
	// and scale well.  The various magic numbers below were 
	// determined empirically to match four important criteria:
	// 
	// 1) The peak for a given element must be at the same place as
	//    on the color-matching table.
	// 2) The width of a given element's curve must be about the same.
	// 3) The width of the clipped portion of a given element must
	//    be about the same.
	// 4) The integral of each element's curve must be approximately
	//    the same as the integral of each of the other elements.

	// When I derived the functions, I went with the assumption that
	// 0 is near-UV, and 1 is near-IR.  When I looked at the code, I
	// realized that it wanted exactly the reverse.  Thus the "1.0-"
	SNGL hc = 0.964 - 0.934 * ((SNGL)(elem - 1) / (SNGL)(nelems - 1));

	// The blue component.  The peak is at hc = 0.28, and there is no
	// clipped part.  0.98 is a scaling factor to make the integrals
	// come out even.  4 determines the width of the nonzero part 
	// of the curve; the larger the narrower.  The 1 helps determine
	// the width of the clipped portion (but blue has no clipped 
	// portion.)  Four constraints, four magic numbers.  Who'da thunk
	// it?
	SNGL b = 0.98 * (1.0 - Sqr(4.0 * (hc-0.28)));
	if(b < 0.0)
		b = 0.0;
	hue[pBLUE] = b;

	// This is substantially the same code as the blue code above,
	// with different magic numbers.
	SNGL g = 0.97 * (1.1 - Sqr(4.5 * (hc - 0.57)));
	if(g < 0.0)
		g = 0.0;
	hue[pGREEN] = g;

	// This is also substantially the same code as the blue, with 
	// one exception: the red component has a second, smaller peak
	// at the violet end of the spectrum.  That is represented by
	// the second set of magic numbers below.  Also, red is the 
	// component to which the others are standardized (because it
	// had the smallest integral to begin with) so there is no 
	// 0.9x fudge-factor.
	SNGL r = 1.15 - Sqr(5.0 * (hc - 0.75));
	if(r < 0.0)
		r = 0.12 - Sqr(4.0 * (hc - 0.12));
	if(r < 0.0)
		r = 0.0;
	hue[pRED] = r;
}

DBL Trace::ComputeDispersionFactor(DBL disp, unsigned int elem, unsigned int nelems)
{
	return pow(disp, DBL(elem - 1) / DBL(nelems - 1) - 0.5);
}

void Trace::ComputeDiffuseLight(FINISH *finish, Vector3d& ipoint, Ray& eye, Vector3d& layer_normal, Colour& layer_pigment_colour,
                                Colour& colour, DBL attenuation, ObjectPtr object)
{
	Vector3d reye;

	if((finish->Diffuse == 0.0) && (finish->Specular == 0.0) && (finish->Phong == 0.0))
		return;

	if(finish->Specular != 0.0)
	{
		reye[X] = -eye.Direction[X];
		reye[Y] = -eye.Direction[Y];
		reye[Z] = -eye.Direction[Z];
	}

	// global light sources, if not turned off for this object
	if((object->Flags & NO_GLOBAL_LIGHTS_FLAG) != NO_GLOBAL_LIGHTS_FLAG)
	{
		for(int i = 0; i < threadData->lightSources.size(); i++)
			ComputeOneDiffuseLight(threadData->lightSources[i], reye, finish, ipoint, eye, layer_normal, layer_pigment_colour, colour, attenuation, object);
	}

	// local light sources from a light group, if any
	if(!object->LLights.empty())
	{
		for(int i = 0; i < object->LLights.size(); i++)
			ComputeOneDiffuseLight(object->LLights[i], reye, finish, ipoint, eye, layer_normal, layer_pigment_colour, colour, attenuation, object);
	}
}

void Trace::ComputePhotonDiffuseLight(FINISH *Finish, Vector3d& IPoint, Ray& Eye,  Vector3d&  Layer_Normal,  Vector3d& Raw_Normal, Colour& Layer_Pigment_Colour, Colour& colour, DBL Attenuation, ObjectPtr Object, PhotonGatherer& gatherer)
{
  DBL Cos_Shadow_Angle;
  Ray Light_Source_Ray;
  Vector3d REye;
  Colour Light_Colour, Colour2;
  DBL r;
  int n;
  int j;
  DBL thisDensity=0;
  DBL prevDensity=0.0000000000000001; // avoid div-by-zero error
  int expanded = false;
  DBL att;  // attenuation for lambertian compensation & filters

  if (!sceneData->photonSettings.photonsEnabled || sceneData->surfacePhotonMap.numPhotons<1)
  {
    Make_ColourA(*colour,0.0,0.0,0.0,0.0,0.0);
    return;
  }

  if ((Finish->Diffuse == 0.0) && (Finish->Specular == 0.0) && (Finish->Phong == 0.0))
  {
    Make_ColourA(*colour,0.0,0.0,0.0,0.0,0.0);
    return;
  }

  //  [NK] moved this to ComputeLightedTexture so that we re-use the same gatherer
  // for multiple layers of texture at the same point
  //PhotonGatherer gatherer(&sceneData->surfacePhotonMap, sceneData->photonSettings);

  // statistics
  //TODO FIXME STATS
  // Increase_Counter(stats[Gather_Performed_Count]);

  if (Finish->Specular != 0.0)
  {
    REye[X] = -Eye.Direction[X];
    REye[Y] = -Eye.Direction[Y];
    REye[Z] = -Eye.Direction[Z];
  }

  Make_Colour(*colour,0,0,0);

  if(gatherer.gathered)
	  r = gatherer.alreadyGatheredRadius;
  else
	r=gatherer.gatherPhotonsAdaptive(*IPoint,*Layer_Normal,true);

  n=gatherer.gatheredPhotons.numFound;

  Make_Colour(*Colour2,0,0,0);

  // now go through these photons and add up their contribution
    for(j=0; j<n; j++)
    {
      // DBL theta,phi;
      int theta,phi;

      // convert small color to normal color
      photonRgbe2colour(*Light_Colour, gatherer.gatheredPhotons.photonGatherList[j]->Colour);

      // convert theta/phi to vector direction 
      // Use a pre-computed array of sin/cos to avoid many calls to the
      // sin() and cos() functions.  These arrays were initialized in
      // InitBacktraceEverything.
      theta = gatherer.gatheredPhotons.photonGatherList[j]->theta+127;
      phi = gatherer.gatheredPhotons.photonGatherList[j]->phi+127;
      
      Light_Source_Ray.Direction[Y] = sinCosData.sinTheta[theta];
      Light_Source_Ray.Direction[X] = sinCosData.cosTheta[theta];

      Light_Source_Ray.Direction[Z] = Light_Source_Ray.Direction[X]*sinCosData.sinTheta[phi];
      Light_Source_Ray.Direction[X] = Light_Source_Ray.Direction[X]*sinCosData.cosTheta[phi];

      VSub(Light_Source_Ray.Origin, gatherer.gatheredPhotons.photonGatherList[j]->Loc, Light_Source_Ray.Direction);

      // this compensates for real lambertian (diffuse) lighting (see paper) 
      // use raw normal, not layer normal 
      // VDot(att, Layer_Normal, Light_Source_Ray.Direction); 
      VDot(att, *Raw_Normal, Light_Source_Ray.Direction);
      if (att>1) att=1.0;
      if (att<.1) att = 0.1; // limit to 10x - otherwise we get bright dots 
      att = 1.0 / fabs(att);

      // do gaussian filter 
      //att *= 0.918*(1.0-(1.0-exp((-1.953) * gatherer.photonDistances[j])) / (1.0-exp(-1.953)) );
      // do cone filter 
      //att *= 1.0-(sqrt(gatherer.photonDistances[j])/(4.0 * r)) / (1.0-2.0/(3.0*4.0));

      VScaleEq(*Light_Colour,att);

      // See if light on far side of surface from camera. 
      if (!(Test_Flag(Object, DOUBLE_ILLUMINATE_FLAG)))
      {
        VDot(Cos_Shadow_Angle, *Layer_Normal, Light_Source_Ray.Direction);
        if (Cos_Shadow_Angle < EPSILON)
          continue;
      }

      // now add diffuse, phong, specular, irid contribution 
      if (Finish->Diffuse > 0.0)
      {
        ComputeDiffuseColour(Finish,Light_Source_Ray,Layer_Normal,Colour2,Light_Colour,Layer_Pigment_Colour, Attenuation);
      }

      /* NK rad 
        don't compute highlights for radiosity gather rays, since this causes
        problems with colors being far too bright
        */
      if(true) // && (Radiosity_Trace_Level <= 1) ) // TODO FIXME radiosity
      {
        if (Finish->Phong > 0.0)
        {
          Vector3d ed(Eye.Direction);
          ComputePhongColour(Finish,Light_Source_Ray,ed,Layer_Normal,Colour2,Light_Colour, Layer_Pigment_Colour);
        }
        if (Finish->Specular > 0.0)
        {
          ComputeSpecularColour(Finish,Light_Source_Ray,REye,Layer_Normal,Colour2,Light_Colour, Layer_Pigment_Colour);
        }
      }

      if (Finish->Irid > 0.0)
      {
        ComputeIridColour(Finish,Light_Source_Ray,Layer_Normal,IPoint,Colour2);
      }


    }

  // finish the photons equation
  VScaleEq(*Colour2, (DBL)(1.0)/(M_PI*r*r));

  // add photon contribution to total lighting
  VAddEq(*colour, *Colour2);
}

void Trace::ComputeOneDiffuseLight(LightSource *lightsource, Vector3d& reye, FINISH *finish, Vector3d& ipoint, Ray& eye,
                                      Vector3d& layer_normal, Colour& layer_pigment_colour, Colour& colour, DBL attenuation, ObjectPtr object)
{
	DBL lightsourcedepth, cos_shadow_angle;
	Ray lightsourceray(eye);
	Colour lightcolour;

	// Get a colour and a ray.
	ComputeOneLightRay(lightsource, lightsourcedepth, lightsourceray, eye, ipoint, lightcolour);

	// Don't calculate spotlights when outside of the light's cone.
	if((fabs(lightcolour[pRED])   < EPSILON) &&
	   (fabs(lightcolour[pGREEN]) < EPSILON) &&
	   (fabs(lightcolour[pBLUE])  < EPSILON))
		return;

	// See if light on far side of surface from camera.
	if(!(Test_Flag(object, DOUBLE_ILLUMINATE_FLAG)) // NK 1998 double_illuminate - changed to Test_Flag
           && !lightsource->Use_Full_Area_Lighting) // JN2007: Easiest way of getting rid of sharp shadow lines
	{
		VDot(cos_shadow_angle, *layer_normal, lightsourceray.Direction);
		if(cos_shadow_angle < EPSILON)
			return;
	}

	// If light source was not blocked by any intervening object, then
	// calculate it's contribution to the object's overall illumination.
	if((qualityFlags & Q_SHADOW) && ((lightsource->Projected_Through_Object != NULL) || (lightsource->Light_Type != FILL_LIGHT_SOURCE)))
		TraceShadowRay(lightsource, lightsourcedepth, lightsourceray, eye, ipoint, lightcolour);

	if((fabs(lightcolour[pRED])   > EPSILON) ||
	   (fabs(lightcolour[pGREEN]) > EPSILON) ||
	   (fabs(lightcolour[pBLUE])  > EPSILON))
	{
            if(lightsource->Area_Light && lightsource->Use_Full_Area_Lighting &&
               (qualityFlags & Q_AREA_LIGHT)) // JN2007: Full area lighting
            {
                ComputeFullAreaDiffuseLight(lightsource, reye, finish, ipoint, eye,
                                            layer_normal, layer_pigment_colour, colour, attenuation,
                                            lightsourcedepth, lightsourceray, lightcolour,
                                            Test_Flag(object, DOUBLE_ILLUMINATE_FLAG));
                return;
            }

		if(finish->Diffuse > 0.0)
			ComputeDiffuseColour(finish, lightsourceray, layer_normal, colour, lightcolour, layer_pigment_colour, attenuation);

		// NK rad - don't compute highlights for radiosity gather rays, since this causes
		// problems with colors being far too bright
		if((lightsource->Light_Type != FILL_LIGHT_SOURCE)) // && (Radiosity_Trace_Level <= 1) ) // TODO FIXME radiosity
		{
			if(finish->Phong > 0.0)
			{
				Vector3d ed(eye.Direction);
				ComputePhongColour(finish, lightsourceray, ed, layer_normal, colour, lightcolour, layer_pigment_colour);
			}

			if(finish->Specular > 0.0)
				ComputeSpecularColour(finish, lightsourceray, reye, layer_normal, colour, lightcolour, layer_pigment_colour);
		}

		if(finish->Irid > 0.0)
			ComputeIridColour(finish, lightsourceray, layer_normal, ipoint, colour);
	}
}

// JN2007: Full area lighting:
void Trace::ComputeFullAreaDiffuseLight(LightSource *lightsource, Vector3d& reye, FINISH *finish, Vector3d& ipoint, Ray& eye,
                                        Vector3d& layer_normal, Colour& layer_pigment_colour, Colour& colour, DBL attenuation,
                                        DBL lightsourcedepth, Ray& lightsourceray, Colour& lightcolour, bool isDoubleIlluminated)
{
    Vector3d newAxis1, newAxis2, temp;
    Vector3d axis1Temp, axis2Temp;
    DBL axis1_Length, cos_shadow_angle;
    bool restoreAxis = false;

    if(lightsource->Orient == true)
    {
        // Orient the area light to face the intersection point [ENB 9/97] 
        Assign_Vector(*axis1Temp, lightsource->Axis1);
        Assign_Vector(*axis2Temp, lightsource->Axis2);
        restoreAxis = true;

        // Do Light source to get the correct lightsourceray 
        ExtractedFromAreaLightCodeMegaPOVMess(lightsource, lightsourcedepth,
                                              lightsourceray, eye, ipoint, Vector3d(lightsource->Center));
        VScaleEq(lightsourceray.Direction, -1);

        // Save the lengths of the axises 
        VLength(axis1_Length, lightsource->Axis1);

        // Make axis 1 be perpendicular with the light-ray 
        if(fabs(fabs(lightsourceray.Direction[Z]) - 1.0) < 0.01)
        {
            // too close to vertical for comfort, so use cross product with horizon 
            temp[X] = 0.0;
            temp[Y] = 1.0;
            temp[Z] = 0.0;
        }
        else
        {
            temp[X] = 0.0;
            temp[Y] = 0.0;
            temp[Z] = 1.0;
        }

        VCross(lightsource->Axis1, *temp, lightsourceray.Direction);
        VNormalizeEq(lightsource->Axis1);

        // Make axis 2 be perpendicular with the light-ray and with Axis1.  A simple cross-product will do the trick.
        VCross(lightsource->Axis2, lightsource->Axis1, lightsourceray.Direction);
        VNormalizeEq(lightsource->Axis2);

        // make it square 
        VScaleEq(lightsource->Axis1, axis1_Length);
        VScaleEq(lightsource->Axis2, axis1_Length);

        VScaleEq(lightsourceray.Direction, -1);
    }

    Colour sampleLightcolour = lightcolour;
    Scale_Colour(*sampleLightcolour, *sampleLightcolour, 1.0/(lightsource->Area_Size1*lightsource->Area_Size2));

    for(int v = 0; v < lightsource->Area_Size2; ++v)
    {
        for(int u = 0; u < lightsource->Area_Size1; ++u)
        {
            Vector3d center(lightsource->Center);
            Ray lsr(lightsourceray);
            DBL jitter_u = (DBL)u;
            DBL jitter_v = (DBL)v;

            if(lightsource->Jitter)
            {
                jitter_u += randomNumberGenerator() - 0.5;
                jitter_v += randomNumberGenerator() - 0.5;
            }

            // Create circular are lights [ENB 9/97]
            // First, make jitter_u and jitter_v be numbers from -1 to 1
            // Second, set scaleFactor to the abs max (jitter_u,jitter_v) (for shells)
            // Third, divide scaleFactor by the length of <jitter_u,jitter_v>
            // Fourth, scale jitter_u & jitter_v by scaleFactor
            // Finally scale Axis1 by jitter_u & Axis2 by jitter_v
            if(lightsource->Circular == true)
            {
                jitter_u = jitter_u / (lightsource->Area_Size1 - 1) - 0.5 + 0.001;
                jitter_v = jitter_v / (lightsource->Area_Size2 - 1) - 0.5 + 0.001;
                DBL scaleFactor = ((fabs(jitter_u) > fabs(jitter_v)) ? fabs(jitter_u) : fabs(jitter_v));
                scaleFactor /= sqrt(jitter_u * jitter_u + jitter_v * jitter_v);
                jitter_u *= scaleFactor;
                jitter_v *= scaleFactor;
                VScale(*newAxis1, lightsource->Axis1, jitter_u);
                VScale(*newAxis2, lightsource->Axis2, jitter_v);
            }
            else
            {
                if(lightsource->Area_Size1 > 1)
                {
                    DBL scaleFactor = jitter_u / (DBL)(lightsource->Area_Size1 - 1) - 0.5;
                    VScale(*newAxis1, lightsource->Axis1, scaleFactor);
                }
                else
                    Make_Vector(*newAxis1, 0.0, 0.0, 0.0);

                if(lightsource->Area_Size2 > 1)
                {
                    DBL scaleFactor = jitter_v / (DBL)(lightsource->Area_Size2 - 1) - 0.5;
                    VScale(*newAxis2, lightsource->Axis2, scaleFactor);
                }
                else
                    Make_Vector(*newAxis2, 0.0, 0.0, 0.0);
            }

            VAddEq(*center, *newAxis1);
            VAddEq(*center, *newAxis2);

            // Recalculate the light source ray but not the colour 
            ExtractedFromAreaLightCodeMegaPOVMess(lightsource, lightsourcedepth, lsr, eye, ipoint, center);

            // If not double-illuminated, check if the normal is pointing away:
            if(!isDoubleIlluminated)
            {
                VDot(cos_shadow_angle, *layer_normal, lsr.Direction);
                if(cos_shadow_angle < EPSILON)
                    continue;
            }

            if(finish->Diffuse > 0.0)
                ComputeDiffuseColour(finish, lsr, layer_normal, colour, sampleLightcolour, layer_pigment_colour, attenuation);

            // NK rad - don't compute highlights for radiosity gather rays, since this causes
            // problems with colors being far too bright
            if((lightsource->Light_Type != FILL_LIGHT_SOURCE)) // && (Radiosity_Trace_Level <= 1) ) // TODO FIXME radiosity
            {
                if(finish->Phong > 0.0)
                {
                    Vector3d ed(eye.Direction);
                    ComputePhongColour(finish, lsr, ed, layer_normal, colour, sampleLightcolour, layer_pigment_colour);
                }

                if(finish->Specular > 0.0)
                    ComputeSpecularColour(finish, lsr, reye, layer_normal, colour, sampleLightcolour, layer_pigment_colour);
            }

        }
    }

    if (restoreAxis)
    {
        Assign_Vector(lightsource->Axis1, *axis1Temp);
        Assign_Vector(lightsource->Axis2, *axis2Temp);
    }

    if(finish->Irid > 0.0)
        ComputeIridColour(finish, lightsourceray, layer_normal, ipoint, colour);
}

void Trace::ComputeOneLightRay(LightSource *lightsource, DBL& lightsourcedepth, Ray& lightsourceray, Ray& eyeray, Vector3d& ipoint, Colour& lightcolour)
{
	DBL attenuation,a;
	Vector3d v1;

	// Get the light source colour.
	Assign_Colour(*lightcolour, lightsource->Colour);

	// Get the light ray starting at the intersection point and pointing
	// towards the light source.
	Assign_Vector(lightsourceray.Origin, *ipoint);

	// NK 1998 parallel beams for cylinder source - added if
	if(lightsource->Light_Type == CYLINDER_SOURCE)
	{
		DBL distToPointsAt;
		VECTOR toLightCtr;

		// use old code to approximate Light_Source_Depth
		VSub(lightsourceray.Direction, lightsource->Center, *ipoint);
		VLength(lightsourcedepth, lightsourceray.Direction);

		// use new code to get ray direction - use center - points_at for direction
		VSub(lightsourceray.Direction, lightsource->Center, lightsource->Points_At);

		// get vector pointing to center of light
		VSub(toLightCtr, lightsource->Center, *ipoint);

		// project light_ctr-intersect_point onto light_ctr-point_at
		VLength(distToPointsAt, lightsourceray.Direction);
		VDot(lightsourcedepth, toLightCtr, lightsourceray.Direction);

		// lenght of shadow ray is the length of the projection
		lightsourcedepth /= distToPointsAt;

		VNormalizeEq(lightsourceray.Direction);
	}
	else
	{
		// NK 1998 parallel beams for cylinder source - the stuff in this 'else'
		//   block used to be all that there was... the first half of the if
		//   statement (before the 'else') is new
		VSub(lightsourceray.Direction, lightsource->Center, *ipoint);

		VLength(lightsourcedepth, lightsourceray.Direction);

		VInverseScaleEq(lightsourceray.Direction, lightsourcedepth);
	}

	// Attenuate light source color.
	attenuation = Attenuate_Light(lightsource, lightsourceray, lightsourcedepth);

	// Recalculate for Parallel light sources
	if(lightsource->Parallel) 
	{
		if(lightsource->Area_Light) 
		{
			VSub(*v1, lightsource->Center, lightsource->Points_At);
			VNormalizeEq(*v1);
			VDot(a, *v1, lightsource->Direction);
			lightsourcedepth *= a;
			Assign_Vector(lightsource->Direction, *v1);
		} 
		else 
		{
			VDot(a,lightsource->Direction, lightsourceray.Direction);
			lightsourcedepth *= (-a);
			Assign_Vector(lightsourceray.Direction, lightsource->Direction);
			VScaleEq(lightsourceray.Direction, -1.0);
		}
	}

	// Now scale the color by the attenuation
	VScaleEq(*lightcolour, attenuation);
}

void Trace::TraceShadowRay(LightSource *light, DBL depth, Ray& lightsourceray, Ray& eyeray, Vector3d& point, Colour& colour)
{
	// test and set highest level traced. We do it differently than TraceRay() does,
	// for compatibility with the way max_trace_level is tested and reported in v3.6
	// and earlier.
	if(traceLevel > maxAllowedTraceLevel)
	{
		colour.clear();
		return;
	}

	maxFoundTraceLevel = max(maxFoundTraceLevel, traceLevel);
	traceLevel++;

	DBL newdepth;
	Intersection isect;
	Ray newray(lightsourceray);

	// Store current depth and ray because they will be modified.
	newdepth = depth;

	// NOTE: shadow rays are never photon rays, so flag can be hard-coded to false
	newray.SetFlags(Ray::OtherRay, true, false);

	// Get shadows from current light source.
	if((light->Area_Light) && (qualityFlags & Q_AREA_LIGHT))
		TraceAreaLightShadowRay(light, newdepth, newray, eyeray, point, colour, 0, 0, 0, 0, 0);
	else
		TracePointLightShadowRay(light, newdepth, newray, colour);

	// If there's some distance left for the ray to reach the light source
	// we have to apply atmospheric stuff to this part of the ray.

	if((newdepth > SHADOW_TOLERANCE) && (light->Media_Interaction) && (light->Media_Attenuation))
	{
		isect.Depth = newdepth;
		isect.Object = NULL;
		ComputeShadowMedia(newray, isect, colour, (light->Media_Interaction) && (light->Media_Attenuation));
	}

	traceLevel--;
}

// moved this here (was originally inside TracePointLightShadowRay) because
// for some reason the Intel compiler (version W_CC_PC_8.1.027) will fail
// to link the exe, complaining of an unresolved external.
//
// TODO: try moving it back in at some point in the future.
struct NoShadowFlagRayObjectCondition : public RayObjectCondition 
{
	virtual bool operator()(Ray&, ObjectPtr object) const { return !Test_Flag(object, NO_SHADOW_FLAG); }
};

void Trace::TracePointLightShadowRay(LightSource *lightsource, DBL& lightsourcedepth, Ray& lightsourceray, Colour& lightcolour)
{
	Intersection boundedIntersection;
	IStack localStack(stackPool);
	ObjectPtr cacheObject = NULL;
	bool foundTransparentObjects = false;
	bool foundIntersection;

	// Projected through main tests
	DBL projectedDepth = 0.0;

	if(lightsource->Projected_Through_Object != NULL)
	{
		Intersection tempIntersection;

		if(FindIntersection(lightsource->Projected_Through_Object, tempIntersection, lightsourceray))
		{
			if((tempIntersection.Depth - lightsourcedepth) < 0.0)
				projectedDepth = lightsourcedepth - fabs(tempIntersection.Depth) + SMALL_TOLERANCE;
			else 
			{
				lightcolour.red() =
				lightcolour.green() =
				lightcolour.blue() = 0.0;
				return;
			}
		}
		else 
		{
			lightcolour.red() =
			lightcolour.green() =
			lightcolour.blue() = 0.0;
			return;      
		}

		if(lightsource->Light_Type == FILL_LIGHT_SOURCE)
			return;
	}

	NoShadowFlagRayObjectCondition precond;
	TrueRayObjectCondition postcond;

	if(lightsource->lightGroupLight == false)
	{
		if((traceLevel == 2) && (lightSourceLevel1ShadowCache[lightsource->index] != NULL))
			cacheObject = lightSourceLevel1ShadowCache[lightsource->index];
		else if(lightSourceOtherShadowCache[lightsource->index] != NULL)
			cacheObject = lightSourceOtherShadowCache[lightsource->index];

		// if there was an object in the light source shadow cache, check that first
		if(cacheObject != NULL)
		{
			if(FindIntersection(cacheObject, boundedIntersection, lightsourceray, lightsourcedepth - projectedDepth) == true)
			{
				if(!Test_Flag(boundedIntersection.Object, NO_SHADOW_FLAG))
				{
					ComputeShadowColour(lightsource, boundedIntersection, lightsourceray, lightcolour);

					if((fabs(lightcolour[pRED])   < EPSILON) &&
					   (fabs(lightcolour[pGREEN]) < EPSILON) &&
					   (fabs(lightcolour[pBLUE])  < EPSILON) &&
					   (Test_Flag(boundedIntersection.Object, OPAQUE_FLAG)))
					{
						threadData->Stats[Shadow_Ray_Tests]++;
						threadData->Stats[Shadow_Rays_Succeeded]++;
						threadData->Stats[Shadow_Cache_Hits]++;
						return;
					}
				}
				else
					cacheObject = NULL;
			}
			else
				cacheObject = NULL;
		}
	}

	foundTransparentObjects = false;

	while(true)
	{
		boundedIntersection.Object = NULL;
		boundedIntersection.Depth = lightsourcedepth - projectedDepth;

		threadData->Stats[Shadow_Ray_Tests]++;

		foundIntersection = FindIntersection(boundedIntersection, lightsourceray, precond, postcond);

		if((foundIntersection == true) && (boundedIntersection.Object != cacheObject) &&
		   (boundedIntersection.Depth < lightsourcedepth - SHADOW_TOLERANCE) &&
		   (lightsourcedepth - boundedIntersection.Depth > projectedDepth) &&
		   (boundedIntersection.Depth > SHADOW_TOLERANCE))
		{
			threadData->Stats[Shadow_Rays_Succeeded]++;

			ComputeShadowColour(lightsource, boundedIntersection, lightsourceray, lightcolour);

			if((fabs(lightcolour[pRED])   < EPSILON) &&
			   (fabs(lightcolour[pGREEN]) < EPSILON) &&
			   (fabs(lightcolour[pBLUE])  < EPSILON) &&
			   (Test_Flag(boundedIntersection.Object, OPAQUE_FLAG)))
			{
				if((lightsource->lightGroupLight == false) && (foundTransparentObjects == false))
				{
					if(boundedIntersection.Csg != NULL)
						cacheObject = boundedIntersection.Csg;
					else
						cacheObject = boundedIntersection.Object;

					if(traceLevel == 2)
						lightSourceLevel1ShadowCache[lightsource->index] = cacheObject;
					else
						lightSourceOtherShadowCache[lightsource->index] = cacheObject;
				}
				break;
			}

			foundTransparentObjects = true;

			// Move the ray to the point of intersection, plus some
			lightsourcedepth -= boundedIntersection.Depth;

			Assign_Vector(lightsourceray.Origin, boundedIntersection.IPoint);
		}
		else // No intersections in the direction of the ray.
			break;
	}
}

void Trace::TraceAreaLightShadowRay(LightSource *lightsource, DBL& lightsourcedepth, Ray& lightsourceray, Ray& eyeray,
                                       Vector3d& ipoint, Colour& lightcolour, int u1, int  v1, int  u2, int  v2, int level)
{
	Colour sample_Colour[4];
	Vector3d newAxis1, newAxis2, temp;
	Vector3d axis1Temp, axis2Temp;
	DBL axis1_Length;
	int i, j, u, v, new_u1, new_v1, new_u2, new_v2;
	DBL jitter_u, jitter_v, scaleFactor;
	bool restoreAxis = false ;

	// First call, initialize 
	if((u1 == 0) && (v1 == 0) && (u2 == 0) && (v2 == 0))
	{
		lightGrid.resize(lightsource->Area_Size1 * lightsource->Area_Size2);

		// Flag uncalculated points with a negative value for Red 
                /*
		for(i = 0; i < lightsource->Area_Size1; i++)
		{
			for(j = 0; j < lightsource->Area_Size2; j++)
				lightGrid[i * lightsource->Area_Size2 + j][pRED] = -1.0; // TODO FIXME - Old bug: This will not work with negative color values! [trf]
		}
                */
                for(size_t ind = 0; ind < lightGrid.size(); ++ind)
                    lightGrid[ind][pRED] = -1.0;

		//u1 = 0;
		//v1 = 0;
		u2 = lightsource->Area_Size1 - 1;
		v2 = lightsource->Area_Size2 - 1;

		if(lightsource->Orient == true)
		{
			// Orient the area light to face the intersection point [ENB 9/97] 
			Assign_Vector(*axis1Temp, lightsource->Axis1);
			Assign_Vector(*axis2Temp, lightsource->Axis2);
			restoreAxis = true;

			// Do Light source to get the correct lightsourceray 
			ExtractedFromAreaLightCodeMegaPOVMess(lightsource, lightsourcedepth, lightsourceray, eyeray, ipoint, Vector3d(lightsource->Center));
			VScaleEq(lightsourceray.Direction, -1);

			// Save the lengths of the axises 
			VLength(axis1_Length, lightsource->Axis1);

			// Make axis 1 be perpendicular with the light-ray 
			if(fabs(fabs(lightsourceray.Direction[Z]) - 1.0) < 0.01)
			{
				// too close to vertical for comfort, so use cross product with horizon 
				temp[X] = 0.0;
				temp[Y] = 1.0;
				temp[Z] = 0.0;
			}
			else
			{
				temp[X] = 0.0;
				temp[Y] = 0.0;
				temp[Z] = 1.0;
			}

			VCross(lightsource->Axis1, *temp, lightsourceray.Direction);
			VNormalizeEq(lightsource->Axis1);

			// Make axis 2 be perpendicular with the light-ray and with Axis1.  A simple cross-product will do the trick.
			VCross(lightsource->Axis2, lightsource->Axis1, lightsourceray.Direction);
			VNormalizeEq(lightsource->Axis2);

			// make it square 
			VScaleEq(lightsource->Axis1, axis1_Length);
			VScaleEq(lightsource->Axis2, axis1_Length);

			VScaleEq(lightsourceray.Direction, -1);
		}
	}

	// Sample the four corners of the region 
	for(i = 0; i < 4; i++)
	{
		Vector3d center(lightsource->Center);
		Ray lsr(lightsourceray);

		switch(i)
		{
			case 0: u = u1; v = v1; break;
			case 1: u = u2; v = v1; break;
			case 2: u = u1; v = v2; break;
			case 3: u = u2; v = v2; break;
			default: u = v = 0;  // Should never happen! 
		}

		if(lightGrid[u * lightsource->Area_Size2 + v][pRED] >= 0.0)
			// We've already calculated this point, reuse it 
			Assign_Colour(*sample_Colour[i], *lightGrid[u * lightsource->Area_Size2 + v]);
		else
		{
			jitter_u = (DBL)u;
			jitter_v = (DBL)v;

			if(lightsource->Jitter)
			{
				jitter_u += randomNumberGenerator() - 0.5;
				jitter_v += randomNumberGenerator() - 0.5;
			}

			// Create circular are lights [ENB 9/97]
			// First, make jitter_u and jitter_v be numbers from -1 to 1
			// Second, set scaleFactor to the abs max (jitter_u,jitter_v) (for shells)
			// Third, divide scaleFactor by the length of <jitter_u,jitter_v>
			// Fourth, scale jitter_u & jitter_v by scaleFactor
			// Finally scale Axis1 by jitter_u & Axis2 by jitter_v
			if(lightsource->Circular == true)
			{
				jitter_u = jitter_u / (lightsource->Area_Size1 - 1) - 0.5 + 0.001;
				jitter_v = jitter_v / (lightsource->Area_Size2 - 1) - 0.5 + 0.001;
				scaleFactor = ((fabs(jitter_u) > fabs(jitter_v)) ? fabs(jitter_u) : fabs(jitter_v));
				scaleFactor /= sqrt(jitter_u * jitter_u + jitter_v * jitter_v);
				jitter_u *= scaleFactor;
				jitter_v *= scaleFactor;
				VScale(*newAxis1, lightsource->Axis1, jitter_u);
				VScale(*newAxis2, lightsource->Axis2, jitter_v);
			}
			else
			{
				if(lightsource->Area_Size1 > 1)
				{
					scaleFactor = jitter_u / (DBL)(lightsource->Area_Size1 - 1) - 0.5;
					VScale(*newAxis1, lightsource->Axis1, scaleFactor);
				}
				else
					Make_Vector(*newAxis1, 0.0, 0.0, 0.0);

				if(lightsource->Area_Size2 > 1)
				{
					scaleFactor = jitter_v / (DBL)(lightsource->Area_Size2 - 1) - 0.5;
					VScale(*newAxis2, lightsource->Axis2, scaleFactor);
				}
				else
					Make_Vector(*newAxis2, 0.0, 0.0, 0.0);
			}

			VAddEq(*center, *newAxis1);
			VAddEq(*center, *newAxis2);

			// Recalculate the light source ray but not the colour 
			ExtractedFromAreaLightCodeMegaPOVMess(lightsource, lightsourcedepth, lsr, eyeray, ipoint, center);

			Assign_Colour(*sample_Colour[i], *lightcolour);

			TracePointLightShadowRay(lightsource, lightsourcedepth, lsr, sample_Colour[i]);

			Assign_Colour(*lightGrid[u * lightsource->Area_Size2 + v], *sample_Colour[i]);
		}
	}

	if((u2 - u1 > 1) || (v2 - v1 > 1))
	{
		if((level < lightsource->Adaptive_Level) ||
		   (Colour_Distance(*sample_Colour[0], *sample_Colour[1]) > 0.1) ||
		   (Colour_Distance(*sample_Colour[1], *sample_Colour[3]) > 0.1) ||
		   (Colour_Distance(*sample_Colour[3], *sample_Colour[2]) > 0.1) ||
		   (Colour_Distance(*sample_Colour[2], *sample_Colour[0]) > 0.1))
		{
			Vector3d center(lightsource->Center);

			for (i = 0; i < 4; i++)
			{
				switch (i)
				{
					case 0:
						new_u1 = u1;
						new_v1 = v1;
						new_u2 = (int)floor((u1 + u2)/2.0);
						new_v2 = (int)floor((v1 + v2)/2.0);
						break;
					case 1:
						new_u1 = (int)ceil((u1 + u2)/2.0);
						new_v1 = v1;
						new_u2 = u2;
						new_v2 = (int)floor((v1 + v2)/2.0);
						break;
					case 2:
						new_u1 = u1;
						new_v1 = (int)ceil((v1 + v2)/2.0);
						new_u2 = (int)floor((u1 + u2)/2.0);
						new_v2 = v2;
						break;
					case 3:
						new_u1 = (int)ceil((u1 + u2)/2.0);
						new_v1 = (int)ceil((v1 + v2)/2.0);
						new_u2 = u2;
						new_v2 = v2;
						break;
					default:  // Should never happen! 
						new_u1 = new_u2 = new_v1 = new_v2 = 0;
				}

				// Recalculate the light source ray but not the colour 
				ExtractedFromAreaLightCodeMegaPOVMess(lightsource, lightsourcedepth, lightsourceray, eyeray, ipoint, center);

				Assign_Colour(*sample_Colour[i], *lightcolour);

				TraceAreaLightShadowRay(lightsource, lightsourcedepth, lightsourceray, eyeray,
				                        ipoint, sample_Colour[i], new_u1, new_v1, new_u2, new_v2, level + 1);
			}
		}
	}

	// Add up the light contributions 
	Add_Colour(*lightcolour, *sample_Colour[0], *sample_Colour[1]);
	Add_Colour(*lightcolour, *lightcolour, *sample_Colour[2]);
	Add_Colour(*lightcolour, *lightcolour, *sample_Colour[3]);
	Scale_Colour(*lightcolour, *lightcolour, 0.25);

	if (restoreAxis)
	{
		Assign_Vector(lightsource->Axis1, *axis1Temp);
		Assign_Vector(lightsource->Axis2, *axis2Temp);
	}
}

void Trace::ComputeShadowColour(LightSource *lightsource, Intersection& isect, Ray& lightsourceray, Colour& colour)
{
	WeightedTextureVector wtextures;
	Vector3d ipoint;
	Vector3d raw_Normal;
	Colour fc1, temp_Colour;
	Vector2d uv_Coords;
	DBL normaldirection;

	// Here's the issue:
	// Imagine "LightA" shoots photons at "GlassSphereB", which refracts light and
	// hits "PlaneC".
	// When computing Diffuse/Phong/etc lighting for PlaneC, if there were no
	// photons, POV would compute a filtered shadow ray from PlaneC through
	// GlassSphereB to LightA.  If photons are used for the combination of objects,
	// this filtered shadow ray should be completely black.  The filtered shadow
	// ray should be forced to black UNLESS any of the following conditions are
	// true (which would indicate that photons were not shot from LightA through
	// GlassSphereB to PlaneC):
	// 1) PlaneC has photon collection set to "off"
	// 2) GlassSphereB is not a photon target
	// 3) GlassSphereB has photon refraction set to "off"
	// 4) LightA has photon refraction set to "off"
	// 5) Neither GlassSphereB nor LightA has photon refraction set to "on"
	  if((sceneData->photonSettings.photonsEnabled == true) &&
	   (sceneData->surfacePhotonMap.numPhotons > 0) &&
	   (!threadData->litObjectIgnoresPhotons) &&
	   (Test_Flag(isect.Object,PH_TARGET_FLAG)) &&
	   (!Test_Flag(isect.Object,PH_RFR_OFF_FLAG)) &&
	   (!Test_Flag(lightsource,PH_RFR_OFF_FLAG)) &&
	   ((Test_Flag(isect.Object,PH_RFR_ON_FLAG) || Test_Flag(lightsource,PH_RFR_ON_FLAG)))
	   )
	{
		colour.red()=colour.green()=colour.blue() = 0.0;
		return;
	}

	Assign_Vector(*ipoint, isect.IPoint);

	if(!(qualityFlags & Q_SHADOW))
		return;

	// If the object is opaque there's no need to go any further. [DB 8/94]
	if(Test_Flag(isect.Object, OPAQUE_FLAG))
	{
		Make_Colour(*colour, 0.0, 0.0, 0.0);
		return;
	}

	// Get the normal to the surface
	isect.Object->Normal(*raw_Normal, &isect, threadData);

	// I added this to flip the normal if the object is inverted (for CSG).
	// However, I subsequently commented it out for speed reasons - it doesn't
	// make a difference (no pun intended). The preexisting flip code below
	// produces a similar (though more extensive) result. [NK]
	//
	// Actually, we should keep this code to guarantee that Normal_Direction
	// is set properly. [NK]
	if(Test_Flag(isect.Object, INVERTED_FLAG))
	{
		raw_Normal[X] = -raw_Normal[X];
		raw_Normal[Y] = -raw_Normal[Y];
		raw_Normal[Z] = -raw_Normal[Z];
	}

	// If the surface normal points away, flip its direction.
	VDot(normaldirection, *raw_Normal, lightsourceray.Direction);
	if(normaldirection > 0.0)
		VScaleEq(*raw_Normal, -1.0);

	Assign_Vector(isect.INormal, *raw_Normal);
	// and save to intersection -hdf-
	Assign_Vector(isect.PNormal, *raw_Normal);

	if(Test_Flag(isect.Object, UV_FLAG))
	{
		// get the UV vect of the intersection
		isect.Object->UVCoord(*uv_Coords, &isect, threadData);
		// save the normal and UV coords into Intersection
		Assign_UV_Vect(isect.Iuv, *uv_Coords);
	}

	// now switch to UV mapping if we need to
	if(Test_Flag(isect.Object, UV_FLAG))
	{
		ipoint[X] = uv_Coords[U];
		ipoint[Y] = uv_Coords[V];
		ipoint[Z] = 0;
	}

	bool isMultiTextured = Test_Flag(isect.Object, MULTITEXTURE_FLAG);

	// Deal with cutaway textures.
	//
	// isect.Csg is the root-level object in this CSG tree if the intersection object
	// is the child of a multi-tex (cutaway_texture) CSG object. If that member points
	// to a valid CSG object, and the intersected object has a NULL texture, we use
	// the texture list provided by the CSG. Otherwise we just use the provided texture
	// of the intersected object (i.e. we skip the call to Determine_Textures()).
	if((isect.Object->Texture != NULL) && (isect.Csg != NULL) && (Test_Flag (isect.Csg, IS_CSG_OBJECT) != false))
		isMultiTextured = false;

	// get textures and weights
	if(isMultiTextured == true)
	{
		isect.Object->Determine_Textures(&isect, normaldirection > 0.0, wtextures, threadData);
	}
	else if(isect.Object->Texture != NULL)
	{
		if((normaldirection > 0.0) && (isect.Object->Interior_Texture != NULL))
			wtextures.push_back(WeightedTexture(1.0, isect.Object->Interior_Texture)); /* Chris Huff: Interior Texture patch */
		else
			wtextures.push_back(WeightedTexture(1.0, isect.Object->Texture));
	}
	else
	{
		// don't need to do anything as the texture list will be empty.
		// TODO: could we perform these tests earlier ? [cjc]
		return;
	}

	Make_ColourA(*temp_Colour, 0.0, 0.0, 0.0, 0.0, 0.0);

	for(WeightedTextureVector::iterator i(wtextures.begin()); i != wtextures.end(); i++)
	{
		TextureVector warps(texturePool);

		// If contribution of this texture is neglectable skip ahead.
		if((i->weight < adcBailout) || (i->texture == NULL))
			continue;

		ComputeOneTextureColour(fc1, i->texture, *warps, ipoint, raw_Normal, lightsourceray, 0.0, isect, true, false);

		temp_Colour[pRED]     += i->weight * fc1[pRED];
		temp_Colour[pGREEN]   += i->weight * fc1[pGREEN];
		temp_Colour[pBLUE]    += i->weight * fc1[pBLUE];
		temp_Colour[pFILTER]  += i->weight * fc1[pFILTER];
		temp_Colour[pTRANSM]  += i->weight * fc1[pTRANSM];
	}

	if(fabs(temp_Colour[pFILTER]) + fabs(temp_Colour[pTRANSM]) < adcBailout)
		Make_Colour(*colour, 0.0, 0.0, 0.0);
	else
	{
		colour[pRED]   *= temp_Colour[pFILTER] * temp_Colour[pRED]  + temp_Colour[pTRANSM];
		colour[pGREEN] *= temp_Colour[pFILTER] * temp_Colour[pGREEN]+ temp_Colour[pTRANSM];
		colour[pBLUE]  *= temp_Colour[pFILTER] * temp_Colour[pBLUE] + temp_Colour[pTRANSM];
	}

	// Get atmospheric attenuation.
    ComputeShadowMedia(lightsourceray, isect, colour, (lightsource->Media_Interaction) && (lightsource->Media_Attenuation));
}

void Trace::ComputeDiffuseColour(FINISH *finish, Ray& lightsourceray, Vector3d& layer_normal, Colour& colour, Colour& light_colour, Colour& layer_pigment_colour, DBL attenuation)
{
	DBL cos_angle_of_incidence, intensity;

	VDot(cos_angle_of_incidence, *layer_normal, lightsourceray.Direction);

	// Brilliance is likely to be 1.0 (default value)
	if(finish->Brilliance != 1.0)
		intensity = pow(fabs(cos_angle_of_incidence), (DBL) finish->Brilliance);
	else
		intensity = fabs(cos_angle_of_incidence);

	intensity *= finish->Diffuse * attenuation;

// TODO FIXME thread issue	if(finish->Crand > 0.0)
// TODO FIXME thread issue		intensity -= FRAND() * finish->Crand;

	colour[pRED]   += intensity * layer_pigment_colour[pRED]   * light_colour[pRED];
	colour[pGREEN] += intensity * layer_pigment_colour[pGREEN] * light_colour[pGREEN];
	colour[pBLUE]  += intensity * layer_pigment_colour[pBLUE]  * light_colour[pBLUE];
}

void Trace::ComputeIridColour(FINISH *finish, Ray& lightsourceray, Vector3d& layer_normal, Vector3d& ipoint, Colour& colour)
{
	DBL rwl, gwl, bwl;
	DBL cos_angle_of_incidence, interference;
	DBL film_thickness;
	DBL noise, intensity;
	TURB turb;

	film_thickness = finish->Irid_Film_Thickness;

	if(finish->Irid_Turb != 0)
	{
		// Uses hardcoded octaves, lambda, omega
		turb.Omega=0.5;
		turb.Lambda=2.0;
		turb.Octaves=5;

		noise = Turbulence(*ipoint, &turb, sceneData->noiseGenerator) * finish->Irid_Turb;

		film_thickness *= noise;
	}

	// Approximate dominant wavelengths of primary hues.
	// Source: 3D Computer Graphics by John Vince (Addison Wesely)
	// These are initialized in parse.c (Parse_Frame)
	// and are user-adjustable with the irid_wavelength keyword.
	// Red = 700 nm  Grn = 520 nm Blu = 480 nm
	// Divided by 100 gives: rwl = 0.70;  gwl = 0.52;  bwl = 0.48;
	//
	// However... I originally "guessed" at the values and came up with
	// the following, which I'm using as the defaults, since it seems
	// to work better:  rwl = 0.25;  gwl = 0.18;  bwl = 0.14;

	// Could avoid these assignments if we want to
	rwl = sceneData->iridWavelengths[pRED];
	gwl = sceneData->iridWavelengths[pGREEN];
	bwl = sceneData->iridWavelengths[pBLUE];

	// NOTE: Shouldn't we compute Cos_Angle_Of_Incidence just once?
	VDot(cos_angle_of_incidence, *layer_normal, lightsourceray.Direction);

	// Calculate phase offset.
	interference = 4.0 * M_PI * film_thickness * cos_angle_of_incidence;

	intensity = cos_angle_of_incidence * finish->Irid;

	// Modify color by phase offset for each wavelength.
	colour[pRED]   += finish->Irid * (intensity * (1.0 - 0.5 * cos(interference / rwl)));
	colour[pGREEN] += finish->Irid * (intensity * (1.0 - 0.5 * cos(interference / gwl)));
	colour[pBLUE]  += finish->Irid * (intensity * (1.0 - 0.5 * cos(interference / bwl)));
}

void Trace::ComputePhongColour(FINISH *finish, Ray& lightsourceray, Vector3d& eye, Vector3d& layer_normal, Colour& colour, Colour& light_colour, Colour& layer_pigment_colour)
{
	DBL cos_angle_of_incidence, intensity;
	Vector3d reflect_direction;
	DBL ndotl, x, f;
	Colour cs;

	VDot(cos_angle_of_incidence, *eye, *layer_normal);

	cos_angle_of_incidence *= -2.0;

	VLinComb2(*reflect_direction, 1.0, *eye, cos_angle_of_incidence, *layer_normal);

	VDot(cos_angle_of_incidence, *reflect_direction, lightsourceray.Direction);

	if(cos_angle_of_incidence > 0.0)
	{
		if((finish->Phong_Size < 60) || (cos_angle_of_incidence > 0.0008)) // rgs
			intensity = finish->Phong * pow(cos_angle_of_incidence, (DBL)finish->Phong_Size);
		else
			intensity = 0.0; // ad

		if(finish->Metallic > 0.0)
		{
			// Calculate the reflected color by interpolating between
			// the light source color and the surface color according
			// to the (empirical) Fresnel reflectivity function. [DB 9/94]

			VDot(ndotl, *layer_normal, lightsourceray.Direction);

			x = fabs(acos(ndotl)) / M_PI_2;

			f = 0.014567225 / Sqr(x - 1.12) - 0.011612903;

			f = min(1.0, max(0.0, f));
			cs[pRED]   = light_colour[pRED]   * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pRED]   - 1.0));
			cs[pGREEN] = light_colour[pGREEN] * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pGREEN] - 1.0));
			cs[pBLUE]  = light_colour[pBLUE]  * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pBLUE]  - 1.0));

			CRGBAddScaledEq(*colour, intensity, *cs);
		}
		else
		{
			colour[pRED]   += intensity * light_colour[pRED];
			colour[pGREEN] += intensity * light_colour[pGREEN];
			colour[pBLUE]  += intensity * light_colour[pBLUE];
		}
	}
}

void Trace::ComputeSpecularColour(FINISH *finish, Ray& lightsourceray, Vector3d& eye, Vector3d& layer_normal, Colour& colour, Colour& light_colour, Colour& layer_pigment_colour)
{
	DBL cos_angle_of_incidence, intensity, halfway_length;
	Vector3d halfway;
	DBL ndotl, x, f;
	Colour cs;

	VHalf(*halfway, *eye, lightsourceray.Direction);

	VLength(halfway_length, *halfway);

	if(halfway_length > 0.0)
	{
		VDot(cos_angle_of_incidence, *halfway, *layer_normal);

		cos_angle_of_incidence /= halfway_length;

		if(cos_angle_of_incidence > 0.0)
		{
			intensity = finish->Specular * pow(cos_angle_of_incidence, (DBL)finish->Roughness);

			if(finish->Metallic > 0.0)
			{
				// Calculate the reflected color by interpolating between
				// the light source color and the surface color according
				// to the (empirical) Fresnel reflectivity function. [DB 9/94]

				VDot(ndotl, *layer_normal, lightsourceray.Direction);

				x = fabs(acos(ndotl)) / M_PI_2;

				f = 0.014567225 / Sqr(x - 1.12) - 0.011612903;

				f = min(1.0, max(0.0, f));
				cs[pRED]   = light_colour[pRED]   * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pRED]   - 1.0));
				cs[pGREEN] = light_colour[pGREEN] * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pGREEN] - 1.0));
				cs[pBLUE]  = light_colour[pBLUE]  * (1.0 + finish->Metallic * (1.0 - f) * (layer_pigment_colour[pBLUE]  - 1.0));

				CRGBAddScaledEq(*colour, intensity, *cs);
			}
			else
			{
				colour[pRED]   += intensity * light_colour[pRED];
				colour[pGREEN] += intensity * light_colour[pGREEN];
				colour[pBLUE]  += intensity * light_colour[pBLUE];
			}
		}
	}
}

void Trace::ComputeReflectivity(DBL& weight, Colour& reflectivity, const Colour& reflection_max, const Colour& reflection_min,
                                int reflection_type, DBL reflection_falloff, DBL cos_angle, Ray& ray, Interior *interior)
{
	DBL temp_Weight_Min, temp_Weight_Max;
	DBL reflection_Frac;
	DBL g, f;
	DBL ior;

	if(reflection_type == 1)
	{
		// Get ratio of iors depending on the interiors the ray is traversing.
		if(ray.GetInteriors().empty())
			// The ray is entering from the atmosphere.
			ior = sceneData->atmosphereIOR / interior->IOR;
		else
		{
			// The ray is currently inside an object.
			if(ray.IsInterior(interior) == true)
			{
				if(ray.GetInteriors().size() == 1)
					// The ray is leaving into the atmosphere.
					ior = interior->IOR / sceneData->atmosphereIOR;
				else
					// The ray is leaving into another object.
					ior = interior->IOR / ray.GetInteriors().back()->IOR;
			}
			else
				// The ray is entering a new object.
				ior = ray.GetInteriors().back()->IOR / interior->IOR;
		}

		ior = 1.0 / ior;
	}

	switch(reflection_type)
	{
		case 0: // Standard reflection
		{
			temp_Weight_Max = max3(reflection_max[pRED], reflection_max[pGREEN], reflection_max[pBLUE]);
			temp_Weight_Min = max3(reflection_min[pRED], reflection_min[pGREEN], reflection_min[pBLUE]);
			weight = weight * max(temp_Weight_Max, temp_Weight_Min);

			if(fabs(reflection_falloff - 1.0) > EPSILON)
				reflection_Frac = pow(1.0 - cos_angle, reflection_falloff);
			else
				reflection_Frac = 1.0 - cos_angle;

			if(fabs(reflection_Frac) < EPSILON)
				Assign_RGB(*reflectivity, *reflection_min);
			else if (fabs(reflection_Frac - 1.0)<EPSILON)
				Assign_RGB(*reflectivity, *reflection_max);
			else
				CRGBLinComb2(*reflectivity, reflection_Frac, *reflection_max, (1 - reflection_Frac), *reflection_min);
			break;
		}
		case 1:  // Fresnel
		{
			// Christoph's tweak to work around possible negative argument in sqrt
			DBL sqx = Sqr(ior) + Sqr(cos_angle) - 1.0;

			if(sqx > 0.0)
			{
				g = sqrt(sqx);
				f = 0.5 * (Sqr(g - cos_angle) / Sqr(g + cos_angle));
				f = f * (1.0 + Sqr(cos_angle * (g + cos_angle) - 1.0) / Sqr(cos_angle * (g - cos_angle) + 1.0));

				f = min(1.0, max(0.0, f));
				CRGBLinComb2(*reflectivity, f, *reflection_max, (1.0 - f), *reflection_min);
			}
			else
				Assign_RGB(*reflectivity, *reflection_max);

			weight = weight * max3(reflectivity.red(), reflectivity.green(), reflectivity.blue());
			break;
		}
		default:
			throw POV_EXCEPTION_STRING("Illegal reflection_type."); // TODO FIXME - wrong place to report this [trf]
	}
}

void Trace::ExtractedFromAreaLightCodeMegaPOVMess(LightSource *lightsource, DBL& lightsourcedepth, Ray& lightsourceray, Ray& eyeray, Vector3d& ipoint, const Vector3d& center)
{
	DBL a;
	Vector3d v1;

	// Get the light ray starting at the intersection point and pointing towards the light source.
	Assign_Vector(lightsourceray.Origin, *ipoint);
	// NK 1998 parallel beams for cylinder source - added 'if' 
	if(lightsource->Light_Type == CYLINDER_SOURCE)
	{
		DBL distToPointsAt;
		Vector3d toLightCtr;

		// use new code to get ray direction - use center - points_at for direction 
		VSub(lightsourceray.Direction, *center, lightsource->Points_At);

		// get vector pointing to center of light 
		VSub(*toLightCtr, *center, *ipoint);

		// project light_ctr-intersect_point onto light_ctr-point_at
		VLength(distToPointsAt, lightsourceray.Direction);
		VDot(lightsourcedepth, *toLightCtr, lightsourceray.Direction);

		// lenght of shadow ray is the length of the projection 
		lightsourcedepth /= distToPointsAt;
		VNormalizeEq(lightsourceray.Direction);
	}
	else
	{
		// NK 1998 parallel beams for cylinder source - the stuff in this 'else'
		// block used to be all that there was... the first half of the if
		// statement (before the 'else') is new
		VSub(lightsourceray.Direction, *center, *ipoint);
		VLength(lightsourcedepth, lightsourceray.Direction);
		VInverseScaleEq(lightsourceray.Direction, lightsourcedepth);
	}

	// Attenuate light source color. 
	// Attenuation = Attenuate_Light(lightsource, lightsourceray, *Light_Source_Depth);
	// Recalculate for Parallel light sources 
	if(lightsource->Parallel)
	{
		if(lightsource->Area_Light)
		{
			VSub(*v1, *center, lightsource->Points_At);
			VNormalizeEq(*v1);
			VDot(a, *v1, lightsourceray.Direction);
			lightsourcedepth *= a;
			Assign_Vector(lightsourceray.Direction, *v1);
		}
		else
		{
			VDot(a, lightsource->Direction, lightsourceray.Direction);
			lightsourcedepth *= (-a);
			Assign_Vector(lightsourceray.Direction, lightsource->Direction);
			VScaleEq(lightsourceray.Direction, -1.0);
		}
	}
}

void Trace::ComputeSky(Ray& ray, Colour& colour)
{
	int i;
	DBL att, trans;
	Colour col, col_Temp, filterc;
	Vector3d p;

	colour = sceneData->backgroundColour;

	if((sceneData->skysphere == NULL) || (sceneData->skysphere->Pigments == NULL))
		return;

	Make_Colour(*col, 0.0, 0.0, 0.0);

	Make_ColourA(*filterc, 1.0, 1.0, 1.0, 1.0, 1.0);

	trans = 1.0;

	// Transform point on unit sphere.
	if(sceneData->skysphere->Trans != NULL)
		MInvTransPoint(*p, ray.Direction, sceneData->skysphere->Trans);
	else
		Assign_Vector(*p, ray.Direction);

	for(i = sceneData->skysphere->Count - 1; i >= 0; i--)
	{
		// Compute sky colour from colour map.

		// NK 1998 - added NULL as final parameter
		Compute_Pigment(*col_Temp, sceneData->skysphere->Pigments[i], *p, NULL, threadData);

		att = trans * (1.0 - col_Temp[pFILTER] - col_Temp[pTRANSM]);

		CRGBAddScaledEq(*col, att, *col_Temp);

		filterc[pRED]    *= col_Temp[pRED];
		filterc[pGREEN]  *= col_Temp[pGREEN];
		filterc[pBLUE]   *= col_Temp[pBLUE];
		filterc[pFILTER] *= col_Temp[pFILTER];
		filterc[pTRANSM] *= col_Temp[pTRANSM];

		trans = fabs(filterc[pFILTER]) + fabs(filterc[pTRANSM]);
	}

	colour[pRED]    = col[pRED]    + colour[pRED]   * (filterc[pRED]   * filterc[pFILTER] + filterc[pTRANSM]);
	colour[pGREEN]  = col[pGREEN]  + colour[pGREEN] * (filterc[pGREEN] * filterc[pFILTER] + filterc[pTRANSM]);
	colour[pBLUE]   = col[pBLUE]   + colour[pBLUE]  * (filterc[pBLUE]  * filterc[pFILTER] + filterc[pTRANSM]);
	colour[pFILTER] = colour[pFILTER] * filterc[pFILTER];
	colour[pTRANSM] = colour[pTRANSM] * filterc[pTRANSM];
}

void Trace::ComputeFog(Ray& ray, Intersection& isect, Colour& colour)
{
	DBL att, att_inv, width;
	Colour col_fog;
	Colour sum_att; // total attenuation. 
	Colour sum_col; // total color.       

	// Why are we here. 
	if(sceneData->fog == NULL)
		return;

	// Init total attenuation and total color. 
	Make_ColourA(*sum_att, 1.0, 1.0, 1.0, 1.0, 1.0);
	Make_ColourA(*sum_col, 0.0, 0.0, 0.0, 0.0, 0.0);

	// Loop over all fogs. 
	for(FOG *fog = sceneData->fog; fog != NULL; fog = fog->Next)
	{
		// Don't care about fogs with zero distance. 
		if(fabs(fog->Distance) > EPSILON)
		{
			width = isect.Depth;

			switch(fog->Type)
			{
				case GROUND_MIST:
					att = ComputeGroundFogColour(ray, 0.0, width, fog, col_fog);
					break;
				default:
					att = ComputerConstantFogColour(ray, 0.0, width, fog, col_fog);
					break;
			}

			// Check for minimum transmittance. 
			if(att < col_fog[pTRANSM])
				att = col_fog[pTRANSM];

			// Get attenuation sum due to filtered/unfiltered translucency. 
			sum_att[pRED]    *= att * ((1.0 - col_fog[pFILTER]) + col_fog[pFILTER] * col_fog[pRED]);
			sum_att[pGREEN]  *= att * ((1.0 - col_fog[pFILTER]) + col_fog[pFILTER] * col_fog[pGREEN]);
			sum_att[pBLUE]   *= att * ((1.0 - col_fog[pFILTER]) + col_fog[pFILTER] * col_fog[pBLUE]);
			sum_att[pFILTER] *= att * col_fog[pFILTER];
			sum_att[pTRANSM] *= att * col_fog[pTRANSM];

			if(!ray.IsShadowTestRay())
			{
				att_inv = 1.0 - att;
				CRGBAddScaledEq(*sum_col, att_inv, *col_fog);
			}
		}
	}

	// Add light coming from background. 
	colour[pRED]   = sum_col[pRED]   + sum_att[pRED]   * colour[pRED];
	colour[pGREEN] = sum_col[pGREEN] + sum_att[pGREEN] * colour[pGREEN];
	colour[pBLUE]  = sum_col[pBLUE]  + sum_att[pBLUE]  * colour[pBLUE];
	colour[pTRANSM] *= GREY_SCALE(sum_att);
}

DBL Trace::ComputerConstantFogColour(Ray &ray, DBL depth, DBL width, FOG *fog, Colour& colour)
{
	VECTOR p;
	DBL k;

	if(fog->Turb != NULL)
	{
		depth += width / 2.0;

		VEvaluateRay(p, ray.Origin, depth, ray.Direction);
		VEvaluateEq(p, fog->Turb->Turbulence);

		// The further away the less influence turbulence has. 
		k = exp(-width / fog->Distance);

		width *= 1.0 - k * min(1.0, Turbulence(p, fog->Turb, sceneData->noiseGenerator) * fog->Turb_Depth);
	}

	Assign_Colour(*colour, fog->Colour);

	return (exp(-width / fog->Distance));
}

/*****************************************************************************
*   Here is an ascii graph of the ground fog density, it has a maximum
*   density of 1.0 at Y <= 0, and approaches 0.0 as Y goes up:
*
*   ***********************************
*        |           |            |    ****
*        |           |            |        ***
*        |           |            |           ***
*        |           |            |            | ****
*        |           |            |            |     *****
*        |           |            |            |          *******
*   -----+-----------+------------+------------+-----------+-----
*       Y=-2        Y=-1         Y=0          Y=1         Y=2
*
*   ground fog density is 1 / (Y*Y+1) for Y >= 0 and equals 1.0 for Y <= 0.
*   (It behaves like regular fog for Y <= 0.)
*
*   The integral of the density is atan(Y) (for Y >= 0).
******************************************************************************/

DBL Trace::ComputeGroundFogColour(Ray& ray, DBL depth, DBL width, FOG *fog, Colour& colour)
{
	DBL fog_density, delta;
	DBL start, end;
	DBL y1, y2, k;
	Vector3d p, p1, p2;

	// Get start point. 
	VEvaluateRay(*p1, ray.Origin, depth, ray.Direction);

	// Get end point. 
	VLinComb2(*p2, 1.0, *p1, width, ray.Direction);

	// Could preform transfomation here to translate Start and End
	// points into ground fog space.
	VDot(y1, *p1, fog->Up);
	VDot(y2, *p2, fog->Up);

	start = (y1 - fog->Offset) / fog->Alt;
	end   = (y2 - fog->Offset) / fog->Alt;

	// Get integral along y-axis from start to end. 
	if(start <= 0.0)
	{
		if(end <= 0.0)
			fog_density = 1.0;
		else
			fog_density = (atan(end) - start) / (end - start);
	}
	else
	{
		if(end <= 0.0)
			fog_density = (atan(start) - end) / (start - end);
		else
		{
			delta = start - end;

			if(fabs(delta) > EPSILON)
				fog_density = (atan(start) - atan(end)) / delta;
			else
				fog_density = 1.0 / (Sqr(start) + 1.0);
		}
	}

	// Apply turbulence. 
	if (fog->Turb != NULL)
	{
		VHalf(*p, *p1, *p2);
		VEvaluateEq(*p, fog->Turb->Turbulence);

		// The further away the less influence turbulence has. 
		k = exp(-width / fog->Distance);
		width *= 1.0 - k * min(1.0, Turbulence(*p, fog->Turb, sceneData->noiseGenerator) * fog->Turb_Depth);
	}

	Assign_Colour(*colour, fog->Colour);

	return (exp(-width * fog_density / fog->Distance));
}

/**
 * NOTE: this computes two things
 *   1) media and fog attenuation of the shadow ray (optional)
 *   2) entry/exit of interiors
 *
 *  In other words, you can't skip this whole thing, because the entry/exit is important.
 */
void Trace::ComputeShadowMedia(Ray& light_source_ray, Intersection& isect, Colour& resultcolour, bool media_attenuation_and_interaction)
{
	if((resultcolour.red() < EPSILON) && (resultcolour.green() < EPSILON) && (resultcolour.blue() < EPSILON))
		return;

	// Calculate participating media effects.
	if(media_attenuation_and_interaction && (qualityFlags & Q_VOLUME) && ((light_source_ray.IsHollowRay() == true) || (isect.Object != NULL && isect.Object->interior != NULL)))
	{
		media.ComputeMedia(sceneData->atmosphere, light_source_ray, isect, resultcolour);

    	if((sceneData->fog != NULL) && (light_source_ray.IsHollowRay() == true) && (light_source_ray.IsPhotonRay() == false))
        	ComputeFog(light_source_ray, isect, resultcolour);
	}

	// TODO FIXME - this was removed in some 3.7 beta, why? [trf] if((sceneData->fog != NULL) && (light_source_ray.IsHollowRay() == true) && ((qualityFlags & Q_VOLUME) && !light_source_ray.IsPhotonRay()))
	// TODO FIXME - this was removed in some 3.7 beta, why? [trf] 	ComputeFog(light_source_ray, isect, resultcolour);
    // NK: this code is still here, just a couple lines up... inside the "if(quality and hollow)"

	// If ray is entering from the atmosphere or the ray is currently *not* inside an object add it,
	// but it it is currently inside an object, the ray is leaving the current object and is removed
	if((isect.Object != NULL) && ((light_source_ray.GetInteriors().empty()) || (light_source_ray.RemoveInterior(isect.Object->interior) == false)))
		light_source_ray.AppendInterior(isect.Object->interior);
}



void Trace::ComputeRainbow(Ray& ray, Intersection& isect, Colour& colour)
{
	int n;
	DBL dot, k, ki, index, x, y, l, angle, fade, f;
	Vector3d Temp;
	Colour Cr, Ct;

	// Why are we here. 
	if(sceneData->rainbow == NULL)
		return;

	Make_ColourA(*Ct, 0.0, 0.0, 0.0, 1.0, 1.0);

	n = 0;

	for(RAINBOW *Rainbow = sceneData->rainbow; Rainbow != NULL; Rainbow = Rainbow->Next)
	{
		if((Rainbow->Pigment != NULL) && (Rainbow->Distance != 0.0) && (Rainbow->Width != 0.0))
		{
			// Get angle between ray direction and rainbow's up vector. 
			VDot(x, ray.Direction, Rainbow->Right_Vector);
			VDot(y, ray.Direction, Rainbow->Up_Vector);

			l = Sqr(x) + Sqr(y);

			if(l > 0.0)
			{
				l = sqrt(l);
				y /= l;
			}

			angle = fabs(acos(y));

			if(angle <= Rainbow->Arc_Angle)
			{
				// Get dot product between ray direction and antisolar vector. 
				VDot(dot, ray.Direction, Rainbow->Antisolar_Vector);

				if(dot >= 0.0)
				{
					// Get index ([0;1]) into rainbow's colour map. 
					index = (acos(dot) - Rainbow->Angle) / Rainbow->Width;

					// Jitter index. 
					if(Rainbow->Jitter > 0.0)
						index += (2.0 * randomNumberGenerator() - 1.0) * Rainbow->Jitter;

					if((index >= 0.0) && (index <= 1.0 - EPSILON))
					{
						// Get colour from rainbow's colour map. 
						Make_Vector(*Temp, index, 0.0, 0.0);
						Compute_Pigment(*Cr, Rainbow->Pigment, *Temp, &isect, threadData);

						// Get fading value for falloff. 
						if((Rainbow->Falloff_Width > 0.0) && (angle > Rainbow->Falloff_Angle))
						{
							fade = (angle - Rainbow->Falloff_Angle) / Rainbow->Falloff_Width;
							fade = (3.0 - 2.0 * fade) * fade * fade;
						}
						else
							fade = 0.0;

						// Get attenuation factor due to distance. 
						k = exp(-isect.Depth / Rainbow->Distance);

						// Colour's transm value is used as minimum attenuation value. 
						k = max(k, fade * (1.0 - Cr[pTRANSM]) + Cr[pTRANSM]);

						// Now interpolate the colours. 
						ki = 1.0 - k;

						// Attenuate filter value. 
						f = Cr[pFILTER] * ki;

						Ct[pRED]    += k * colour[pRED]   * ((1.0 - f) + f * Cr[pRED])   + ki * Cr[pRED];
						Ct[pGREEN]  += k * colour[pGREEN] * ((1.0 - f) + f * Cr[pGREEN]) + ki * Cr[pGREEN];
						Ct[pBLUE]   += k * colour[pBLUE]  * ((1.0 - f) + f * Cr[pBLUE])  + ki * Cr[pBLUE];
						Ct[pFILTER] *= k * Cr[pFILTER];
						Ct[pTRANSM] *= k * Cr[pTRANSM];

						n++;
					}
				}
			}
		}
	}

	if(n > 0)
	{
		COLC tmp = 1.0 / n;

		colour[pRED] = Ct[pRED] * tmp;
		colour[pGREEN] = Ct[pGREEN] * tmp;
		colour[pBLUE] = Ct[pBLUE] * tmp;

		colour[pFILTER] *= Ct[pFILTER];
		colour[pTRANSM] *= Ct[pTRANSM];
	}
}

bool Trace::TestShadow(LightSource *light, DBL& depth, Ray& light_source_ray, Ray& eyeray, Vector3d& p, Colour& colour)
{
	ComputeOneLightRay(light, depth, light_source_ray, eyeray, p, colour);

	// There's no need to test for shadows if no light
	// is coming from the light source.
	//
	// Test for PURE zero, because we only want to skip this if we're out
	// of the range of a spot light or cylinder light.  Very dim lights
	// should not be ignored.

	if((fabs(colour[X]) < EPSILON) && (fabs(colour[Y]) < EPSILON) && (fabs(colour[Z]) < EPSILON))
		return true;
	else
	{
		// Test for shadows.
		if((qualityFlags & Q_SHADOW) && ((light->Projected_Through_Object != NULL) || (light->Light_Type != FILL_LIGHT_SOURCE)))
		{
			TraceShadowRay(light, depth, light_source_ray, eyeray, p, colour);

			if((fabs(colour[X]) < EPSILON) && (fabs(colour[Y]) < EPSILON) && (fabs(colour[Z]) < EPSILON))
				return true;
		}
	}

	return false;
}

bool Trace::IsObjectInCSG(ObjectPtr object, ObjectPtr parent)
{
	bool found = false;

	if(object == parent) // TODO FIXME - Why does this make the object a CSG object??? [trf]
		return true;

	if(parent->Type & IS_COMPOUND_OBJECT)
	{
		for(vector<ObjectPtr>::iterator Sib = ((CSG *)parent)->children.begin(); Sib != ((CSG *)parent)->children.end(); Sib++)
		{
			if(IsObjectInCSG(object, *Sib))
				found = true;
		}
	}

	return found;
}

}
