/*******************************************************************************
 * radiosity.cpp
 *
 * This file contains radiosity computation task code.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/lighting/radiosity.cpp $
 * $Revision: #29 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

/************************************************************************
*  Radiosity calculation routies.
*
*  (This does not work the way that most radiosity programs do, but it accomplishes
*  the diffuse interreflection integral the hard way and produces similar results. It
*  is called radiosity here to avoid confusion with ambient and diffuse, which
*  already have well established meanings within POV).
*  Inspired by the paper "A Ray Tracing Solution for Diffuse Interreflection"
*  by Ward, Rubinstein, and Clear, in Siggraph '88 proceedings.
*
*  Basic Idea:  Never use a constant ambient term.  Instead,
*     - For first pixel, cast a whole bunch of rays in different directions
*       from the object intersection point to see what the diffuse illumination
*       really is.  Save this value, after estimating its
*       degree of reusability.  (Method 1)
*     - For second and subsequent pixels,
*         - If there are one or more nearby values already computed,
*           average them and use the result (Method 2), else
*         - Use method 1.
*
*  Implemented by and (c) 1994-6 Jim McElhiney, mcelhiney@acm.org or 71201,1326
*  All standard POV distribution rights granted.  All other rights reserved.
*************************************************************************/

#include <string.h>
#include <algorithm>
#include <boost/thread.hpp>

#include "backend/frame.h"
#include "backend/scene/view.h"
#include "backend/render/tracetask.h"
#include "backend/lighting/photons.h"
#include "backend/lighting/radiosity.h"
#include "backend/math/vector.h"
#include "backend/support/fileutil.h"
#include "backend/support/octree.h"
#include "backend/colour/colour.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace std;
using namespace pov_base;

extern BYTE_XYZ rad_samples[];

#define RAD_GRADIENT 1
#define SAW_METHOD 1

// #define SIGMOID_METHOD 1 
// #define SHOW_SAMPLE_SPOTS 1 // try this!  bright spots at sample pts 
// #define LOW_COUNT_BRIGHT 1  // this will highlight areas of low density if no extra samples are taken in the final pass 

// #define RADDEBUG 1

const DBL AVG_NEAR_EPSILON = 0.000001;
const DBL RAD_EPSILON = 0.001;

// structure used to gather weighted average during tree traversal
struct WT_AVG
{
	RGBColour Weights_Times_Illuminances; // Aggregates during traversal 
	DBL Weights;                          // Aggregates during traversal 
	int Weights_Count;                    // Count of points used, aggregates during trav 
	int Good_Count;                       // Count of points used, aggregates during trav 
	Vector3d P, N;                        // Point and Normal:  input to traverse 
	DBL Current_Error_Bound;              // see Radiosity_Error_Bound 

	RGBColour Weight_Times_Illuminance[MAX_NEAREST_COUNT];
	DBL Weight[MAX_NEAREST_COUNT];
	DBL Distance[MAX_NEAREST_COUNT];
	int Close_Count;
};


inline void VUnpack(VECTOR dest_vec, const BYTE_XYZ *pack);

/*****************************************************************************
*
* FUNCTION  VUnpack()  -  Unpacks "pack_vec" into "dest_vec" and normalizes it.
*
* INPUT
*
* OUTPUT
*
* RETURNS   Nothing
*
* AUTHOUR   Jim McElhiney
*
* DESCRIPTION
*
*  The precomputed radiosity rays are packed into a lookup array with one byte
*  for each of dx, dy, and dz.  dx and dy are scaled from the range (-1. to 1.),
*  and dz is scaled from the range (0.0 to 1.0), and both are stored in the range
*  0 to 255.
*
*  The reason for this function is that it saves a bit of memory.  There are 2000
*  entries in the table, and packing them saves 21 bytes each, or 42KB.
*
* CHANGES
*
*   --- Jan 1996 : Creation.
*
******************************************************************************/

inline void VUnpack(Vector3d& dest_vec, const BYTE_XYZ * pack_vec)
{
	dest_vec[X] = ((double)pack_vec->x * (1.0 / 255.0)) * 2.0 - 1.0;
	dest_vec[Y] = ((double)pack_vec->y * (1.0 / 255.0)) * 2.0 - 1.0;
	dest_vec[Z] = ((double)pack_vec->z * (1.0 / 255.0));

	dest_vec /= dest_vec.length(); // normalise - already good to about 1%, but we can do better 
}

inline unsigned int GetRadiosityQualityFlags(const RadiositySettings& rs)
{
	unsigned int qf = QUALITY_9;

	qf &= ~Q_AREA_LIGHT;

	if(!rs.Use_Media)
		qf &= ~Q_VOLUME;

	return qf;
}

RadiosityFunction::RadiosityFunction(shared_ptr<SceneData> sd, TraceThreadData *td, unsigned int mtl, DBL adcb,
                                     const RadiositySettings& rs, RadiosityCache& rc, Trace::CooperateFunctor& cf, bool ft) :
	trace(sd, td, min((unsigned int)(rs.Recursion_Limit), mtl), adcb * rs.adcBailout, GetRadiosityQualityFlags(rs), cf, media, *this),
	media(td, &trace, &photonGatherer),
	photonGatherer(&sd->surfacePhotonMap, sd->photonSettings),
	radiosityCache(rc),
	adcBailout(adcb),
	errorBound(rs.Error_Bound),
	realErrorBound(rs.Real_Error_Bound),
	minimumReuseRadius(rs.Min_Reuse),
	nearestReuseCount(rs.Nearest_Count),
	sampleCount(rs.Count),
	maxAllowedRadiosityLevel(rs.Recursion_Limit),
	grayLight(rs.Gray),
	ambientBrightness(rs.Brightness),
	maximumSampleBrightness(rs.Maximum_Sample_Brightness),
	addSamplesOnFinalTrace(rs.Add_On_Final_Trace),
	useLayerNormal(rs.Use_Normal),
	isFinalTrace(ft)
{
	// at run-time we need to find a way to check that
	//   (trace level of calling trace object) + (radiosity trace level) < (max trace level of calling trace)
	// the max trace level stored in our trace object only relates to the level of radiosity.
	if (ft == false)
		errorBound *= rs.Low_Error_Factor;
	for(unsigned int i = 0; i < 1600; i++)
		VUnpack(sampleDirections[i], &rad_samples[i]);
}

void RadiosityFunction::ComputeAmbient(const Ray& ray, const Vector3d& ipoint, const Vector3d& raw_normal, Vector3d layer_normal, RGBColour& ambient_colour, DBL weight)
{
	DBL temp_error_bound = errorBound;
	int reuse;

	if(useLayerNormal == false)
		layer_normal = raw_normal;

	if(weight < 0.25)
		temp_error_bound += (0.25 - weight);

	reuse = radiosityCache.FindReusableBlock(temp_error_bound, ipoint, layer_normal, ambient_colour, trace.GetCurrentTraceLevel() + 1);

	// allow more samples on final trace (rather than radiosity pretrace) - unless user says not to
	if((reuse >= nearestReuseCount) || ((isFinalTrace == true) && (addSamplesOnFinalTrace == false) && (reuse > 0)))
	{
		// TODO FIXME ra_reuse_count++;

		#ifdef LOW_COUNT_BRIGHT // this will highlight areas of low density if no extra samples are taken in the final pass - not on by default [trf]
			// use this for testing - it will tell you where too few are found
			if(reuse < nearestReuseCount)
				ambient_colour.set(4.0f);
		#endif
	}
	else
	{
		GatherLight(ray, ipoint, raw_normal, layer_normal, ambient_colour, weight);

		// always use reuse - avoids bright/dark dots [nk]
		reuse = radiosityCache.FindReusableBlock(errorBound, ipoint, layer_normal, ambient_colour, trace.GetCurrentTraceLevel() + 1);
		// TODO FIXME ra_gather_count++; // keep a running count
	}

	// note grey spelling:  american options structure with worldbeat calculations!
	ambient_colour = (ambient_colour * (1.0f - grayLight)) + (grayLight * GREY_SCALE(ambient_colour));

	// Scale up by current brightness factor prior to return
	ambient_colour *= ambientBrightness;
}

// returns true if radiosity can be traced, false otherwise (that is, if the radiosity max trace level was already reached)
bool RadiosityFunction::CheckRadiosityTraceLevel()
{
	return (trace.GetCurrentTraceLevel() < maxAllowedRadiosityLevel);
}

/*****************************************************************************
*
* FUNCTION
*
*   ra_gather
*
* INPUT
*   ipoint - a point at which the illumination is needed
*   raw_normal - the surface normal (not perturbed by the current layer) at that point
*   illuminance - a place to put the return result
*   weight - the weight of this point in final output, to drive ADC_Bailout
*   
* OUTPUT
*   The average colour of light of objects visible from the specified point.
*   The colour is returned in the illuminance parameter.
*
*   
* RETURNS
*   
* AUTHOUR
*
*   Jim McElhiney
*   
* DESCRIPTION
*    Gather up the incident light and average it.
*    Return the results in illuminance, and also cache them for later.
*    Note that last parameter is similar to weight parameter used
*    to control ADC_Bailout as a parameter to Trace(), but it also
*    takes into account that this subsystem calculates only ambient
*    values.  Therefore, coming in at the top level, the value might
*    be 0.3 if the first object hit had an ambient of 0.3, whereas
*    Trace() would have been passed a parameter of 1.0 (since it
*    calculates the whole pixel value).
*
* CHANGES
*
*   --- 1994 : Creation.
*
******************************************************************************/

void RadiosityFunction::GatherLight(const Ray& ray, const Vector3d& ipoint, const Vector3d& raw_normal, const Vector3d& layer_normal, RGBColour& illuminance, DBL weight)
{
	unsigned int cur_sample_count = sampleCount;
	unsigned int cur_nearest_reuse_count = nearestReuseCount;
	DBL cur_minimum_reuse_radius = minimumReuseRadius;

	Vector3d direction, up, min_dist_vec;
	Vector3d n2, n3;
	RGBColour dxs, dys, dzs;
	RGBColour colour_sums, temp_colour;
	DBL inverse_distance_sum, mean_dist,
	    deemed_depth, min_dist, reuse_dist_min, to_eye,
	    sum_of_inverse_dist, sum_of_dist, gradient_count;
	OT_BLOCK *block;
	bool use_raw_normal = false;

	// NK rad - compute dist_max on the fly
	// TODO - note that 3.6 code uses camera origin, not ray origin (as 3.7 does), in the distance calculation. [CJC]
	DBL maximum_distance = Vector3d(Vector3d(ray.Origin) - ipoint).length(); // radset.Dist_Max

	// to_eye only valid for primary rays (trace level is one) [trf]
	to_eye = maximum_distance;

	maximum_distance *= 0.2;

	// The number of rays to trace varies with our recursion depth 
	if(ray.IsPrimaryRay() == false)
	{
		if(trace.GetCurrentTraceLevel() > 8) // Do not change this constant for fun - it keps shift operations within valid limits! [trf]
		{
			cur_nearest_reuse_count = 1;
			cur_minimum_reuse_radius = 100.0;
			cur_sample_count = 0;
			maximum_distance = HUGE_VAL;
		}
		else
		{
			DBL shift(1 << trace.GetCurrentTraceLevel());
			cur_nearest_reuse_count = 1;
			cur_minimum_reuse_radius *= shift;
			cur_sample_count >>= (trace.GetCurrentTraceLevel() * 2);
			maximum_distance *= shift;
		}
	}

	// This appeared in POV-Ray 3.5 - it forces caching and makes "count" values below 5 in radiosity next to pointless [trf]
	if(cur_sample_count < 5)
		cur_sample_count = 5;

	// Since we'll be calculating averages, zero the accumulators 
	inverse_distance_sum = 0.0;

	min_dist = BOUND_HUGE;

	if(fabs(fabs(layer_normal[Z]) - 1.0) < 0.1)
		up = Vector3d(0.0, 1.0, 0.0); // too close to vertical for comfort, so use cross product with horizon // TODO - Does this refer to the cross products a few lines below? [trf]
	else
		up = Vector3d(0.0, 0.0, 1.0);

	n2 = cross(layer_normal, up);
	n2 /= n2.length(); // normalise
	n3 = cross(layer_normal, n2);
	n3 /= n3.length(); // normalise

	// Note that this max() forces at least one ray to be shot.
	// Otherwise, the loop does nothing, since every call to 
	// Trace() just bails out immediately!
	weight = max(trace.GetADCBailout(), weight / (DBL)cur_sample_count);

	// Initialized the accumulators for the integrals which will be come the rad gradient 
	sum_of_inverse_dist = sum_of_dist = gradient_count = 0.0;

	for(unsigned int i = 0, cursample = 0, hit = 0; i < cur_sample_count; i++)
	{
		bool ray_ok = ComputeSampleDirection(cur_sample_count, raw_normal, layer_normal, n2, n3, direction, cursample, use_raw_normal);
		Ray nray(*ipoint, *direction, Ray::OtherRay, ray.IsShadowTestRay(), false, true); // Build a ray pointing in the chosen direction
		Colour temp_full_colour;
		DBL depth = trace.TraceRay(nray, temp_full_colour, weight); // Go down in recursion, trace the result, and come back up
		RGBColour temp_colour = RGBColour(temp_full_colour);

		// NK rad - each sample is limited to a user-specified brightness 
		// this is necessary to fix problems splotchiness caused by very 
		// bright objects
		// changed lighting.c to ignore phong/specular if tracing radiosity beam 
		COLC max_ill = max3(temp_colour[pRED], temp_colour[pGREEN], temp_colour[pBLUE]);

		if((max_ill > maximumSampleBrightness) && (maximumSampleBrightness > 0.0))
			temp_colour *= (maximumSampleBrightness / max_ill);

		// Add into illumination gradient integrals 
		deemed_depth = depth;
		if(deemed_depth < maximum_distance * 10.0)
		{
			DBL depth_weight_for_this_gradient = 1.0 / deemed_depth;

			sum_of_inverse_dist += 1.0 / deemed_depth;
			sum_of_dist += deemed_depth;
			gradient_count++;

			dxs += (temp_colour * depth_weight_for_this_gradient * direction[X] * fabs(direction[X]));
			dys += (temp_colour * depth_weight_for_this_gradient * direction[Y] * fabs(direction[Y]));
			dzs += (temp_colour * depth_weight_for_this_gradient * direction[Z] * fabs(direction[Z]));
		}

		if(depth > maximum_distance)
			depth = maximum_distance;
		else
		{
			#ifdef RADSTATS
				hit++;
			#endif
		}

		if(depth < min_dist)
		{
			min_dist = depth;
			min_dist_vec = direction;
		}

		// Add into total illumination integral 

		// Ok, now we will scale the color
		colour_sums += temp_colour;

		inverse_distance_sum += 1.0 / depth;
	} // end ray sampling loop 

	// Use the accumulated values to calculate the averages needed. The sphere
	// of influence of this primary-method sample point is based on the
	// harmonic mean distance to the points encountered. (An harmonic mean is
	// the inverse of the mean of the inverses).
	illuminance = colour_sums / (COLC)cur_sample_count;

	mean_dist = (DBL)cur_sample_count / inverse_distance_sum;

	// Keep a running total of the final Illuminances we calculated 
	if(ray.IsPrimaryRay() == true)
	{
		// TODO FIXME - stats: Gather_Total += illuminance;
		// TODO FIXME - stats: Gather_Total_Count++;
	}

	// We want to cached this block for later reuse.  But,
	// if ground units not big enough, meaning that the value has very
	// limited reuse potential, forget it.
	if((ray.IsPrimaryRay() == true) || (cur_sample_count >= 5))
	{
		if(mean_dist > (maximum_distance * 0.0001)) // TODO FIXME - Should this be similar to RAD_EPSILON? Otherwise select some other *meaningful* constant! [trf]
		{
			// Theory:  We don't want to calculate a primary method ray loop at every
			// point along the inside edges, so a minimum effectivity is practical.
			// It is expressed as a fraction of the distance to the eyepoint.  1/2%
			// is a good number.  This enhancement was Greg Ward's idea, but the use
			// of % units is my idea.  [JDM]

			reuse_dist_min = to_eye * cur_minimum_reuse_radius;
			if(mean_dist < reuse_dist_min)
				mean_dist = reuse_dist_min;

			#ifdef RADSTATS
				ot_blockcount++;
			#endif

			// After end of ray loop, we've decided that this point is worth storing 
			// Allocate a block, and fill it with values for reuse in cacheing later 
			block = radiosityCache.NewBlock();

			// beta 
			if(gradient_count > 10)
			{
				DBL constant_term = gradient_count / (sum_of_inverse_dist * sum_of_dist); // TODO - check validity of this change [trf]

				block->dx = dxs * constant_term;
				block->dy = dys * constant_term;
				block->dz = dzs * constant_term;
			}

			// Fill up the values in the octree (ot_) cache block 

			block->Illuminance = illuminance;
			block->To_Nearest_Surface = min_dist_vec;
			block->Harmonic_Mean_Distance = SNGL(mean_dist);
			block->Nearest_Distance = SNGL(min_dist);
			block->Bounce_Depth = short(trace.GetCurrentTraceLevel() + 1);
			block->Point = ipoint;
			block->S_Normal = layer_normal;
			block->next = NULL;

			radiosityCache.InsertBlock(ipoint, mean_dist * realErrorBound, block);
		}
	}
}

/*****************************************************************************
*
* DESCRIPTION
*    A bit of theory: The goal is to create a set of "random" direction rays
*    so that the probability of close-to-normal versus close-to-tangent rolls
*    off in a cos-theta curve, where theta is the deviation from normal.
*    That is, lots of rays close to normal, and very few close to tangent.
*    You also want to have all of the rays be evenly spread, no matter how
*    many you want to use.  The lookup array has an array of points carefully
*    chosen to meet all of these criteria.
*
******************************************************************************/

bool RadiosityFunction::ComputeSampleDirection(unsigned int sample_count, const Vector3d& raw_normal, const Vector3d& layer_normal,
                                               const Vector3d& n2, const Vector3d& n3, Vector3d& direction, unsigned int& cursample, bool& use_raw_normal)
{
	Vector3d random_vec;
	DBL ray_ok = -1.0;
	unsigned int lastsample = 0;

	if(use_raw_normal == false)
	{
		lastsample = min(sample_count * 5 + 1, 1600u);

		// loop through here choosing rays until we get one that is not behind the surface
		if(fabs(layer_normal[Z] - 1.0) < RAD_EPSILON) // pretty well straight Z, we are within 1/20 degree of pointing in the Z axis: use all vectors as is - they're precomputed this way
		{
			while((ray_ok <= 0.0) && (cursample < lastsample))
			{
				///Increase_Counter(stats[Gather_Performed_Count]);
				direction = sampleDirections[cursample];
				ray_ok = dot(direction, raw_normal); // make sure we don't go behind raw_normal 
				cursample++;
			}
		}
		else
		{
			while((ray_ok <= 0.0) && (cursample < lastsample))
			{
				///Increase_Counter(stats[Gather_Performed_Count]);
				random_vec = sampleDirections[cursample];
				direction = ((n2 * random_vec[X]) + (n3 * random_vec[Y]) + (layer_normal * random_vec[Z]));
				ray_ok = dot(direction, raw_normal); // make sure we don't go behind raw_normal 
				cursample++;
			}
		}

		if(ray_ok <= 0.0)
			use_raw_normal = true;
	}

	if(ray_ok <= 0.0)
	{
		if(cursample == 0)
			lastsample = 1600 - 1;
		else
			lastsample = cursample - 1;

		// loop through here choosing rays until we get one that is not behind the surface
		if(fabs(raw_normal[Z] - 1.0) < RAD_EPSILON) // pretty well straight Z, we are within 1/20 degree of pointing in the Z axis: use all vectors as is - they're precomputed this way
		{
			while((ray_ok <= 0.0) && (cursample != lastsample))
			{
				///Increase_Counter(stats[Gather_Performed_Count]);
				direction = sampleDirections[cursample];
				ray_ok = dot(direction, raw_normal); // make sure we don't go behind raw_normal 
				cursample++;
				if(cursample >= 1600) // don't go beyond range of samples
					cursample = 0;
			}
		}
		else
		{
			while((ray_ok <= 0.0) && (cursample != lastsample))
			{
				///Increase_Counter(stats[Gather_Performed_Count]);
				random_vec = sampleDirections[cursample];
				direction = ((n2 * random_vec[X]) + (n3 * random_vec[Y]) + (raw_normal * random_vec[Z]));
				ray_ok = dot(direction, raw_normal); // make sure we don't go behind raw_normal 
				cursample++;
				if(cursample >= 1600) // don't go beyond range of samples
					cursample = 0;
			}
		}
	}

	return (ray_ok > 0.0);

/* [trf] This used to be:
	DBL rayOk = -1.0;
	int lockupTest = 0;

	// loop through here choosing rays until we get one that is not behind the surface
	while((rayOk <= 0.0) && (lockupTest < 1600))
	{
		lockupTest++;
		///Increase_Counter(stats[Gather_Performed_Count]);
		random_vec = fast_rad_samples[sampleNum++];

		// don't go beyond range of samples 
		if(sampleNum >= 1600)
			sampleNum = 0;

		// if we've taken too many incorrect samples, use raw normal instead of layer normal 
		if(sampleNum > current_radiosity_count * 5)
			layer_normal = raw_normal;

		if(fabs(layer_normal[Z] - 1.0) < RAD_EPSILON) // pretty well straight Z, folks 
			direction = random_vec; // we are within 1/20 degree of pointing in the Z axis: use all vectors as is - they're precomputed this way
		else
			direction = ((n2 * random_vec[X]) + (n3 * random_vec[Y]) + (layer_normal * random_vec[Z]));

		// make sure we don't go behind raw_normal 
		rayOk = dot(direction, raw_normal);
	}
*/
}


/*****************************************************************************
*
* FUNCTION  Initialize_Radiosity_Code
*
* INPUT     Nothing.
*
* OUTPUT    Sets various global states used by radiosity.  Notably,
*           ot_fd - the file identifier of the file used to save radiosity values
*
* RETURNS   1 for Success, 0 for failure  (e.g., could not open cache file)
*
* AUTHOUR   Jim McElhiney
*
* DESCRIPTION
*
* CHANGES
*
*   --- Jan 1996 : Creation.
*
******************************************************************************/

RadiosityCache::RadiosityCache(RadiositySettings& radset) :
	blockPool(NULL)
{
	ra_reuse_count = 0;
	ra_gather_count = 0;
	ot_root = NULL;
	ot_fd = NULL;
	Gather_Total_Count = 0;
	Setting_Total_Count = 0;

	#ifdef RADSTATS
		ot_seenodecount = 0;
		ot_seeblockcount = 0;
		ot_doblockcount = 0;
		ot_dotokcount = 0;
		ot_lastcount = 0;
		ot_lowerrorcount = 0;
	#endif

	bool retval = true;

	// always clear these even if radiosity isn't enabled, as otherwise
	// we get misleading statistics on subsequent non-radiosity renders
	radset.Preview_Done = 0;

	if(radset.Enabled)
	{
		// build the file name for the radiosity cache file 
		// strcpy(rad_cache_filename, opts.Scene_Name); // TODO FIXME - move to parser or elsewhere
		// strcat(rad_cache_filename, RADIOSITY_CACHE_EXTENSION);

		radset.Real_Error_Bound = radset.Error_Bound; // TODO FIXME - what is this for? [trf]
/*
		// NK rad 
		if(radset.Load_File_Name)
		{
			fd = New_Checked_IStream(radset.Load_File_Name, POV_File_Data_RCA);
			if(fd != NULL)
			{
				ot_read_file(fd);
				delete fd;
			}
			POV_FREE(radset.Load_File_Name);
			radset.Load_File_Name = NULL;
		}
		// NK ---- 

		used_existing_file = false;
		if(//((opts.Options & CONTINUE_TRACE) && radset.File_ReadOnContinue) ||  radset.File_AlwaysReadAtStart) // TODO FIXME - Continue trace detection
		{
			fd = New_Checked_IStream(rad_cache_filename, POV_File_Data_RCA);   // "myname.rca" 
			if(fd != NULL)
			{
				used_existing_file = ot_read_file(fd);
				retval &= used_existing_file;
				delete fd;
			}
		}
		else
		{
			DELETE_FILE(rad_cache_filename);  // default case, force a clean start 
		}

		if(radset.File_SaveWhileRendering)
		{
			// If we are writing a file, but not using what's there, we truncate,
			// since we conclude that what is there is bad.
			// But, if we are also using what's there, then it must be good, so
			// we just append to it.
			ot_fd = New_Checked_OStream(rad_cache_filename, POV_File_Data_RCA, used_existing_file);
			if(ot_fd == NULL)
				throw "TODO"; // TODO FIXME
		}
*/
	}

	if(retval == false)
		throw "TODO"; // TODO FIXME
}

/*****************************************************************************
*
* FUNCTION  Deinitialize_Radiosity_Code()
*
* INPUT     Nothing.
*
* OUTPUT    Sets various global states used by radiosity.  Notably,
*           ot_fd - the file identifier of the file used to save radiosity values
*
* RETURNS   1 for total success, 0 otherwise (e.g., could not save cache tree)
*
* AUTHOUR   Jim McElhiney
*
* DESCRIPTION
*   Wrap up and free any radiosity-specific features.
*   Note that this function is safe to call even if radiosity was not on.
*
* CHANGES
*
*   --- Jan 1996 : Creation.
*
******************************************************************************/

RadiosityCache::~RadiosityCache() // TODO FIXME - fix whole method!!!
{
	if (blockPool != NULL)
	{
		boost::mutex::scoped_lock lock(blockPoolMutex);
		while(blockPool != NULL)
		{
			BlockPoolUnit *b = blockPool;
			blockPool = blockPool->next;
			delete b;
		}
	}

	if (ot_root != NULL)
	{
		boost::mutex::scoped_lock lock(otMutex);
		ot_free_tree(&ot_root);
	}

/*
	char rad_cache_filename[256];
	OStream *fd;

	if(radset.Enabled)
	{
		// if the global file identifier is set, close it 
		if(ot_fd != NULL)
		{
			delete ot_fd;
			ot_fd = NULL;
		}

		// build the file name for the radiosity cache file 
// TODO FIXME		strcpy(rad_cache_filename, opts.Scene_Name);
		strcat(rad_cache_filename, RADIOSITY_CACHE_EXTENSION);

		// If user has not asked us to save the radiosity cache file, delete it 
		if(radset.File_SaveWhileRendering && !(radset.File_KeepAlways || (Stop_Flag && radset.File_KeepOnAbort)))
		{
			DELETE_FILE(rad_cache_filename);
		}

		// after-the-fact version.  This is an alternative to putting a call to 
		// ot_write_node after the call to ot_ins in ra_gather().
		// The on-the-fly version (all of the code which uses ot_fd) is superior
		// in that you will get partial results if you restart your rendering
		// with a different resolution or camera angle.  This version is superior
		// in that your rendering goes a lot quicker.

		if(radset.Save_File_Name)
		{
			fd = New_Checked_OStream(radset.Save_File_Name, POV_File_Data_RCA, false);
			if(fd != NULL)
			{
				ot_save_tree(ot_root, fd);
				delete fd;
			}
			POV_FREE(radset.Save_File_Name);
			radset.Save_File_Name = NULL;
		}

		if(!(radset.File_KeepAlways || (Stop_Flag && radset.File_KeepOnAbort)) &&  
		   !radset.File_SaveWhileRendering && ot_root != NULL)
		{
			fd = New_Checked_OStream(rad_cache_filename, POV_File_Data_RCA, false);

			if(fd != NULL)
			{
				retval &= ot_save_tree(ot_root, fd);
				delete fd;
			}
			else
				retval = false;
		}

		// Note that multiframe animations should call this free function if they have
		// moving objects and want correct results.
		// They should NOT call this function if they have no moving objects (like
		// fly-throughs) and want speed
		if(ot_root != NULL)
			retval &= ot_free_tree(&ot_root);   // this zeroes the root pointer 
	}

	if(fast_rad_samples != NULL)
		POV_FREE(fast_rad_samples);
*/
//	return retval;
}

OT_BLOCK *RadiosityCache::NewBlock()
{
	OT_BLOCK *block = NULL;
	boost::mutex::scoped_lock lock(blockPoolMutex);

	if(blockPool == NULL)
		blockPool = new BlockPoolUnit(NULL);

	block = &(blockPool->blocks[blockPool->nextFreeBlock]);

	blockPool->nextFreeBlock++;

	if(blockPool->nextFreeBlock >= BLOCK_POOL_UNIT_SIZE)
		blockPool = new BlockPoolUnit(blockPool);

	return block;
}

void RadiosityCache::InsertBlock(const Vector3d& ipoint, DBL radius, OT_BLOCK *block)
{
	OT_ID id;

	// figure out the block id 
	ot_index_sphere(const_cast<DBL *>(*ipoint), radius, &id); // TODO - cleanup octree code to avoid const_cast

	{
		// store the info block in the oct tree 
		boost::mutex::scoped_lock lock(otMutex);
		ot_ins(&ot_root, block, &id);
	}

	// In case the rendering is suspended, save the cache tree values to a file 
// TODO FIXME	if(radset.File_SaveWhileRendering && (ot_fd != NULL))
// TODO FIXME		ot_write_block(block, ot_fd);
}

/*****************************************************************************
*
* FUNCTION
*
*   ra_reuse
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOUR
*
*   Jim McElhiney
*
* DESCRIPTION
*
*   Returns whether or not there were some prestored values close enough to
*   reuse.
*
* CHANGES
*
*   --- 1994 : Creation.
*
******************************************************************************/

int RadiosityCache::FindReusableBlock(DBL errorbound, const Vector3d& ipoint, const Vector3d& snormal, RGBColour& illuminance, int tracelevel)
{
	if(ot_root != NULL)
	{
		WT_AVG gather;

		gather.Weights = 0.0;

		gather.P = ipoint;
		gather.N = snormal;

		gather.Weights_Count = 0;
		gather.Good_Count = 0;
		gather.Close_Count = 0;
		gather.Current_Error_Bound = errorbound * DBL(1 << tracelevel);

		{
			// Go through the tree calculating a weighted average of all of the usable points near this one

			// CJC I have not yet evaluated how safe it is to allow traversal of the tree
			// whilst at the same time inserts could be happening ... technically we should
			// assert the below lock until we work out a more efficient way of protecting
			// the traversal code. however it kills performance (as expected), so we'll surf
			// on the risky side and leave it unlocked for now.

			/* boost::mutex::scoped_lock lock(otMutex); */
			ot_dist_traverse(ot_root, const_cast<DBL *>(*ipoint), tracelevel, AverageNearBlock, (void *)&gather); // TODO - cleanup octree code to avoid const_cast
		}

		// Did we get any nearby points we could reuse? 
		if(gather.Good_Count > 0)
		{
			// NK rad - Average together all of the samples (sums were returned by 
			// ot_dist_traverse).  We are using nearest_count as a lower bound,
			// not an upper bound.
			illuminance = gather.Weights_Times_Illuminances / gather.Weights;
		}

		return gather.Good_Count;
	}

	return 0; // No tree, so no reused values
}

/*****************************************************************************
*
* FUNCTION
*
*   ra_average_near
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOUR
*
*   Jim McElhiney
*   
* DESCRIPTION
*
*   Tree traversal function used by ra_reuse()
*   Calculate the weight of this cached value, taking into account how far
*   it is from our test point, and the difference in surface normal angles.
*
*   Given a node with an old cached value, check to see if it is reusable, and
*   aggregate its info into the weighted average being built during the tree
*   traversal. block contains Point, Normal, Illuminance,
*   Harmonic_Mean_Distance
*
* CHANGES
*
*   --- 1994 : Creation.
*
******************************************************************************/

bool RadiosityCache::AverageNearBlock(OT_BLOCK *block, void *void_info)
{
	WT_AVG *info = (WT_AVG *)void_info;
	Vector3d delta(info->P - block->Point);   // a = b - c, which is test p minus old pt 
	DBL square_dist = VSumSqr(*delta);
	DBL quickcheck_rad = (DBL)block->Harmonic_Mean_Distance * info->Current_Error_Bound;

	#ifdef RADSTATS
		ot_doblockcount++;
	#endif

	// first we do a tuning test--this func gets called a LOT 
	if(square_dist < (quickcheck_rad * quickcheck_rad))
	{
		DBL dist = sqrt(square_dist);
		DBL ri = (DBL)block->Harmonic_Mean_Distance;
		bool dist_greater_epsilon = (dist > AVG_NEAR_EPSILON);
		Vector3d delta_unit;

		if(dist_greater_epsilon == true)
		{
			DBL cos_diff_from_nearest;

			delta_unit = delta / dist; // normalise

			// This block reduces the radius of influence when it points near the nearest
			// surface found during sampling.
			cos_diff_from_nearest = dot(block->To_Nearest_Surface, delta_unit);
			if(cos_diff_from_nearest > 0.0)
				ri = (cos_diff_from_nearest * (DBL)block->Nearest_Distance) + ((1.0 - cos_diff_from_nearest) * ri);
		}

		if(dist < (ri * info->Current_Error_Bound))
		{
			DBL dir_diff = dot(info->N, block->S_Normal);

			// NB error_reuse varies from 0 to 3.82 (1+ 2 root 2) 
			DBL error_reuse_translate = dist / ri;
			DBL error_reuse_rotate = 2.0 * sqrt(fabs(1.0 - dir_diff));
			DBL error_reuse = error_reuse_translate + error_reuse_rotate;

			// is this old point within a reasonable error distance? 
			if(error_reuse < info->Current_Error_Bound)
			{
				DBL in_front = 1.0;

				#ifdef RADSTATS
					ot_lowerrorcount++;
				#endif

				if(dist_greater_epsilon == true)
				{
					// Make sure that the old point is not in front of this point, the
					// old surface might shadow this point and make the result  meaningless
					Vector3d half(info->N + block->S_Normal);

					half /= half.length(); // normalise - needed so check can be to constant // What about this check that can be to constant? Which check? Which constant? [trf]

					in_front = dot(delta_unit, half);
				}

				// Theory:        eliminate the use of old points well in front of our
				// new point we are calculating, but not ones which are just a little
				// tiny bit in front.  This (usually) avoids eliminating points on the
				// same surface by accident.

				if(in_front > (-0.05))
				{
					DBL weight;

					#ifdef RADSTATS
						ot_dotokcount++;
					#endif

					#ifdef SIGMOID_METHOD
						weight = error_reuse / info->Current_Error_Bound;  // 0 < t < 1 
						weight = (cos(weight * M_PI) + 1.0) * 0.5;         // 0 < w < 1 
					#endif

					#ifdef SAW_METHOD
						weight = 1.0 - (error_reuse / info->Current_Error_Bound); // 0 < t < 1 
						weight = sqrt(sqrt(weight));  // less splotchy 
						//weight = sqrt(sqrt(sqrt(weight)));   maybe even less splotchy 
						//weight = weight*weight*weight*weight*weight;  more splotchy 
					#endif

					if(weight > RAD_EPSILON) // avoid floating point oddities near zero 
					{
						// This is the block where we use the gradient to improve the prediction 
						#ifdef RAD_GRADIENT
							RGBColour d((block->dx * delta[X]) + (block->dy * delta[Y]) + (block->dz * delta[Z]));
						#else
							RGBColour d(0.0f);
						#endif

						RGBColour prediction;

						// NK 6-May-2003 removed clipping - not sure why it was here in the
						// first place, but it sure causes problems for HDR scenes, and removing
						// it doesn't seem to cause problems for non-HRD scenes.
						// But we want to make sure that our deltas don't cause a positive illumination
						// to go below zero, while allowing negative illuminations to stay negative.
						if((d[pRED] + block->Illuminance[pRED] < 0.0) && (block->Illuminance[pRED]>  0.0))
							d[pRED] = -block->Illuminance[pRED];

						if((d[pGREEN] + block->Illuminance[pGREEN] < 0.0) && (block->Illuminance[pGREEN] > 0.0))
							d[pGREEN] = -block->Illuminance[pGREEN];

						if((d[pBLUE] + block->Illuminance[pBLUE] < 0.0) && (block->Illuminance[pBLUE] > 0.0))
							d[pBLUE] = -block->Illuminance[pBLUE];

						prediction = block->Illuminance + d;

						#ifdef SHOW_SAMPLE_SPOTS
							if(dist < radset.Dist_Max * 0.015)
								prediction.set(3.0);
						#endif

						// The predicted colour is an extrapolation based on the old value 
						info->Weights_Times_Illuminances += (prediction * weight);

						info->Weights += weight;
						info->Weights_Count++;
						info->Good_Count++;

						// NK rad - it fit in the error bound, so keep it.  We use all 
						// that fit the error bounding criteria.  There is no need to put
						// a maximum on the number of samples that are averaged.
					}
				}
			}
		}
	}

	return true;
}

}
