/*******************************************************************************
 * targa.cpp
 *
 * This module contains the code to read and write the Targa output file
 * format.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2009 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/base/image/targa.cpp $
 * $Revision: #29 $
 * $Change: 5033 $
 * $DateTime: 2010/06/25 12:58:53 $
 * $Author: clipka $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *      unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

/****************************************************************************
*
*  Explanation:
*
*    -
*
*  ---
*
*  May 1994 : Support for 24-bit RLE Targa output files added: John Baily
*             and David Payne.
*
*  Jul 1994 : Resume trace support and minor algorithm fix (one more still
*             needed, see comments in Write_Targa_Line); resume will force
*             Targa format to match the original trace format -- the T or C
*             format flag is adjusted as necessary: Charles Marslett,
*
*  Jun 1995 : Added support for 32-bit Targa input and output files.
*             The alpha channel has a value of 0 for 100% transparency
*             and a value of 255 for 0% transparency. [DB]
*
*  Sep 2010 : Added metadata in Write, changing targa version from 1.0 to 2.0
*
*
*****************************************************************************/

#include <vector>

// configbase.h must always be the first POV file included within base *.cpp files
#include "base/configbase.h"
#include "base/image/image.h"
#include "base/image/targa.h"
#include "base/types.h"

#include "metadata.h"
// this must be the last file included
#include "base/povdebug.h"

namespace pov_base
{

namespace Targa
{

typedef struct pix
{
  unsigned int b;
  unsigned int g;
  unsigned int r;
  unsigned int a;
} Pixel;

Pixel *GetPix (const Image *image, int x, int y, Pixel *pixel, GammaCurvePtr gamma, bool premul)
{
  image->GetEncodedRGBAValue (x, y, gamma, 255, pixel->r, pixel->g, pixel->b, pixel->a, premul);
  return (pixel);
}

void PutPix (vector<unsigned char>& line, pix *pixel, bool opaque)
{
  line.push_back (pixel->b);
  line.push_back (pixel->g);
  line.push_back (pixel->r);
  if (!opaque)
    line.push_back (pixel->a);
}
typedef char Targa_footer[26];
#define FOO_EXT_OFF 0
#define FOO_SIG_OFF 8
#define FOO_DOT_OFF 24
#define FOO_NUL_OFF 25
typedef char Targa_extension[495];
#define EXT_SIZE_OFF 0
#define EXT_SIZE_DIM 2
#define EXT_COMMENT_OFF 43
#define EXT_COMMENT_DIM 81
#define EXT_DATE_OFF 367
#define EXT_DATE_DIM 12
#define EXT_SOFT_OFF 426
#define EXT_SOFT_DIM 41
#define EXT_VERSION_OFF 467
#define EXT_VERSION_DIM 3
#define EXT_SCAN_OFF 490
#define EXT_SCAN_DIM 4
#define EXT_ALPHA_OFF 494
#define EXT_ALPHA_DIM 1
#define EXT_GAMMA_OFF 478
// and another one for pixel ratio
#define EXT_PIXRATIO_OFF 474

void Write (OStream *file, const Image *image, const Image::WriteOptions& options)
{
  pix                   current;
  pix                   next;
  bool                  opaque = !options.alphachannel;
  bool                  compress = options.compress > 0 ;
  vector<unsigned char> header;
  vector<unsigned char> line;
  GammaCurvePtr         gamma(options.encodingGamma);
  Metadata              meta;

  if (!gamma)
    gamma = SRGBGammaCurve::Get();

  // TODO ALPHA - check if TGA should really keep presuming non-premultiplied alpha
  // We presume non-premultiplied alpha, unless the user overrides
  // (e.g. to handle a non-compliant file).
  bool premul = false;
  if (options.premultiplyOverride)
    premul = options.premultiply;

  header.reserve (18);
  header.push_back (0);
  header.push_back (0);

  // byte 2 - targa file type (compressed/uncompressed)  
  header.push_back (options.compress ? 10 : 2);

  // Byte 3 - Index of first color map entry LSB
  // Byte 4 - Index of first color map entry MSB
  // Byte 5 - Color map length LSB
  // Byte 6 - Color map legth MSB
  // Byte 7 - Color map size
  // x origin set to "First_Column" - Bytes 8, 9
  // y origin set to "First_Line"   - Bytes 10, 11
  for (int i = 0; i < 9; i++)
    header.push_back (0);

  // write width and height - Bytes 12 - 15
  unsigned int w = image->GetWidth();
  unsigned int h = image->GetHeight();
  header.push_back (w % 256);
  header.push_back (w / 256);
  header.push_back (h % 256);
  header.push_back (h / 256);

  if (!opaque)
  {
    header.push_back(32);    /* 32 bits/pixel (BGRA) */
    header.push_back(0x28);  /* Data starts at top left, 8 bits Alpha */
  }
  else
  {
    header.push_back(24);    /* 24 bits/pixel (BGR) */
    header.push_back(0x20);  /* Data starts at top left, 0 bits Alpha */
  }

  if (!file->write (&header[0], 18))
    throw POV_EXCEPTION(kFileDataErr, "header write failed for targa file") ;

  line.reserve (w * 4);
  if (options.compress == 0)
  {
    Pixel pixel;
    for (int row = 0; row  < h; row++)
    {
      line.clear ();
      for (int col = 0; col < w; col++)
        PutPix (line, GetPix (image, col, row, &pixel, gamma, premul), opaque);
      if (!file->write (&line[0], line.size()))
        throw POV_EXCEPTION(kFileDataErr, "row write failed for targa file") ;
    }
  }
  else
  {
    // RLE compressed data

    for (int row = 0; row  < h; row++)
    {
      line.clear ();

      int llen = w;
      int startx = 0;
      int cnt = 1;
      int ptype = 0;
      bool writenow = false;

      GetPix (image, 0, row, &current, gamma, premul);
      while (true)
      {
        if (startx + cnt < llen)
          GetPix (image, startx + cnt, row, &next, gamma, premul);
        else
          next = current;
        if (memcmp (&current, &next, sizeof (pix)) == 0)
        {
          if (ptype == 0)
          {
            cnt++;
            if ((cnt >= 128) || ((startx + cnt) >= llen))
              writenow = true;
          }
          else
          {
            cnt--;
            writenow = true;
          }
        }
        else
        {
          if ((ptype == 1) || (cnt <= 1))
          {
            current = next;
            ptype = 1;
            cnt++;
            if ((cnt >= 128) || ((startx + cnt) >= llen))
              writenow = true;
          }
          else
            writenow = true;
        }

        if (writenow)
        {
          Pixel pixel;
          /* This test SHOULD be unnecessary!  However, it isn't!  [CWM] */
          if (startx + cnt > llen)
            cnt = llen - startx;
          if (ptype == 0)
          {
            line.push_back ((unsigned char) ((cnt - 1) | 0x80));
            PutPix (line, &current, opaque);
            current = next;
          }
          else
          {
            line.push_back ((unsigned char) cnt - 1);
            for (int x = 0; x < cnt; x++)
              PutPix (line, GetPix (image, startx + x, row, &pixel, gamma, premul), opaque);
          }
          startx += cnt;
          writenow = false;
          ptype = 0;
          cnt = 1;
          if (!file->write (&line[0], line.size()))
            throw POV_EXCEPTION(kFileDataErr, "write failed for targa file") ;
          if (startx >= llen)
             break;
          line.clear ();
        }
      }
    }
  }
  POV_LONG curpos;
  Targa_extension ext;
  Targa_footer foo;
  // A lot of "unused" get happy when filled with 0
  memset(ext,0,sizeof(ext));
  memset(foo,0,sizeof(foo));
  curpos = file->tellg();

  foo[FOO_EXT_OFF+3]= 0x0ff & (curpos >> 24);
  foo[FOO_EXT_OFF+2]= 0x0ff & (curpos >> 16);
  foo[FOO_EXT_OFF+1]= 0x0ff & (curpos >> 8);
  foo[FOO_EXT_OFF]= 0x0ff & (curpos);
  // fill signature, Reserved char & binary string terminator in one go
  sprintf(&foo[FOO_SIG_OFF],"TRUEVISION-XFILE.");

  //let's prepare the Extension area, then write Extension then footer
  ext[EXT_SIZE_OFF+1] = 0x0ff &(495 >>8);
  ext[EXT_SIZE_OFF] = 0x0ff &(495);
  string soft = meta.getSoftware().substr(0,EXT_SOFT_DIM-1);
  sprintf(&ext[EXT_SOFT_OFF],"%s",soft.c_str());
  string com = meta.getComment1().substr(0,EXT_COMMENT_DIM-1);
  sprintf(&ext[EXT_COMMENT_OFF],"%s",com.c_str());
  com = meta.getComment2().substr(0,EXT_COMMENT_DIM-1);
  sprintf(&ext[EXT_COMMENT_OFF+EXT_COMMENT_DIM],"%s",com.c_str());
  com = meta.getComment3().substr(0,EXT_COMMENT_DIM-1);
  sprintf(&ext[EXT_COMMENT_OFF+EXT_COMMENT_DIM*2],"%s",com.c_str());
  com = meta.getComment4().substr(0,EXT_COMMENT_DIM-1);
  sprintf(&ext[EXT_COMMENT_OFF+EXT_COMMENT_DIM*3],"%s",com.c_str());
  // we do not use the version field, but All 0 is not good enough
  meta.fillDate(&ext[EXT_DATE_OFF]);
  ext[EXT_VERSION_OFF]=0;
  ext[EXT_VERSION_OFF+1]=0;
  ext[EXT_VERSION_OFF+2]=' ';
  // scanline offset is 18 (as 4 bytes) as we did not fill image ID nor color map 
  ext[EXT_SCAN_OFF] = 18;
  // alpha handling
  ext[EXT_ALPHA_OFF] = opaque ? 0:premul ?4:3; 
  // gamma handling (range 0.0 to 10.0, on short, hard code a denominator to allow 10.0)
  ext[EXT_GAMMA_OFF] = (int)(6553.0/gamma->ApproximateEncodingGamma())%256;
  ext[EXT_GAMMA_OFF+1] = (int)(6553.0/gamma->ApproximateEncodingGamma())/256;
  ext[EXT_GAMMA_OFF+2] = 6553%256;
  ext[EXT_GAMMA_OFF+3] = 6553/256;
  file->write(ext,sizeof(ext));
  file->write(foo,sizeof(foo));
}

void ConvertColor (Pixel *pixel, unsigned pixelsize, unsigned char *bytes)
{
  unsigned char r;
  unsigned char g;
  unsigned char b;
  unsigned char a = 255; // default to solid colour

  switch (pixelsize)
  {
    case 8:
      r = g = b = bytes[0];
      break;

    case 15:
      r = ((bytes[1] & 0x7c) << 1);
      g = (((bytes[1] & 0x03) << 3) | ((bytes[0] & 0xe0) >> 5)) << 3;
      b = (bytes[0] & 0x1f) << 3;
      break;

    case 16:
      r = ((bytes[1] & 0x7c) << 1);
      g = (((bytes[1] & 0x03) << 3) | ((bytes[0] & 0xe0) >> 5)) << 3;
      b = (bytes[0] & 0x1f) << 3;
      a = bytes[1] & 0x80 ? 255 : 0;
      break;

    case 24:
      r = bytes[2];
      g = bytes[1];
      b = bytes[0];
      break;

    case 32:
      r = bytes[2];
      g = bytes[1];
      b = bytes[0];
      a = bytes[3];
      break;

    default:
      throw POV_EXCEPTION(kParamErr, "Bad pixelsize in targa color");
  }

  pixel->r = r;
  pixel->g = g;
  pixel->b = b;
  pixel->a = a;
}

void ConvertColor (Image::RGBAMapEntry *pixel, unsigned pixelsize, unsigned char *bytes, GammaCurvePtr gamma)
{
  unsigned char r;
  unsigned char g;
  unsigned char b;
  unsigned char a = 255; // default to solid colour

  switch (pixelsize)
  {
    case 8:
      r = g = b = bytes[0];
      break;

    case 15:
      r = ((bytes[1] & 0x7c) << 1);
      g = (((bytes[1] & 0x03) << 3) | ((bytes[0] & 0xe0) >> 5)) << 3;
      b = (bytes[0] & 0x1f) << 3;
      break;

    case 16:
      r = ((bytes[1] & 0x7c) << 1);
      g = (((bytes[1] & 0x03) << 3) | ((bytes[0] & 0xe0) >> 5)) << 3;
      b = (bytes[0] & 0x1f) << 3;
      a = bytes[1] & 0x80 ? 255 : 0;
      break;

    case 24:
      r = bytes[2];
      g = bytes[1];
      b = bytes[0];
      break;

    case 32:
      r = bytes[2];
      g = bytes[1];
      b = bytes[0];
      a = bytes[3];
      break;

    default:
      throw POV_EXCEPTION(kParamErr, "Bad pixelsize in targa color");
  }

  pixel->red   = IntDecode(gamma, r, 255);
  pixel->green = IntDecode(gamma, g, 255);
  pixel->blue  = IntDecode(gamma, b, 255);
  pixel->alpha = IntDecode(a, 255);
}

/*****************************************************************************
*
* FUNCTION
*
*   Read_Targa_Image
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   Reads a Targa image into an RGB image buffer.  Handles 8, 16, 24, 32 bit
*   formats.  Raw or color mapped. Simple raster and RLE compressed pixel
*   encoding. Right side up or upside down orientations.
*
* CHANGES
*
*   Jun 1995 : Added code for 32 bit Targa files. [DB]
*
******************************************************************************/

Image *Read (IStream *file, const Image::ReadOptions& options)
{
  int                   temp;
  int                   h;
  unsigned int          ftype;
  unsigned int          idlen;
  unsigned int          cmlen;
  unsigned int          cmsiz;
  unsigned int          cmsizB;
  unsigned int          psize;
  unsigned int          psizeB;
  unsigned int          orien;
  unsigned int          width;
  unsigned int          height;
  unsigned char         bytes[256];
  unsigned char         tgaheader[18];
  Pixel                 pixel;
  Image                 *image ;

  // TARGA files have no clearly defined gamma by default.
  // Since 1989, TARGA 2.0 defines gamma metadata which could be used if present.
  // However, as of now (2009), such information seems to be rarely included, if at all.
  GammaCurvePtr gamma;
  if (options.gammacorrect)
    gamma = GammaCurvePtr(options.defaultGamma);

  // TODO ALPHA - check if TGA should really keep presuming non-premultiplied alpha
  // We presume non-premultiplied alpha, so that's the preferred mode to use for the image container unless the user overrides
  // (e.g. to handle a non-compliant file).
  bool premul = false;
  if (options.premultiplyOverride)
    premul = options.premultiply;

  if (!file->read ((char *) tgaheader, 18))
    throw POV_EXCEPTION(kFileDataErr, "Cannot read targa file header");

  /* Decipher the header information */
  idlen  = tgaheader[ 0];
  ftype  = tgaheader[ 2];
  cmlen  = tgaheader[ 5] + (tgaheader[ 6] << 8);
  cmsiz  = tgaheader[ 7];
  width  = tgaheader[12] + (tgaheader[13] << 8);
  height = tgaheader[14] + (tgaheader[15] << 8);
  psize  = tgaheader[16];
  orien  = tgaheader[17] & 0x20; /* Right side up ? */

  cmsizB = (cmsiz+7)/8;
  psizeB = (psize+7)/8;

  bool opaque = (psize <= 24) && (psize != 16);
  bool compressed = ftype == 9 || ftype == 10 || ftype == 11;

  /* Determine if this is a supported Targa type */
  if (ftype < 1 || ftype > 11 || (ftype > 3 && ftype < 9))
    throw POV_EXCEPTION(kFileDataErr, "Unsupported file type in targa image");
  if (psize != 8 && psize != 15 && psize != 16 && psize != 24 && psize != 32)
    throw POV_EXCEPTION(kFileDataErr, "Unsupported pixel size in targa image");
  if (cmlen > 256)
    throw POV_EXCEPTION(kFileDataErr, "Unsupported color map length in targa image");

  /* Skip over the picture ID information */
  if (idlen > 0)
    if (!file->read (bytes, idlen))
      throw POV_EXCEPTION(kFileDataErr, "Cannot read from targa image");

  /* Read in the the color map (if any) */
  if (cmlen > 0)
  {
    if (psize != 8)
      throw POV_EXCEPTION(kFileDataErr, "Unsupported color map bit depth in targa image");

    vector<Image::RGBAMapEntry> colormap ;
    Image::RGBAMapEntry entry;
    for (int i = 0; i < cmlen; i++)
    {
      for (int j = 0; j < cmsizB; j++)
      {
        if ((temp = file->Read_Byte ()) == EOF)
          throw POV_EXCEPTION(kFileDataErr, "Cannot read color map from targa image");
        bytes[j] = (unsigned char) temp;
      }
      ConvertColor (&entry, cmsiz, bytes, gamma);
      colormap.push_back(entry);
    }
    Image::ImageDataType imagetype = options.itype;
    if (imagetype == Image::Undefined)
      imagetype = Image::Colour_Map ;
    image = Image::Create (width, height, imagetype, colormap) ;
    image->SetPremultiplied(premul); // specify whether the color map data has premultiplied alpha
    gamma.reset(); // gamma has been taken care of by transforming the color table.
  }
  else
  {
    Image::ImageDataType imagetype = options.itype;
    if (imagetype == Image::Undefined)
    {
      if (GammaCurve::IsNeutral(gamma))
        // No gamma correction required, raw values can be stored "as is".
        imagetype = opaque ? Image::RGB_Int8 : Image::RGBA_Int8 ;
      else
        // Gamma correction required; use an image container that will take care of that.
        imagetype = opaque ? Image::RGB_Gamma8 : Image::RGBA_Gamma8 ;
    }
    image = Image::Create (width, height, imagetype) ;
    image->SetPremultiplied(premul); // set desired storage mode regarding alpha premultiplication
    image->TryDeferDecoding(gamma, 255); // try to have gamma adjustment being deferred until image evaluation.
  }

  if (compressed)
  {
    /* RLE compressed images */
    int x = 0;
    int y = 0;
    while (y < height)
    {
      int line = orien != 0 ? y : height - y - 1 ;

      /* Grab a header */
      if ((h = file->Read_Byte ()) == EOF)
        throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
      if (h & 0x80)
      {
        h &= 0x7F;
        if (cmlen == 0)
        {
          for (int k = 0; k < psizeB; k++)
          {
            if ((temp = file->Read_Byte ()) == EOF)
              throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
            bytes[k] = (unsigned char) temp;
          }
          ConvertColor (&pixel, psize, bytes);
          while (h-- >= 0)
          {
            image->SetEncodedRGBAValue (x, line, gamma, 255, pixel.r, pixel.g, pixel.b, pixel.a, premul);
            if (++x == width)
            {
              y++;
              x = 0;
            }
          }
        }
        else
        {
          if ((temp = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
          if ((unsigned char) temp >= cmlen)
            throw POV_EXCEPTION(kFileDataErr, "Invalid color map index in targa image");
          while (h-- >= 0)
          {
            image->SetIndexedValue (x, line, (unsigned char) temp) ;
            if (++x == width)
            {
              y++;
              x = 0;
            }
          }
        }

      }
      else
      {
        /* Copy buffer */
        while (h-- >= 0)
        {
          if (cmlen == 0)
          {
            for (int k = 0; k < psizeB; k++)
            {
              if ((temp = file->Read_Byte ()) == EOF)
                throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
              bytes[k] = (unsigned char) temp;
            }
            ConvertColor (&pixel, psize, bytes);
            image->SetEncodedRGBAValue (x, line, gamma, 255, pixel.r, pixel.g, pixel.b, pixel.a, premul);
          }
          else
          {
            if ((temp = file->Read_Byte ()) == EOF)
              throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
            if ((unsigned char) temp >= cmlen)
              throw POV_EXCEPTION(kFileDataErr, "Invalid color map index in targa image");
            image->SetIndexedValue (x, line, (unsigned char) temp) ;
          }

          if (++x == width)
          {
            if (++y == height)
              break ;
            line = orien != 0 ? y : height - y - 1 ;
            x = 0;
          }
        }
      }
    }
  }
  else
  {
    /* Simple raster image file, read in all of the pixels */
    for (int y = 0; y < height; y++)
    {
      int line = orien != 0 ? y : height - y - 1 ;
      for (int x = 0; x < width; x++)
      {
        if (cmlen == 0)
        {
          for (int z = 0; z < psizeB; z++)
          {
            if ((temp = file->Read_Byte ()) == EOF)
              throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
            bytes[z] = (unsigned char) temp;
          }
          ConvertColor (&pixel, psize, bytes);
          image->SetEncodedRGBAValue (x, line, gamma, 255, pixel.r, pixel.g, pixel.b, pixel.a, premul);
        }
        else
        {
          if ((temp = file->Read_Byte ()) == EOF)
            throw POV_EXCEPTION(kFileDataErr, "Cannot read data from targa image");
          if ((unsigned char) temp >= cmlen)
            throw POV_EXCEPTION(kFileDataErr, "Invalid color map index in targa image");
          image->SetIndexedValue (x, line, (unsigned char) temp) ;
        }
      }
    }
  }

  return (image);
}

} // end of namespace Targa

}

