/*******************************************************************************
 * pvfrontend.cpp
 *
 * This module contains the default C++ interface for render frontend.
 *
 * Author: Christopher J. Cason and Thorsten Froelich.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/windows/pvfrontend.cpp $
 * $Revision: #59 $
 * $Change: 4537 $
 * $DateTime: 2008/02/09 01:25:49 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#define POVWIN_FILE
#define _WIN32_IE COMMONCTRL_VERSION
#define WIN32_LEAN_AND_MEAN

#include <windows.h>

#include "pvfrontend.h"
#include "pvengine.h"
#include "resource.h"
#include "pvdisplay.h"
#include "pvxpdisp.h"

// this must be the last file included
#include "syspovdebug.h"

using namespace std;

namespace povwin
{
  bool                  ListenMode ;
  extern int            io_restrictions ;
  extern char           *WriteDirSpecs [] ;
  extern char           *ReadDirSpecs [] ;
  extern bool           first_frame ;
  extern bool           rendering_animation ;
  extern bool           Use2kXPRenderwin ;
  extern bool           preserve_bitmap ;
  extern bool           ErrorOccurred;
  extern bool           ErrorNotified;
  extern string         ErrorMessage ;
  extern string         ErrorFilename ;
  extern unsigned       ErrorLine ;
  extern unsigned       ErrorCol ;
}

namespace pov_frontend
{
using namespace povwin;
using namespace vfe;
using namespace vfePlatform;

static vfeWinSession    *gSession;

//////////////////////////////////////////////////////////////////////////////
// WinDisplayCreator
//
// This function is set as the session DisplayCreator by CreateFrontend below.
// Whenever the core POV code wants to create an output window, the below code
// will therefore be executed.
//////////////////////////////////////////////////////////////////////////////
vfeDisplay *WinDisplayCreator (unsigned int width, unsigned int height, COLC gammaFactor, vfeSession *session, bool visible)
{
  // we attempt to minimize 'flashing' of the window (destroy followed by a re-create)
  // by checking to see if the previous window (if any) had the same dimensions. if it
  // did then we effectively re-use it by taking over its resources. the actual instance
  // is however automatically destroyed once the global display shared pointer is reset.
  WinDisplay *display = GetRenderWindow () ;
  if (display != NULL && display->GetWidth() == width && display->GetHeight() == height)
  {
    WinDisplay *p ;
#ifdef INCLUDE_NEW_LAYERED_WINDOW
    if (Use2kXPRenderwin)
      p = new WinXPDisplay (width, height, gammaFactor, session, false) ;
    else
#endif
      p = new WinLegacyDisplay (width, height, gammaFactor, session, false) ;
    if (p->TakeOver (display))
    {
      bool anim = gSession->RenderingAnimation();

      // if we're not running an animation, or if we are and it's the first
      // frame, OR the user doesn't want frame preservation, clear the window.
      if (!anim || !preserve_bitmap || first_frame)
        p->Clear () ;
      return p ;
    }
    delete p;
  }
#ifdef INCLUDE_NEW_LAYERED_WINDOW
  if (Use2kXPRenderwin)
    return new WinXPDisplay (width, height, gammaFactor, session, visible) ;
  else
#endif
    return new WinLegacyDisplay (width, height, gammaFactor, session, visible) ;
}

//////////////////////////////////////////////////////////////////////////////
// CreateFrontend
//
// Calling the below function is one of the very first things that the main
// PVENGINE code does once it starts up. Since the legacy POVWIN code only
// supports one rendering session at once, we store the session pointer in
// a global variable (gSession). The session itself is an instance of the
// vfeWinSession class declared in the windows platform-specific portion of
// VFE, and is a descendant of vfeSession. Note that gSession is static and
// not directly available outside this file; however a reference to the
// current session may be obtained by the POVWIN code via GetSession() below.
//////////////////////////////////////////////////////////////////////////////
void CreateFrontend (void)
{
  if (gSession != NULL)
    throw POV_EXCEPTION_STRING ("Session already open");
  try
  {
    gSession = new vfeWinSession();
  }
  catch(vfeException& e)
  {
    throw POV_EXCEPTION_STRING (e.what());
  }
  if (gSession == NULL)
    throw POV_EXCEPTION_STRING ("Failed to create session");
  gSession->OptimizeForConsoleOutput(false);
  if (gSession->Initialize(NULL, NULL) != vfeNoError)
  {
    gSession->Shutdown();
    string str = gSession->GetErrorString();
    delete gSession;
    gSession = NULL;
    throw POV_EXCEPTION_STRING (str.c_str());
  }
  gSession->SetDisplayCreator(WinDisplayCreator);
}

//////////////////////////////////////////////////////////////////////////////
// SetupFrontend
//
// This function loads the IO restriction paths into the session. It is called
// initially upon startup by the main POVWIN code, and subsequently any time
// the user changes the IO restriction settings via the menu options.
//////////////////////////////////////////////////////////////////////////////
void SetupFrontend (void)
{
  // 0: no restrictions
  // 1: restrict write
  // 2: restrict read/write
  gSession->ClearPaths();
  if (io_restrictions != 0)
  {
    for (char **dirspec = WriteDirSpecs ; *dirspec != NULL ; dirspec++)
      gSession->AddWritePath (string (*dirspec), true);
    if (io_restrictions > 1)
      for (char **dirspec = ReadDirSpecs ; *dirspec != NULL ; dirspec++)
        gSession->AddReadPath (string (*dirspec), true);
  }
}

//////////////////////////////////////////////////////////////////////////////
// DeleteFrontend
//
// Destroys the global session after calling its Shutdown method.
// This is more or less the last thing done by PVENGINE before exiting.
//////////////////////////////////////////////////////////////////////////////
void DeleteFrontend (void)
{
  if (gSession != NULL)
  {
    gSession->Shutdown() ;
    delete gSession;
    gSession = NULL;
  }
}

//////////////////////////////////////////////////////////////////////////////
// GetSession
//
// Returns a reference to the current vfeWinSession instance. Throws an
// exception if one does not exist. (Since sessions in the current POVWIN
// code last until exit, it is safe to return a reference).
//////////////////////////////////////////////////////////////////////////////
vfeWinSession &GetSession (void)
{
  if (gSession == NULL)
    throw POV_EXCEPTION_STRING ("No session open");
  return *gSession;
}

//////////////////////////////////////////////////////////////////////////////
// HaveSession
//
// Used by the POVWIN code to check that a session exists.
//////////////////////////////////////////////////////////////////////////////
bool HaveSession (void)
{
  return gSession != NULL;
}

#if 0
void DrawFrame()
{
  if (gDrawNow)
  {
    if (gDisplay)
    {
      WinLegacyDisplay *p = dynamic_cast<WinLegacyDisplay *>(gDisplay.get());
      p->DrawPixelBlock(0, 0, gDisplay->GetWidth() - 1, gDisplay->GetHeight() - 1, (pov_frontend::Display::RGBA8 *) gFrontBufferPtr);

      RECT rect;
      GetClientRect (p->GetHandle(), &rect);
      HDC dc=GetDC(p->GetHandle());
      BITMAPINFO bmi={p->GetBMIH()};
      StretchDIBits(dc,0,0,rect.right,rect.bottom,0,0,p->GetWidth(),p->GetHeight(),p->GetBitmapSurface(),&bmi,DIB_RGB_COLORS,SRCCOPY);
      ReleaseDC(p->GetHandle(), dc);
    }
    gDrawNow = false;
  }
}
#endif

//////////////////////////////////////////////////////////////////////////////
// ProcessSession
//
// This function is called regularly by the main POVWIN code, primarily from
// its message loop, but also in one or two other places where it might have
// to wait a while before returning to loop processing. The optional delay
// parameter (defaults to 0 if not specified) is as described for the
// vfeSession::GetStatus() method.
//
// The purpose of ProcessSession is to query vfe for any outstanding events
// (it does this by checking the status flags), and if any are available,
// processing them. Typically this involves picking up messages and routing
// them to the message output window or status bar, but it also handles change
// of state (e.g. rendering to not rendering, and so forth, in which case it
// will call back to the main POVWIN code to advise it of such).
//////////////////////////////////////////////////////////////////////////////
bool ProcessSession (int delay)
{
  if (gSession == NULL)
    throw POV_EXCEPTION_STRING ("No session open");
  vfeStatusFlags flags = gSession->GetStatus (true, delay);

  if (flags == 0)
    return (false) ;

  if ((flags & stFailed) != 0)
    ErrorOccurred = true;
  if ((flags & stRenderingAnimation) != 0)
    rendering_animation = true;
  if ((flags & stAnimationFrameCompleted) != 0)
    first_frame = false;
  if ((flags & (stStatusMessage | stAnimationStatus)) != 0)
  {
    vfeSession::StatusMessage msg(*gSession) ;
    while (gSession->GetNextStatusMessage (msg))
    {
      if (msg.m_Type == vfeSession::mGenericStatus)
      {
        strncpy (status_buffer, msg.m_Message.c_str(), sizeof (status_buffer) - 1);
        if (delay_next_status == 0)
          delay_next_status = msg.m_Delay;
      }
      else if (msg.m_Type == vfeSession::mAnimationStatus)
      {
        buffer_stream_message (mDivider, "-");
        SetStatusPanelItemText (IDC_STATUS_DATA_FRAME, "%d/%d", msg.m_Frame, msg.m_TotalFrames);
        sprintf (status_buffer, "Rendering frame %d of %d", msg.m_Frame, msg.m_TotalFrames);
      }
      else
      {
        // huh?
        assert (false);
      }
    }
  }
  if ((flags & (stStreamMessage | stErrorMessage | stWarningMessage)) != 0)
  {
    int line;
    int col;
    char str[32];
    string errormsg;
    string message;
    string filename;
    vfeSession::MessageType type;

    while (gSession->GetNextNonStatusMessage (type, message, filename, line, col))
    {
      switch (type)
      {
        case vfeSession::mDebug:
             buffer_stream_message (mDebug, message.c_str());
             break;

        case vfeSession::mInformation:
             buffer_stream_message (mAll, message.c_str());
             break;

        case vfeSession::mWarning:
        case vfeSession::mPossibleError:
             buffer_stream_message (mWarning, message.c_str());
             break;

        case vfeSession::mError:
             buffer_stream_message (mDivider, "-");
             if (ErrorMessage.empty())
             {
               ErrorMessage = message;
               ErrorLine = line;
               ErrorCol = col;
               ErrorFilename = filename;
             }
             if (filename.empty() == false)
             {
               errormsg = "\"";
               errormsg += filename + "\"";
               if (line > 0)
               {
                 sprintf(str, "%u", line);
                 errormsg += " line ";
                 errormsg += str;
               }
               errormsg += ": " + message;
               buffer_stream_message (mFatal, errormsg.c_str());
             }
             else
               buffer_stream_message (mFatal, message.c_str());
             break;

        case vfeSession::mDivider:
             buffer_stream_message (mDivider, "-");
             break;

        default:
             buffer_stream_message (mUnknown, message.c_str());
             break;
      }
    }
  }
  if ((flags & stBackendStateChanged) != 0)
  {
    State state = gSession->GetBackendState();
    SetCaption (NULL) ;
    SetStatusPanelItemText (IDC_STATUS_DATA_STATE, gSession->GetBackendStateName ()) ;
    if (state == kReady)
      update_menu_for_render (false) ;
    else if (state == kStopping)
      PVEnableMenuItem (CM_STOPRENDER, MF_GRAYED) ;
    else if (state == kPausedRendering || state == kPausedParsing)
      ShowIsPaused();
    PVEnableMenuItem (CM_RENDERSLEEP, gSession->IsPausable () ? MF_ENABLED : MF_GRAYED) ;
  }
  if ((flags & stRenderShutdown) != 0)
    render_stopped () ;
  if ((flags & stShutdown) != 0)
    return (true);
  return (false);
}

}
