/*******************************************************************************
 * platformbase.cpp
 *
 * This file implements code that is needed (and linked to) the base code.
 *
 * Author: Christopher J. Cason
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 2005-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/vfe/win/platformbase.cpp $
 * $Revision: #14 $
 * $Change: 4537 $
 * $DateTime: 2008/02/09 01:25:49 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#define WIN32_LEAN_AND_MEAN

#include <windows.h>
#include <time.h>
#include <sys/timeb.h>
#include "vfe.h"

// this must be the last file included
#include "base/povdebug.h"

////////////////////////////////////////////////////////////////////////////////////////
// Note: do not assume that the VFE code is linked with this!
//       any code in this file must be able to function when linked with the backend
//       alone; that is, if the backend is built into a stand-alone module with the
//       frontend for example using a network to connect to it.
////////////////////////////////////////////////////////////////////////////////////////

using namespace pov_base;
using namespace boost;

namespace povwin
{
  void WinMemThreadStartup();
  void WinMemThreadCleanup();
}

namespace vfePlatform
{
  ////////////////////////////////////////////////////////////////////////////////////////
  //
  // internal functions
  //
  ////////////////////////////////////////////////////////////////////////////////////////

  static bool IsNT4 (void)
  {
    OSVERSIONINFO vi ;

    vi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO) ;
    GetVersionEx (&vi) ;
    return (vi.dwPlatformId == VER_PLATFORM_WIN32_NT && vi.dwMajorVersion >= 4) ;
  }

  static int GetNumberofCPUs (void)
  {
    SYSTEM_INFO           sysinfo ;

    GetSystemInfo (&sysinfo) ;
    return (sysinfo.dwNumberOfProcessors) ;
  }

  static char *GetExceptionDescription (DWORD code)
  {
    switch (code)
    {
      case EXCEPTION_ACCESS_VIOLATION :
          return ("access violation") ;

      case EXCEPTION_DATATYPE_MISALIGNMENT :
          return ("datatype misalignment") ;

      case EXCEPTION_FLT_DENORMAL_OPERAND :
          return ("denormal floating point operand") ;

      case EXCEPTION_FLT_DIVIDE_BY_ZERO :
          return ("floating point divide by zero") ;

      case EXCEPTION_FLT_INEXACT_RESULT :
          return ("inexact floating-point result") ;

      case EXCEPTION_FLT_INVALID_OPERATION :
          return ("invlalid floating-point operation") ;

      case EXCEPTION_FLT_OVERFLOW :
          return ("floating-point overflow") ;

      case EXCEPTION_FLT_STACK_CHECK :
          return ("floating-point stack over/underflow") ;

      case EXCEPTION_FLT_UNDERFLOW :
          return ("floating-point underflow") ;

      case EXCEPTION_INT_DIVIDE_BY_ZERO :
          return ("integer divide by zero") ;

      case EXCEPTION_INT_OVERFLOW :
          return ("integer overflow") ;

      case EXCEPTION_PRIV_INSTRUCTION :
          return ("execution of privileged instruction") ;

      case EXCEPTION_IN_PAGE_ERROR :
          return ("page error") ;

      case EXCEPTION_ILLEGAL_INSTRUCTION :
          return ("execution of illegal instruction") ;

      case EXCEPTION_NONCONTINUABLE_EXCEPTION :
          return ("continuation after noncontinuable exception") ;

      case EXCEPTION_STACK_OVERFLOW :
          return ("stack overflow") ;

      case EXCEPTION_INVALID_DISPOSITION :
          return ("invalid disposition") ;

      case EXCEPTION_GUARD_PAGE :
          return ("guard page") ;

      case EXCEPTION_INVALID_HANDLE :
          return ("invalid handle") ;

      default :
          return ("Unknown exception code") ;
    }
  }
}

namespace pov_base
{
  using namespace vfePlatform;

  ////////////////////////////////////////////////////////////////////////////////////////
  //
  // needed if we define POV_DELAY_IMPLEMENTED in config.h
  //
  ////////////////////////////////////////////////////////////////////////////////////////
  void Delay(unsigned int msec)
  {
    Sleep (msec) ;
  }

  ////////////////////////////////////////////////////////////////////////////////////////
  //
  // thread support
  //
  ////////////////////////////////////////////////////////////////////////////////////////

  /////////////////////////////////////////////////////////////////////////
  // called by the base code each time a worker thread is created (the call
  // is made in the context of the new thread).
  void vfeSysThreadStartup(void)
  {
    // TODO: use a dynamically loaded pointer to SetThreadIdealProcessor()
    //       to assign an ideal processor to each thread (to improve data
    //       coherency). this only works on NT and later.
    //       also need to add any requested priority boost.

    static int count = 0 ;
    if (GetNumberofCPUs() > 1)
      SetThreadIdealProcessor(GetCurrentThread(), (count++ % GetNumberofCPUs ()) + 1) ;
    povwin::WinMemThreadStartup();
  }

  /////////////////////////////////////////////////////////////////////////
  // called by a worker thread just before it exits.
  void vfeSysThreadCleanup(void)
  {
    povwin::WinMemThreadCleanup();
  }

  ////////////////////////////////////////////////////////////////////////////////////////
  //
  // class vfeTimer (OPTIONAL)
  //
  // if you don't want to supply this class, remove the definition for POV_TIMER from
  // config.h. see the base code for documentation on the implementation requirements.
  //
  ////////////////////////////////////////////////////////////////////////////////////////

  vfeTimer::vfeTimer (bool CPUTimeIsThreadOnly)
  {
    m_IsNT = vfePlatform::IsNT4 () ;
    m_ThreadTimeOnly = CPUTimeIsThreadOnly ;
    if (m_IsNT && m_ThreadTimeOnly)
    {
      if (!DuplicateHandle (GetCurrentProcess (), GetCurrentThread (), GetCurrentProcess (), &m_ThreadHandle, 0, TRUE, DUPLICATE_SAME_ACCESS))
      {
        assert (false) ;
        m_ThreadHandle = NULL ;
        m_ThreadTimeOnly = false ;
      }
    }
    else
      m_ThreadHandle = NULL ;
    Reset () ;
  }

  vfeTimer::~vfeTimer ()
  {
    if (m_ThreadHandle != NULL)
      CloseHandle (m_ThreadHandle) ;
  }

  unsigned __int64 vfeTimer::GetWallTime (void) const
  {
    struct timeb tb ;
    ftime(&tb);
    return ((unsigned __int64) tb.time * 1000 + tb.millitm) ;
  }

  unsigned __int64 vfeTimer::GetCPUTime (void) const
  {
    __int64     kt ;
    __int64     ut ;
    FILETIME    ct ;
    FILETIME    et ;

    if (!m_IsNT)
      return (GetWallTime ()) ;

    if (m_ThreadTimeOnly)
    {
      if (!GetThreadTimes (m_ThreadHandle, &ct, &et, (FILETIME *) &kt, (FILETIME *) &ut))
      {
        assert (false) ;
        return (0) ;
      }
    }
    else
    {
      if (!GetProcessTimes (GetCurrentProcess (), &ct, &et, (FILETIME *) &kt, (FILETIME *) &ut))
      {
        assert (false) ;
        return (0) ;
      }
    }
    return ((kt + ut) / 10000) ;
  }

  POV_LONG vfeTimer::ElapsedRealTime (void) const
  {
    struct timeb tb ;
    ftime(&tb);
    unsigned __int64 now = (unsigned __int64) tb.time * 1000 + tb.millitm;
    return (now - m_WallTimeStart) ;
  }

  POV_LONG vfeTimer::ElapsedCPUTime (void) const
  {
    if (m_IsNT)
      return (GetCPUTime () - m_CPUTimeStart) ;
    return (GetWallTime () - m_WallTimeStart) ;
  }

  void vfeTimer::Reset (void)
  {
    struct timeb tb ;
    ftime(&tb);
    m_WallTimeStart = GetWallTime () ;
    m_CPUTimeStart = GetCPUTime () ;
  }

  bool vfeTimer::HasValidCPUTime() const
  {
    return (m_IsNT) ;
  }

  ////////////////////////////////////////////////////////////////////////////////////////
  //
  // path parsing
  //
  ////////////////////////////////////////////////////////////////////////////////////////

  ////////////////////////////////////////////////////////////////////////////////////////
  // The first argument is the input, a UCS2 string.
  //
  // The second argument is the string you are supposed to return the "volume" 
  // name with.  For DOS-style paths this implies i.e. "A:\" is the "volume". 
  // Note that it is essential that the first "path" separator is also part of 
  // the volume name.  If the path is relative, the "volume" name shall be empty. 
  // 
  // This trick is necessary so the code can account for the lack of volume names 
  // in Unix-style paths: In Unix, the POV_PARSE_PATH_STRING function will have 
  // to take a reading "/" or "~/" as "volume" name.  This makes it possible to 
  // determine if a string is absolute or relative based on this 'virtual' 
  // "volume" name rather than some flags.
  // 
  // The third is a vector of strings you have to return, each has to contain the 
  // folder name from left to right, without the path separator, of course.
  // 
  // The fourth argument shall contain the filename, if any was given in the 
  // source string. By definition if the source string does not contain a
  // trailing path separator, whatever comes after the last path separator
  // (or the start of the string if there is none) must be considered a filename,
  // even if it could be a directory (in other words, don't call a system function
  // to find out if it is a dir or not - see below).
  // 
  // Please note that the function must not attempt to determine the validity of
  // a string by accessing the filesystem.  It has to parse anything that it is 
  // given.  If the string provided cannot be parsed for some reason (that is if 
  // you can determine that a given path cannot possibly be valid i.e. because it 
  // contains invalid characters), the function has to return false.  It may not 
  // throw exceptions.  The return value for success is true.
  ////////////////////////////////////////////////////////////////////////////////////////

  #define UCS2toASCIIString POVMS_UCS2toASCIIString
  #define ASCIItoUCS2String POVMS_ASCIItoUCS2String

  bool vfeParsePathString (const UCS2String& path, UCS2String& volume, vector<UCS2String>& components, UCS2String& filename)
  {
    char str [MAX_PATH * 4] ;

    volume.clear() ;
    filename.clear() ;
    components.clear() ;

    if (path.empty() == true)
      return (true);
    if (path.size () >= sizeof (str))
      return (false) ;
    strcpy (str, UCS2toASCIIString (path).c_str ()) ;
    char lastch = str[strlen(str) - 1];

    // now determine if it's a network or drive path.
    // (we could use the shlwapi functions here but I'd rather avoid the shlwapi.dll dependency).
    char *p2 = str ;
    if ((strlen (str) > 1) && ((str [1] == ':') || (str[0] == '\\' && str[1] == '\\')))
    {
      if (str [1] == ':')
      {
        // if it's a drive reference the first character must be in range 'a' - 'z'
        if (!isalpha (str[0]))
          return (false) ;

        // currently we don't support relative paths if a volume is specified
        if (str [2] != '\\')
          return (false) ;
        volume = ASCIItoUCS2String (string (str).substr (0, 3).c_str()) ;
        p2 += 3 ;
      }
      else
      {
        // it's a UNC path ... look for the next separator
        if ((p2 = strchr (str + 2, '\\')) == NULL)
        {
          // no separator; technically this is valid, but it's a relative reference
          // and as above we don't currently support this.
          return (false) ;
        }
        volume = ASCIItoUCS2String (string (str).substr (0, (size_t) (++p2 - str)).c_str()) ;
      }
    }
    else if ((str [0] == '\\') || (str [0] == '/'))
    {
      // it's a path relative to the root of the current drive.
      // we will use '\' as the volume name.
      // for volume-relative paths we also accept '/' as a path separator
      volume = ASCIItoUCS2String("\\");
      p2++;
    }

    // p2 now points at the start of any path or file components
    // the first call to strtok will skip over any extra separators
    // at the start of the path
    for (char *p1 = strtok (p2, "\\/"); p1 != NULL; p1 = strtok (NULL, "/\\"))
    {
      if (*p1 == '\0')
        continue;
      if (p1[0] == '.' && p1[1] == '\0')
        continue;
      if (p1[0] == '.' && p1[1] == '.' && p1[2] == '\0')
      {
        // it's a relative directory reference ... see if we can pop a
        // path from components; if not we leave it in there since it
        // is permitted to refer to a directory above the CWD
        if (components.empty() == false)
        {
          components.pop_back();
          continue;
        }
      }
      components.push_back (ASCIItoUCS2String (p1)) ;
    }

    // the filename, if present, will be the last entry in components.
    // we first check the last character of the supplied path to see
    // if it's a path separator char; if it is there's no filename.
    if (lastch == '\\' || lastch == '/')
      return true;

    if (components.empty() == false)
    {
      filename = components.back();
      components.pop_back();
    }

    return true ;
  }

}
