/*******************************************************************************
 * image.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/base/image/image.cpp $
 * $Revision: #51 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#include <vector>
#include <algorithm>
#include <cassert>

#include "base/image/image.h"
#include "base/image/targa.h"
#include "base/image/gif.h"
#include "base/image/png_pov.h"
#include "base/image/jpeg_pov.h"
#include "base/image/iff.h"
#include "base/image/pgm.h"
#include "base/image/ppm.h"
#include "base/image/tiff_pov.h"
#include "base/image/bmp.h"
#include "base/image/openexr.h"
#include "base/image/hdr.h"

#ifdef USE_SYSPROTO
#include "syspovprotobase.h"
#endif

#define CHECK_BOUNDS(x,y) assert(((x) < width) && ((y) < height))

// this must be the last file included
#include "base/povdebug.h"

namespace pov_base
{

using namespace std;

template<class Allocator = allocator<bool> >
class BitMapImage : public Image
{
	public:
		BitMapImage(unsigned int w, unsigned int h) :
			Image(w, h, Bit_Map) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		~BitMapImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return 1;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return pixels[x + y * width];
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			if(pixels[x + y * width] == true)
				return 1.0f;
			else
				return 0.0f;
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			gray = GetGrayValue(x, y);
			alpha = 1.0f;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			red = green = blue = GetGrayValue(x, y);
			alpha = 0.0f;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			red = green = blue = GetGrayValue(x, y);
			filter = transm = 0.0f;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = bit;
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0.0f);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = ((red * green * blue) != 0.0f);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = ((red + green + blue) != 0);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			fill(pixels.begin(), pixels.end(), bit);
		}
		void FillGrayValue(float gray)
		{
			FillBitValue(gray != 0.0f);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillBitValue(gray != 0);
		}
		void FillGrayAValue(float gray, float)
		{
			FillBitValue(gray != 0.0f);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillBitValue(gray != 0);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillBitValue((red * green * blue) != 0.0f);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillBitValue((red + green + blue) != 0);
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<bool, Allocator> pixels;
};

typedef BitMapImage<> MemoryBitMapImage;

template<class Allocator = allocator<unsigned char> >
class ColourMapImage : public Image
{
	public:
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		~ColourMapImage() { }

		bool IsOpaque() const
		{
			if((colormaptype != RGBAColourMap) && (colormaptype != RGBFTColourMap))
				return true;

			bool transp = false;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				if(colormaptype == RGBAColourMap)
					transp = (colormap[i].filter < 1.0); // this is really alpha
				else if(colormaptype == RGBFTColourMap)
					transp = (colormap[i].transm > 0.0);
				if(transp == true)
					break;
			}

			if(transp == false)
				return true;

			if(colormaptype == RGBAColourMap)
			{
				for(typename vector<unsigned char, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i++)
				{
					if(colormap[*i].filter < 1.0)
						return false;
				}
			}
			else
			{
				for(typename vector<unsigned char, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i++)
				{
					if(colormap[*i].transm > 0.0)
						return false;
				}
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return true;
		}
		bool HasAlphaChannel() const
		{
			return (colormaptype == RGBAColourMap);
		}
		bool HasFilterTransmit() const
		{
			return (colormaptype == RGBFTColourMap);
		}
		unsigned int GetMaxIntValue() const
		{
			return 255;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm, alpha;
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					return ((red * green * blue) != 0.0f);
				case RGBAColourMap:
					GetRGBAValue(x, y, red, green, blue, alpha);
					return ((red * green * blue) != 0.0f) && (alpha == 0.0f);
				case RGBFTColourMap:
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					return ((red * green * blue) != 0.0f) && (filter == 0.0f) && (transm == 0.0f);
				default:
					return 0.0f;
			}
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm, alpha;
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					return RGB2Gray(red, green, blue);
				case RGBAColourMap:
					GetRGBAValue(x, y, red, green, blue, alpha);
					return RGB2Gray(red, green, blue);
				case RGBFTColourMap:
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					return RGB2Gray(red, green, blue);
				default:
					return 0.0f;
			}
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue, filter, transm;
			alpha = 1.0f;
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					gray = RGB2Gray(red, green, blue);
					return;
				case RGBAColourMap:
					GetRGBAValue(x, y, red, green, blue, alpha);
					gray = RGB2Gray(red, green, blue);
					return;
				case RGBFTColourMap:
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					gray = RGB2Gray(red, green, blue);
					// TODO - should filter and transm be converted to alpha? [trf]
					// currently we just return (1.0-transmit) as alpha, this isn't
					// a good solution, especially given transmit can be <0 or >1 [cjc]
					alpha = 1.0f - transm;
					return;
				default:
					gray = 0.0f;
					return;
			}
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
			alpha = 0.0;
			switch(colormaptype)
			{
				case RGBAColourMap:
					alpha = e.filter;
					return;
				case RGBFTColourMap:
					// TODO - should filter and transm be converted to alpha? [trf]
					// currently we just return (1.0-transmit) as alpha, this isn't
					// a good solution, especially given transmit can be <0 or >1 [cjc]
					alpha = 1.0f - e.transm;
					return;
				default:
					return;
			}
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
			filter = 0.0;
			transm = 0.0;
			switch(colormaptype)
			{
				case RGBAColourMap:
					// TODO - should alpha be converted to filter and transm? [trf]
					// currently we just return (1.0-alpha) as transmit, this isn't
					// a good solution. [cjc]
					transm = 1.0f - e.filter; // NB e.filter is really alpha for RGBA maps
					filter = 0.0f;
					return;
				case RGBFTColourMap:
					filter = e.filter;
					transm = e.transm;
					return;
				default:
					return;
			}
		}
		unsigned char GetIndexedValue(unsigned int x, unsigned int y)
		{
			CHECK_BOUNDS(x, y);
			return pixels[x + y * width];
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Bit2Map(bit);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Gray2Map(gray);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Gray2Map(float(gray) / 255.0);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = GrayA2Map(gray, alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = GrayA2Map(float(gray) / 255.0, float(alpha) / 255.0);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGB2Map(red, green, blue);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGB2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBA2Map(red, green, blue, alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBA2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(alpha) / 255.0);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBFT2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(filter) / 255.0, float(transm) / 255.0);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBFT2Map(float(col.red()) / 255.0, float(col.green()) / 255.0, float(col.blue()) / 255.0, float(col.filter()) / 255.0, float(col.transm()) / 255.0);
		}
		void SetIndexedValue(unsigned int x, unsigned int y, unsigned char index)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = index;
		}

		void FillBitValue(bool bit)
		{
			fill(pixels.begin(), pixels.end(), Bit2Map(bit));
		}
		void FillGrayValue(float gray)
		{
			fill(pixels.begin(), pixels.end(), Gray2Map(gray));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), Gray2Map(float(gray) / 255.0));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			fill(pixels.begin(), pixels.end(), GrayA2Map(gray, alpha));
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			fill(pixels.begin(), pixels.end(), GrayA2Map(float(gray) / 255.0, float(alpha) / 255.0));
		}
		void FillRGBValue(float red, float green, float blue)
		{
			fill(pixels.begin(), pixels.end(), RGB2Map(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			fill(pixels.begin(), pixels.end(), RGB2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			fill(pixels.begin(), pixels.end(), RGBA2Map(red, green, blue, alpha));
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			fill(pixels.begin(), pixels.end(), RGBA2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(alpha) / 255.0));
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			fill(pixels.begin(), pixels.end(), RGBFT2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(filter) / 255.0, float(transm) / 255.0));
		}
		void FillIndexedValue(unsigned char index)
		{
			fill(pixels.begin(), pixels.end(), index);
		}
	private:
		vector<unsigned char, Allocator> pixels;

		unsigned char Bit2Map(bool bit) const
		{
			if(bit == true)
				return Gray2Map(1.0f);
			else
				return Gray2Map(0.0f);
		}
		unsigned char Gray2Map(float gray) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(gray, gray, gray);
				case RGBAColourMap:
					return FindBestRGBA(gray, gray, gray, 0.0f);
				case RGBFTColourMap:
					return FindBestRGBFT(gray, gray, gray, 0.0f, 0.0f);
				default:
					return 0;
			}
		}
		unsigned char GrayA2Map(float gray, float alpha) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(gray, gray, gray);
				case RGBAColourMap:
					return FindBestRGBA(gray, gray, gray, alpha);
				case RGBFTColourMap:
					return FindBestRGBFT(gray, gray, gray, 0.0f, 1.0f - alpha); // TODO - should alpha be converted to filter and transm? [trf]
				default:
					return 0;
			}
		}
		unsigned char RGB2Map(float red, float green, float blue) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, 0.0f);
				case RGBFTColourMap:
					return FindBestRGBFT(red, green, blue, 0.0f, 0.0f);
				default:
					return 0;
			}
		}
		unsigned char RGBA2Map(float red, float green, float blue, float alpha) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, alpha);
				case RGBFTColourMap:
					return FindBestRGBFT(red, green, blue, 0.0f, 1.0f - alpha); // TODO - should alpha be converted to filter and transm? [trf]
				default:
					return 0;
			}
		}
		unsigned char RGBFT2Map(float red, float green, float blue, float filter, float transm) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, 1.0f - transm); // TODO - should filter and transm be converted to alpha? [trf]
				case RGBFTColourMap:
					return FindBestRGBFT(red, green, blue, filter, transm);
				default:
					return 0;
			}
		}
		unsigned char FindBestRGB(float red, float green, float blue) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d(RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)));
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
		unsigned char FindBestRGBA(float red, float green, float blue, float alpha) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d((RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)) * 3.0f +
				         fabs(colormap[i].filter - alpha)) / 4.0f);
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
		unsigned char FindBestRGBFT(float red, float green, float blue, float filter, float transm) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d((RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)) * 3.0f +
				         fabs(colormap[i].filter - filter) + fabs(colormap[i].transm - transm)) / 5.0f);
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
};

typedef ColourMapImage<> MemoryColourMapImage;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class GrayImage : public Image
{
	public:
		GrayImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		~GrayImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return (pixels[x + y * width] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return float(pixels[x + y * width]) / float(TMAX);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = float(pixels[x + y * width]) / float(TMAX);
			alpha = 1.0f;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			red = green = blue = GetGrayValue(x, y);
			alpha = 0.0f;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			red = green = blue = GetGrayValue(x, y);
			filter = transm = 0.0f;
		}
		unsigned char GetIndexedValue(unsigned int x, unsigned int y)
		{
			CHECK_BOUNDS(x, y);
			return (unsigned char)(int(pixels[x + y * width]) / ((TMAX + 1) >> 8));
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetGrayValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), T(gray));
		}
		void FillGrayAValue(float gray, float)
		{
			FillGrayValue(gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillGrayValue(gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillGrayValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef GrayImage<unsigned char, 255, Image::Gray_Int8> MemoryGray8Image;

typedef GrayImage<unsigned short, 65535, Image::Gray_Int16> MemoryGray16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class GrayAImage : public Image
{
	public:
		GrayAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		~GrayAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 2)
			{
				if(i[1] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return (pixels[(x + y * width) * 2] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return float(pixels[(x + y * width) * 2]) / float(TMAX);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = float(pixels[(x + y * width) * 2]) / float(TMAX);
			alpha = float(pixels[(x + y * width) * 2 + 1]) / float(TMAX);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
			// TODO - should alpha be converted to filter and transm? [trf]
			filter = 0.0f;
			transm = 1.0f - alpha;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayAValue(x, y, TMAX, TMAX);
			else
				SetGrayAValue(x, y, (unsigned int)0, TMAX);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetGrayAValue(x, y, gray, 1.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayAValue(x, y, gray, (unsigned int)TMAX);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = T(gray * float(TMAX));
			pixels[(x + y * width) * 2 + 1] = T(alpha * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = gray;
			pixels[(x + y * width) * 2 + 1] = alpha;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetGrayValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			SetGrayAValue(x, y, RGB2Gray(red, green, blue), alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			SetGrayAValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)), float(alpha) / float(TMAX));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float transm)
		{
			// TODO - should filter and transm be converted to alpha? [trf]
			SetGrayAValue(x, y, RGB2Gray(red, green, blue), 1.0f - transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			// TODO - should filter and transm be converted to alpha? [trf]
			SetGrayAValue(x, y, RGB2Gray(col.red(), col.green(), col.blue()), 1.0f - col.transm());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayAValue(gray, (unsigned int)0);
		}
		void FillGrayAValue(float gray, float alpha)
		{
			T g(gray / float(TMAX)), a(alpha / float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = g;
				i++;
				*i = a;
			}
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(gray);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillGrayValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			FillGrayAValue(RGB2Gray(red, green, blue), alpha);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			FillGrayAValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)), float(alpha) / float(TMAX));
		}
		void FillRGBFTValue(float red, float green, float blue, float, float transm)
		{
			// TODO - should filter and transm be converted to alpha? [trf]
			FillGrayAValue(RGB2Gray(red, green, blue), 1.0f-transm);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef GrayAImage<unsigned char, 255, Image::GrayA_Int8> MemoryGrayA8Image;

typedef GrayAImage<unsigned short, 65535, Image::GrayA_Int16> MemoryGrayA16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class RGBImage : public Image
{
	public:
		RGBImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		~RGBImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return ((red * green * blue) != 0.0f);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			gray = GetGrayValue(x, y);
			alpha = 1.0f;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			CHECK_BOUNDS(x, y);
			red = float(pixels[(x + y * width) * 3]) / float(TMAX);
			green = float(pixels[(x + y * width) * 3 + 1]) / float(TMAX);
			blue = float(pixels[(x + y * width) * 3 + 2]) / float(TMAX);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetRGBValue(x, y, red, green, blue);
			alpha = 0.0f;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			GetRGBValue(x, y, red, green, blue);
			filter = transm = 0.0f;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBValue(x, y, gray, gray, gray);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = T(red * float(TMAX));
			pixels[(x + y * width) * 3 + 1] = T(green * float(TMAX));
			pixels[(x + y * width) * 3 + 2] = T(blue * float(TMAX));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = T(red);
			pixels[(x + y * width) * 3 + 1] = T(green);
			pixels[(x + y * width) * 3 + 2] = T(blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), gray);
		}
		void FillGrayAValue(float gray, float)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			T r(red / float(TMAX)), g(green / float(TMAX)), b(blue / float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
			}
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
			}
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef RGBImage<unsigned char, 255, Image::RGB_Int8> MemoryRGB8Image;

typedef RGBImage<unsigned short, 65535, Image::RGB_Int16> MemoryRGB16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class RGBAImage : public Image
{
	public:
		RGBAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		~RGBAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 4)
			{
				if(i[3] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return ((red * green * blue) != 0.0f);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue;
			GetRGBAValue(x, y, red, green, blue, alpha);
			gray = RGB2Gray(red, green, blue);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			red = float(pixels[(x + y * width) * 4]) / float(TMAX);
			green = float(pixels[(x + y * width) * 4 + 1]) / float(TMAX);
			blue = float(pixels[(x + y * width) * 4 + 2]) / float(TMAX);
			alpha = float(pixels[(x + y * width) * 4 + 3]) / float(TMAX);
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			// TODO - should alpha be converted to filter and transm? [trf]
			// currently we just return (1.0-alpha) as transmit, this isn't
			// a good solution. [cjc]
			transm = 1.0f - alpha;
			filter = 0.0f;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, 1.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, TMAX);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBAValue(x, y, red, green, blue, 1.0f);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBAValue(x, y, red, green, blue, TMAX);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(red * float(TMAX));
			pixels[(x + y * width) * 4 + 1] = T(green * float(TMAX));
			pixels[(x + y * width) * 4 + 2] = T(blue * float(TMAX));
			pixels[(x + y * width) * 4 + 3] = T(alpha * float(TMAX));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(red);
			pixels[(x + y * width) * 4 + 1] = T(green);
			pixels[(x + y * width) * 4 + 2] = T(blue);
			pixels[(x + y * width) * 4 + 3] = T(alpha);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float transm)
		{
			// TODO - should filter and transm be converted to alpha? [trf]
			// currently we just use (1.0-alpha) as transmit, this isn't
			// a good solution. [cjc]
			SetRGBAValue(x, y, red, green, blue, 1.0f - transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			// TODO - should filter and transm be converted to alpha? [trf]
			// currently we just use (1.0-alpha) as transmit, this isn't
			// a good solution. [cjc]
			pixels[(x + y * width) * 4] = T(col.red());
			pixels[(x + y * width) * 4 + 1] = T(col.green());
			pixels[(x + y * width) * 4 + 2] = T(col.blue());
			pixels[(x + y * width) * 4 + 3] = T(1.0f - col.transm());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillRGBAValue(gray, gray, gray, 1.0f);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillRGBAValue(gray, gray, gray, TMAX);
		}
		void FillGrayAValue(float gray, float alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBAValue(red, green, blue, 1.0f);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBAValue(red, green, blue, TMAX);
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			T r(red / float(TMAX)), g(green / float(TMAX)), b(blue / float(TMAX)), a(alpha / float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
				i++;
				*i = a;
			}
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBFTValue(float red, float green, float blue, float, float transm)
		{
			// TODO - should filter and transm be converted to alpha? [trf]
			FillRGBAValue(red, green, blue, 1.0f - transm);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef RGBAImage<unsigned char, 255, Image::RGBA_Int8> MemoryRGBA8Image;

typedef RGBAImage<unsigned short, 65535, Image::RGBA_Int16> MemoryRGBA16Image;

template<class Allocator = allocator<float> >
class RGBFTImage : public Image
{
	public:
		RGBFTImage(unsigned int w, unsigned int h) :
			Image(w, h, RGBFT_Float) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		~RGBFTImage() { }

		bool IsOpaque() const
		{
			for(typename vector<float, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 5)
			{
				if(i[4] > 0.0f)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return true;
		}
		bool IsInt() const
		{
			return false;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return true;
		}
		unsigned int GetMaxIntValue() const
		{
			return 255;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return ((red * green * blue) != 0.0f);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			gray = RGB2Gray(red, green, blue);
			// TODO - should filter and transm be converted to alpha? [trf]
			// currently we just return (1.0-transmit) as alpha, this isn't
			// a good solution, especially given transmit can be <0 or >1 [cjc]
			alpha = 1.0f - transm;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			// TODO - should filter and transm be converted to alpha? [trf]
			// currently we just return (1.0-transmit) as alpha, this isn't
			// a good solution, especially given transmit can be <0 or >1 [cjc]
			alpha = 1.0f - transm;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			CHECK_BOUNDS(x, y);
			red = pixels[(x + y * width) * 5];
			green = pixels[(x + y * width) * 5 + 1];
			blue = pixels[(x + y * width) * 5 + 2];
			filter = pixels[(x + y * width) * 5 + 3];
			transm = pixels[(x + y * width) * 5 + 4];
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, 1.0f);
			else
				SetGrayValue(x, y, 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBFTValue(x, y, gray, gray, gray, 0.0f, 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayValue(x, y, float(gray) / 255.0f);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			SetRGBFTValue(x, y, gray, gray, gray, 0.0f, 1.0f - alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float c = float(gray) / 255.0f;
			SetRGBFTValue(x, y, c, c, c, 0.0f, float(255 - alpha) / 255.0f);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBFTValue(x, y, red, green, blue, 0.0f, 0.0f);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBFTValue(x, y, float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, 0.0f, 0.0f);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			SetRGBFTValue(x, y, red, green, blue, 0.0f, 1.0f - alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			SetRGBFTValue(x, y, red, green, blue, 0, 255 - alpha);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 5] = red;
			pixels[(x + y * width) * 5 + 1] = green;
			pixels[(x + y * width) * 5 + 2] = blue;
			pixels[(x + y * width) * 5 + 3] = filter;
			pixels[(x + y * width) * 5 + 4] = transm;
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 5] = col.red();
			pixels[(x + y * width) * 5 + 1] = col.green();
			pixels[(x + y * width) * 5 + 2] = col.blue();
			pixels[(x + y * width) * 5 + 3] = col.filter();
			pixels[(x + y * width) * 5 + 4] = col.transm();
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(1.0f);
			else
				FillGrayValue(0.0f);
		}
		void FillGrayValue(float gray)
		{
			FillRGBFTValue(gray, gray, gray, 0.0f, 0.0f);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayValue(float(gray) / 255.0f);
		}
		void FillGrayAValue(float gray, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			FillRGBFTValue(gray, gray, gray, 0.0f, 1.0f - alpha);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			FillGrayAValue(float(gray) / 255.0f, float(alpha) / 255.0f);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBFTValue(red, green, blue, 0.0f, 0.0f);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBFTValue(float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, 0.0f, 0.0f);
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			FillRGBFTValue(red, green, blue, 0.0f, 1.0f - alpha);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			FillRGBFTValue(red, green, blue, 0.0, 1.0f - alpha);
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			for(typename vector<float, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = red;
				i++;
				*i = green;
				i++;
				*i = blue;
				i++;
				*i = filter;
				i++;
				*i = transm;
			}
		}
	private:
		vector<float, Allocator> pixels;
};

typedef RGBFTImage<> MemoryRGBFTImage;

void RGBMap2RGBAMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBAMapEntry>& n);
void RGBMap2RGBFTMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBFTMapEntry>& n);
void RGBAMap2RGBMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBMapEntry>& n);
void RGBAMap2RGBFTMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBFTMapEntry>& n);
void RGBFTMap2RGBMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBMapEntry>& n);
void RGBFTMap2RGBAMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBAMapEntry>& n);

void RGBMap2RGBAMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBAMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBAMapEntry(i->red, i->green, i->blue, 0.0f));
}

void RGBMap2RGBFTMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBFTMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBFTMapEntry(i->red, i->green, i->blue, 0.0f, 0.0f));
}

void RGBAMap2RGBMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBAMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBMapEntry(i->red, i->green, i->blue));
}

void RGBAMap2RGBFTMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBFTMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBAMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBFTMapEntry(i->red, i->green, i->blue, 0.0f, 1.0f - i->alpha)); // TODO - should alpha be converted to filter and transm? [trf]
}

void RGBFTMap2RGBMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBFTMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBMapEntry(i->red, i->green, i->blue));
}

void RGBFTMap2RGBAMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBAMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBFTMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBAMapEntry(i->red, i->green, i->blue, 1.0f - i->transm)); // TODO - should filter and transm be converted to alpha? [trf]
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h);
			case Gray_Int8:
				return new MemoryGray8Image(w, h);
			case Gray_Int16:
				return new MemoryGray16Image(w, h);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h);
				#endif
				return new MemoryRGBFTImage(w, h);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBAMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_RGBFT_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_RGBFT_IMAGE_ALLOCATOR>(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}


Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBFTMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Read(ImageFileType type, IStream *file, const ReadOptions& options)
{
  #ifdef SYS_TO_STANDARD
    if (type == SYS)
      type = SYS_TO_STANDARD ;
  #endif

  switch (type)
  {
    case HDR:
      return (HDR::Read(file, options));

    case EXR:
#ifndef OPENEXR_MISSING
      return (OpenEXR::Read(file, options));
#else
      throw POV_EXCEPTION(kCannotOpenFileErr,
"This unofficial POV-Ray binary was built without support for the OpenEXR \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the OpenEXR library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: HDR.");
      return NULL;
#endif

    case PNG:
#ifndef LIBPNG_MISSING
      return (Png::Read(file, options));
#else
      throw POV_EXCEPTION(kCannotOpenFileErr,
"This unofficial POV-Ray binary was built without support for the PNG \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the libPNG library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: GIF, TGA, IFF, PGM, PPM, BMP.");
      return NULL;
#endif

    case GIF:
      return (Gif::Read(file, options, false));

    case POT:
      return (Gif::Read(file, options, true));

    case TGA:
      return (Targa::Read(file, options));

    case JPEG:
#ifndef LIBJPEG_MISSING
      return (Jpeg::Read(file, options));
#else
      throw POV_EXCEPTION(kCannotOpenFileErr,
"This unofficial POV-Ray binary was built without support for the JPEG \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the libJPEG library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: GIF, TGA, IFF, PGM, PPM, BMP.");
      return NULL;
#endif

    case IFF:
      return (Iff::Read(file, options));

    case PGM:
      return (Pgm::Read(file, options));

    case PPM:
      return (Ppm::Read(file, options));

    case BMP:
      return (Bmp::Read(file, options));

    case TIFF:
#ifndef LIBTIFF_MISSING
      return (Tiff::Read(file, options));
#else
      throw POV_EXCEPTION(kCannotOpenFileErr,
"This unofficial POV-Ray binary was built without support for the TIFF \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the libTIFF library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: GIF, TGA, IFF, PGM, PPM, BMP.");
      return NULL;
#endif

    case SYS:
      throw POV_EXCEPTION(kCannotOpenFileErr, "This platform has not defined a SYS file type");
      return (NULL);

    default :
      throw POV_EXCEPTION(kParamErr, "Invalid file type");
      return (NULL);
  }
}

void Image::Write(ImageFileType type, OStream *file, const Image *image, const WriteOptions& options)
{
  if (image->GetWidth() == 0 || image->GetHeight() == 0)
    throw POV_EXCEPTION(kParamErr, "Invalid image size for output");

  if (file == NULL)
    throw POV_EXCEPTION(kCannotOpenFileErr, "Invalid image file");

#ifdef SYS_TO_STANDARD
  if (type == SYS)
    type = SYS_TO_STANDARD ;
#endif

  switch (type)
  {
    case GIF:
    case IFF:
    case PGM:
    case TIFF:
    case POT:
      throw POV_EXCEPTION(kParamErr, "Unsupported file type for output");
      break;

    case SYS:
      throw POV_EXCEPTION(kCannotOpenFileErr, "This platform has not defined a SYS file type");
      break;

    case HDR:
      HDR::Write(file, image, options);
      break;

    case EXR:
#ifndef OPENEXR_MISSING
      OpenEXR::Write(file, image, options);
#else
      throw POV_EXCEPTION(kParamErr,
"This unofficial POV-Ray binary was built without support for the OpenEXR \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the OpenEXR library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: HDR.");
#endif 
      break;

    case PNG:
#ifndef LIBPNG_MISSING
      Png::Write(file, image, options);
#else
      throw POV_EXCEPTION(kParamErr,
"This unofficial POV-Ray binary was built without support for the PNG \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the libPNG library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: TGA, PPM, BMP.");
#endif
      break;

    case TGA:
      Targa::Write(file, image, options);
      break;

    case PPM:
      Ppm::Write(file, image, options);
      break;

    case BMP:
      Bmp::Write(file, image, options);
      break;

    case JPEG:
#ifndef LIBJPEG_MISSING
      Jpeg::Write(file, image, options);
#else
      throw POV_EXCEPTION(kParamErr,
"This unofficial POV-Ray binary was built without support for the JPEG \
file format.  You must either use an official POV-Ray binary or recompile \
the POV-Ray sources on a system providing you with the libJPEG library \
to make use of this facility.  Alternatively, you may use any of the \
following built-in formats: TGA, PPM, BMP.");
#endif
      break;

    default :
      throw POV_EXCEPTION(kParamErr, "Invalid file type");
      break;
  }
}

void Image::GetRGBIndexedValue(unsigned char index, float& red, float& green, float& blue) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			break;
		case RGBColourMap:
		case RGBAColourMap:
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			break;
	}
}

void Image::GetRGBAIndexedValue(unsigned char index, float& red, float& green, float& blue, float& alpha) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			alpha = 0.0f;
			break;
		case RGBColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			alpha = 0.0f;
			break;
		case RGBAColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			alpha = colormap[index].filter;
			break;
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			alpha = 1.0f - colormap[index].transm; // TODO - should filter and transm be converted to alpha? [trf]
			break;
	}
}

void Image::GetRGBFTIndexedValue(unsigned char index, float& red, float& green, float& blue, float& filter, float& transm) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			filter = 0.0f;
			transm = 0.0f;
			break;
		case RGBColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			filter = transm = 0.0f;
			break;
		case RGBAColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			filter = 0.0f; // TODO - should alpha be converted to filter and transm? [trf]
			transm = 1.0f - colormap[index].filter; // note: filter == alpha in RGBA maps
			break;
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			filter = colormap[index].filter;
			transm = colormap[index].transm;
			break;
	}
}

void Image::SetRGBIndexedValue(unsigned char index, float red, float green, float blue)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
		case RGBAColourMap:
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f;
			colormap[index].transm = 0.0f;
			break;
	}
}

void Image::SetRGBAIndexedValue(unsigned char index, float red, float green, float blue, float alpha)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f;
			colormap[index].transm = 0.0f;
			break;
		case RGBAColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = alpha;
			colormap[index].transm = 0.0f;
			break;
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f; // TODO - should alpha be converted to filter and transm? [trf]
			colormap[index].transm = 1.0f - alpha;
			break;
	}
}

void Image::SetRGBFTIndexedValue(unsigned char index, float red, float green, float blue, float filter, float transm)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f;
			colormap[index].transm = 0.0f;
			break;
		case RGBAColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 1.0f - transm; // note: filter is alpha in RGBA maps
			colormap[index].transm = 0.0f; // TODO - should filter and transm be converted to alpha? [trf]
			break;
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = filter;
			colormap[index].transm = transm;
			break;
	}
}

void Image::SetRGBFTIndexedValue(unsigned char index, const Colour& col)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = 0.0f;
			colormap[index].transm = 0.0f;
			break;
		case RGBAColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = 1.0f - col.transm(); // note: filter is alpha in RGBA maps
			colormap[index].transm = 0.0f; // TODO - should filter and transm be converted to alpha? [trf]
			break;
		case RGBFTColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = col.filter();
			colormap[index].transm = col.transm();
			break;
	}
}

unsigned char Image::GetIndexedValue(unsigned int, unsigned int)
{
	return 0;
}

void Image::SetIndexedValue(unsigned int x, unsigned int y, unsigned char index)
{
	CHECK_BOUNDS(x, y);
	switch(colormaptype)
	{
		case NoColourMap:
			SetBitValue(x,y, false);
			break;
		case RGBColourMap:
			SetRGBValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue);
			break;
		case RGBAColourMap:
			SetRGBAValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter);
			break;
		case RGBFTColourMap:
			SetRGBFTValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::FillIndexedValue(unsigned char index)
{
	switch(colormaptype)
	{
		case NoColourMap:
			FillBitValue(false);
			break;
		case RGBColourMap:
			FillRGBValue(colormap[index].red, colormap[index].green, colormap[index].blue);
			break;
		case RGBAColourMap:
			FillRGBAValue(colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter);
			break;
		case RGBFTColourMap:
			FillRGBFTValue(colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::GetColourMap(vector<RGBMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue);
}

void Image::GetColourMap(vector<RGBAMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBAIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue, m[i].alpha);
}

void Image::GetColourMap(vector<RGBFTMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBFTIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue, m[i].filter, m[i].transm);
}

void Image::SetColourMap(const vector<RGBMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBColourMap;
	colormap.assign(m.begin(), m.end());
}

void Image::SetColourMap(const vector<RGBAMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBAColourMap;
	colormap.assign(m.begin(), m.end());
}

void Image::SetColourMap(const vector<RGBFTMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBFTColourMap;
	colormap.assign(m.begin(), m.end());
}

}
