/*******************************************************************************
 * csg.cpp
 *
 * This module implements routines for constructive solid geometry.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/shape/csg.cpp $
 * $Revision: #40 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#include "backend/frame.h"
#include "backend/math/vector.h"
#include "backend/bounding/bbox.h"
#include "backend/shape/csg.h"
#include "backend/math/matrices.h"
#include "backend/scene/objects.h"
#include "backend/shape/quadrics.h"
#include "backend/shape/hfield.h"
#include "backend/scene/threaddata.h"

#include "lightgrp.h" // TODO

#include <algorithm>

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace std;

inline bool Test_Ray_Flags(Ray& ray, ObjectPtr obj)
{
  // CJC 2005 if ray is primary ray ignore NO_IMAGE_FLAG to support the trace() SDL function
  return ((!ray.IsPhotonRay() && (!Test_Flag(obj, NO_IMAGE_FLAG) || ray.IsReflectionRay() == true || ray.IsPrimaryRay() == true) &&
          (!Test_Flag(obj, NO_REFLECTION_FLAG) || ray.IsReflectionRay() == false)) ||
          (ray.IsPhotonRay() && !Test_Flag(obj, NO_SHADOW_FLAG)));
}

inline bool Test_Ray_Flags_Shadow(Ray& ray, ObjectPtr obj)
{
  return ((!ray.IsPhotonRay() && (!Test_Flag(obj, NO_IMAGE_FLAG) || ray.IsReflectionRay() == true) &&
          (!Test_Flag(obj, NO_REFLECTION_FLAG) || ray.IsReflectionRay() == false)) ||
          (ray.IsPhotonRay() && !Test_Flag(obj, NO_SHADOW_FLAG)) || (ray.IsShadowTestRay() &&
          !Test_Flag(obj, NO_SHADOW_FLAG)));
}

/*****************************************************************************
*
* FUNCTION
*
*   All_CSG_Union_Intersections
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Sep 1994 : Added code to count intersection tests. [DB]
*
******************************************************************************/

bool CSGUnion::All_Intersections(Ray& ray, IStack& Depth_Stack, const TraceThreadData *Thread)
{
  int Found;

  Thread->Stats[Ray_CSG_Union_Tests]++;

  Found = false;

  // Use shortcut if no clip.

  if(Clip.empty())
  {
    for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    {
      if(Test_Ray_Flags(ray, (*Current_Sib)))
      {
        if((*Current_Sib)->Bound.empty() == true || Ray_In_Bound(ray, (*Current_Sib)->Bound, Thread))
        {
          if((*Current_Sib)->All_Intersections(ray, Depth_Stack, Thread))
            Found = true;
        }
      }
    }
  }
  else
  {
    IStack Local_Stack(Thread->stackPool);

    for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    {
      if(Test_Ray_Flags(ray, (*Current_Sib)))
      {
        if((*Current_Sib)->Bound.empty() == true || Ray_In_Bound(ray, (*Current_Sib)->Bound, Thread))
        {
          if((*Current_Sib)->All_Intersections (ray, Local_Stack, Thread))
          {
            while(Local_Stack->size() > 0)
            {
              if(Clip.empty() || Point_In_Clip(Local_Stack->top().IPoint, Clip, Thread))
              {
                // if(Test_Flag(this, MULTITEXTURE_FLAG))
                Local_Stack->top().Csg = this;

                Depth_Stack->push(Local_Stack->top());

                Found = true;
              }

              Local_Stack->pop();
            }
          }
        }
      }
    }
  }

  if(Found)
    Thread->Stats[Ray_CSG_Union_Tests_Succeeded]++;

  return (Found);
}



/*****************************************************************************
*
* FUNCTION
*
*   All_CSG_Intersection_Intersections
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Sep 1994 : Added code to count intersection tests. [DB]
*
******************************************************************************/

bool CSGIntersection::All_Intersections(Ray& ray, IStack& Depth_Stack, const TraceThreadData *Thread)
{
  int Maybe_Found, Found;
  IStack Local_Stack(Thread->stackPool);

  Thread->Stats[Ray_CSG_Intersection_Tests]++;

  Found = false;

  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
  {
    if ((*Current_Sib)->Bound.empty() == true || Ray_In_Bound(ray, (*Current_Sib)->Bound, Thread))
    {
      if((*Current_Sib)->All_Intersections(ray, Local_Stack, Thread))
      {
        while(Local_Stack->size() > 0)
        {
          Maybe_Found = true;

          for(vector<ObjectPtr>::iterator Inside_Sib = children.begin(); Inside_Sib != children.end(); Inside_Sib++)
          {
            if(*Inside_Sib != *Current_Sib)
            {
              if(!((*Inside_Sib)->Type & LIGHT_SOURCE_OBJECT) || (!((LightSource *)(*Inside_Sib))->children.empty()))
              {
                if(!Inside_Object(Local_Stack->top().IPoint, *Inside_Sib, Thread))
                {
                  Maybe_Found = false;
                  break;
                }
              }
            }
          }

          if(Maybe_Found)
          {
            if(Clip.empty() || Point_In_Clip(Local_Stack->top().IPoint, Clip, Thread))
            {
              //if(Test_Flag(this, MULTITEXTURE_FLAG))
              Local_Stack->top().Csg = this;

              Depth_Stack->push(Local_Stack->top());

              Found = true;
            }
          }

          Local_Stack->pop();
        }
      }
    }
  }

  if(Found)
    Thread->Stats[Ray_CSG_Intersection_Tests_Succeeded]++;

  return (Found);
}



/*****************************************************************************
*
* FUNCTION
*
*   All_CSG_Merge_Intersections
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Sep 1994 : Added code to count intersection tests. [DB]
*
******************************************************************************/

bool CSGMerge::All_Intersections(Ray& ray, IStack& Depth_Stack, const TraceThreadData *Thread)
{
  int Found;
  bool inside_flag;
  IStack Local_Stack(Thread->stackPool);

  Thread->Stats[Ray_CSG_Merge_Tests]++;

  Found = false;

  // FIXME - though the name is misleading, the OPTIMISE_SHADOW_TEST flag can be used to
  //  determine if we're in a shadow ray, but it SHOULD be renamed.
  // We should probably change Optimization_Flags to a "ray-type" variable, that will tell
  // us if it is primary, reflection, refraction, shadow, primary photon, photon refleciton, or photon refraction ray.
  int shadow_flag = ray.IsShadowTestRay();

  for(vector<ObjectPtr>::iterator Sib1 = children.begin(); Sib1 != children.end(); Sib1++)
  {
    if ( Test_Ray_Flags_Shadow(ray, (*Sib1)) )
    {
      if ((*Sib1)->Bound.empty() == true || Ray_In_Bound (ray, (*Sib1)->Bound, Thread))
      {
        if ((*Sib1)->All_Intersections (ray, Local_Stack, Thread))
        {
          while (Local_Stack->size() > 0)
          {
            if (Clip.empty() || Point_In_Clip (Local_Stack->top().IPoint, Clip, Thread))
            {
              inside_flag = true;

              for(vector<ObjectPtr>::iterator Sib2 = children.begin(); (Sib2 != children.end()) && (inside_flag == true); Sib2++)
              {
                if (*Sib1 != *Sib2)
                {
                  if (!((*Sib2)->Type & LIGHT_SOURCE_OBJECT) || (!((LightSource *)(*Sib2))->children.empty())) 
                  {
                    if ( Test_Ray_Flags_Shadow(ray, (*Sib2)) )
                    {
                      if (Inside_Object(Local_Stack->top().IPoint, *Sib2, Thread))
                        inside_flag = false;
                    }
                  }
                }
              }

              if (inside_flag == true)
              {
                //if (Test_Flag(this, MULTITEXTURE_FLAG))
                Local_Stack->top().Csg = this;

                Found = true;

                Depth_Stack->push(Local_Stack->top());
              }
            }

            Local_Stack->pop();
          }
        }
      }
    }
  }

  if (Found)
    Thread->Stats[Ray_CSG_Merge_Tests_Succeeded]++;

  return (Found);
}



/*****************************************************************************
*
* FUNCTION
*
*   Inside_CSG_Union
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

bool CSGUnion::Inside(VECTOR IPoint, const TraceThreadData *Thread)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
  {
    if(!((*Current_Sib)->Type & LIGHT_SOURCE_OBJECT) || (!((LightSource *)(*Current_Sib))->children.empty()))
    {
      if(Inside_Object(IPoint, *Current_Sib, Thread))
        return (true);
    }
  }

  return (false);
}



/*****************************************************************************
*
* FUNCTION
*
*   Inside_CSG_Intersection
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

bool CSGIntersection::Inside(VECTOR IPoint, const TraceThreadData *Thread)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    if(!((*Current_Sib)->Type & LIGHT_SOURCE_OBJECT) || (!((LightSource *)(*Current_Sib))->children.empty()))
      if(!Inside_Object(IPoint, (*Current_Sib), Thread))
        return (false);
  return (true);
}




/*****************************************************************************
*
* FUNCTION
*
*   Translate_CSG
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/


void CSG::Translate(VECTOR Vector, TRANSFORM *tr)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    Translate_Object (*Current_Sib, Vector, tr) ;

  Recompute_BBox(&BBox, tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_CSG
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/


void CSG::Rotate(VECTOR Vector, TRANSFORM *tr)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    Rotate_Object (*Current_Sib, Vector, tr) ;

  Recompute_BBox(&BBox, tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_CSG
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/


void CSG::Scale(VECTOR Vector, TRANSFORM *tr)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    Scale_Object (*Current_Sib, Vector, tr) ;

  Recompute_BBox(&BBox, tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_CSG
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/


void CSG::Transform(TRANSFORM *tr)
{
  for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
    (*Current_Sib)->Transform(tr);

  Recompute_BBox(&BBox, tr);
}

/*****************************************************************************
*
* FUNCTION
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void CSG::Invert()
{
	for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
		Invert_Object(*Current_Sib);
	Invert_Flag(this, INVERTED_FLAG);
}

/*****************************************************************************
*
* FUNCTION
*
*   Create_CSG_Union
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   2000 : NK phmap
*
******************************************************************************/

CSGUnion::CSGUnion()
{
  INIT_OBJECT_FIELDS(this, UNION_OBJECT)

  do_split = true;
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_CSG_Merge
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

CSGMerge::CSGMerge()
{
  INIT_OBJECT_FIELDS(this, MERGE_OBJECT)
}

/*****************************************************************************
*
* FUNCTION
*
*   Create_CSG_Intersection
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

CSGIntersection::CSGIntersection(bool diff) : isDifference(diff)
{
  INIT_OBJECT_FIELDS(this, INTERSECTION_OBJECT)

  do_split = false; // TODO - not necessary but makes debugging clearer
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_CSG
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/


ObjectPtr CSGUnion::Copy()
{
  CSGUnion *New = new CSGUnion();
  Destroy_Transform(New->Trans);
  *New = *this;

  New->children.clear();
  New->children.reserve(children.size());
  for(vector<ObjectPtr>::iterator i(children.begin()); i != children.end(); i++)
    New->children.push_back(Copy_Object(*i));

  if(Type & LIGHT_GROUP_OBJECT)
  {
    New->LLights.clear();
    Promote_Local_Lights(New);
  }

  return (New);
}

ObjectPtr CSGMerge::Copy()
{
  CSGMerge *New = new CSGMerge();
  Destroy_Transform(New->Trans);
  *New = *this;

  New->children.clear();
  New->children.reserve(children.size());
  for(vector<ObjectPtr>::iterator i(children.begin()); i != children.end(); i++)
    New->children.push_back(Copy_Object(*i));

  if(Type & LIGHT_GROUP_OBJECT)
  {
    New->LLights.clear();
    Promote_Local_Lights(New);
  }

  return (New);
}

ObjectPtr CSGIntersection::Copy()
{
  CSGIntersection *New = new CSGIntersection(false);
  Destroy_Transform(New->Trans);
  *New = *this;

  New->children.clear();
  New->children.reserve(children.size());
  for(vector<ObjectPtr>::iterator i(children.begin()); i != children.end(); i++)
    New->children.push_back(Copy_Object(*i));

  if(Type & LIGHT_GROUP_OBJECT)
  {
    New->LLights.clear();
    Promote_Local_Lights(New);
  }

  return (New);
}

// NB this probably ought to be part of ObjectBase.
static void AssignBaseFields (ObjectPtr Old, ObjectPtr New)
{
  // copy our members to the new object
  New->Texture = Old->Texture;
  New->Interior_Texture = Old->Interior_Texture;
  New->interior = Old->interior;
  New->Bound = Old->Bound;
  New->Clip = Old->Clip;
  New->LLights = Old->LLights;
  New->BBox = Old->BBox;
  New->Trans = Old->Trans;
  New->UV_Trans = Old->UV_Trans;
  New->Ph_Density = Old->Ph_Density;
  New->Flags = Old->Flags;

  // clear these so they don't get released when we are destroyed
  Old->Texture = NULL;
  Old->Interior_Texture = NULL;
  Old->interior = NULL;
  Old->Bound.clear();
  Old->Clip.clear();
  Old->LLights.clear();
  Old->Trans = NULL;
  Old->UV_Trans = NULL;
}

CSG *CSGMerge::Morph(void)
{
  CSGIntersection *New = new CSGIntersection(false);

  AssignBaseFields (this, New) ;
  New->children = children ;
  children.clear() ;

  delete this ;
  return (New);
}

CSG *CSGUnion::Morph(void)
{
  CSGIntersection *New = new CSGIntersection(false);

  AssignBaseFields (this, New) ;
  New->children = children ;
  children.clear() ;

  delete this ;
  return (New);
}

CSG *CSGIntersection::Morph(void)
{
  CSGMerge *New = new CSGMerge();

  AssignBaseFields (this, New) ;
  New->children = children ;
  children.clear() ;

  delete this ;
  return (New);
}

/*****************************************************************************
*
* FUNCTION
*
*   Compute_CSG_BBox
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Sep 1994 : Improved bounding of quadrics used in CSG intersections. [DB]
*
******************************************************************************/

void CSG::Compute_BBox()
{
	DBL Old_Volume, New_Volume;
	VECTOR NewMin, NewMax, TmpMin, TmpMax, Min, Max;

	if(dynamic_cast<CSGIntersection *>(this) != NULL) // FIXME
	{
		if((dynamic_cast<CSGIntersection *>(this)->isDifference == true) &&
		   (!Test_Flag(this, INVERTED_FLAG)) && (!Test_Flag(this, INFINITE_FLAG)) &&
		   (children.empty() == false))
		{
			Make_min_max_from_BBox(NewMin, NewMax, children.front()->BBox);
		}
		else
		{
			/*
			 * Calculate the bounding box of a CSG intersection
			 * by intersecting the bounding boxes of all children.
			 */

			Make_Vector(NewMin, -BOUND_HUGE, -BOUND_HUGE, -BOUND_HUGE);
			Make_Vector(NewMax,  BOUND_HUGE,  BOUND_HUGE,  BOUND_HUGE);

			vector<Quadric *> Quadrics;

			/* Process all children. */

			for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
			{
				/* Inverted objects and height fields mustn't be considered */

				if(!Test_Flag((*Current_Sib), INVERTED_FLAG) && (dynamic_cast<HField *>(*Current_Sib) == NULL)) // FIXME
				{
					/* We store quadrics since they'll be processed last. */
					if(dynamic_cast<Quadric *>(*Current_Sib) == NULL) // FIXME
					{
						if(dynamic_cast<Plane *>(*Current_Sib) != NULL) // FIXME
							Quadric::Compute_Plane_Min_Max((Plane *)(*Current_Sib), TmpMin, TmpMax);
						else
							Make_min_max_from_BBox(TmpMin, TmpMax, (*Current_Sib)->BBox);

						NewMin[X] = max(NewMin[X], TmpMin[X]);
						NewMin[Y] = max(NewMin[Y], TmpMin[Y]);
						NewMin[Z] = max(NewMin[Z], TmpMin[Z]);
						NewMax[X] = min(NewMax[X], TmpMax[X]);
						NewMax[Y] = min(NewMax[Y], TmpMax[Y]);
						NewMax[Z] = min(NewMax[Z], TmpMax[Z]);
					}
					else
						Quadrics.push_back(dynamic_cast<Quadric *>(*Current_Sib));
				}
			}

			/* Process any quadrics. */

			for(vector<Quadric *>::iterator i = Quadrics.begin(); i != Quadrics.end(); i++)
			{
				Quadric *q = *i;

				Assign_Vector(Min, NewMin);
				Assign_Vector(Max, NewMax);

				q->Compute_BBox(Min, Max);

				Make_min_max_from_BBox(TmpMin, TmpMax, q->BBox);

				NewMin[X] = max(NewMin[X], TmpMin[X]);
				NewMin[Y] = max(NewMin[Y], TmpMin[Y]);
				NewMin[Z] = max(NewMin[Z], TmpMin[Z]);
				NewMax[X] = min(NewMax[X], TmpMax[X]);
				NewMax[Y] = min(NewMax[Y], TmpMax[Y]);
				NewMax[Z] = min(NewMax[Z], TmpMax[Z]);
			}
		}
	}
	else
	{
		/* Calculate the bounding box of a CSG merge/union object. */

		Make_Vector(NewMin,  BOUND_HUGE,  BOUND_HUGE,  BOUND_HUGE);
		Make_Vector(NewMax, -BOUND_HUGE, -BOUND_HUGE, -BOUND_HUGE);

		for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
		{
			Make_min_max_from_BBox(TmpMin, TmpMax, (*Current_Sib)->BBox);

			NewMin[X] = min(NewMin[X], TmpMin[X]);
			NewMin[Y] = min(NewMin[Y], TmpMin[Y]);
			NewMin[Z] = min(NewMin[Z], TmpMin[Z]);
			NewMax[X] = max(NewMax[X], TmpMax[X]);
			NewMax[Y] = max(NewMax[Y], TmpMax[Y]);
			NewMax[Z] = max(NewMax[Z], TmpMax[Z]);
		}
	}

	if((NewMin[X] > NewMax[X]) || (NewMin[Y] > NewMax[Y]) || (NewMin[Z] > NewMax[Z]))
		;// TODO MESSAGE    Warning(0, "Degenerate CSG bounding box (not used!).");
	else
	{
		New_Volume = (NewMax[X] - NewMin[X]) * (NewMax[Y] - NewMin[Y]) * (NewMax[Z] - NewMin[Z]);

		BOUNDS_VOLUME(Old_Volume, BBox);

		if(New_Volume < Old_Volume)
		{
			Make_BBox_from_min_max(BBox, NewMin, NewMax);

			/* Beware of bounding boxes too large. */

			if((BBox.Lengths[X] > CRITICAL_LENGTH) ||
			   (BBox.Lengths[Y] > CRITICAL_LENGTH) ||
			   (BBox.Lengths[Z] > CRITICAL_LENGTH))
				Make_BBox(BBox, -BOUND_HUGE/2, -BOUND_HUGE/2, -BOUND_HUGE/2, BOUND_HUGE, BOUND_HUGE, BOUND_HUGE);
		}
	}
}


/*****************************************************************************
*
* FUNCTION
*
*   Determine_CSG_Textures
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void CSG::Determine_Textures(Intersection *isect, bool hitinside, WeightedTextureVector& textures, const TraceThreadData *threaddata)
{
	// For CSG Differences, use only the first object in the chain
	// (which is the first object in the POV file.  All other objects
	// are the ones that were "removed" from the first one, so their
	// textures should NOT be used.
	if(!children.empty())
	{
		if(Type & CSG_DIFFERENCE_OBJECT)
		{
			if(children[0]->Inside(isect->IPoint, threaddata))
			{
				if(children[0]->Type & IS_COMPOUND_OBJECT)
					children[0]->Determine_Textures(isect, hitinside, textures, threaddata);
				else if(children[0]->Texture != NULL)
					textures.push_back(WeightedTexture(1.0, isect->Object->Texture));
			}
		}
		else
		{
			size_t firstinserted = textures.size();

			for(vector<ObjectPtr>::iterator Current_Sib = children.begin(); Current_Sib != children.end(); Current_Sib++)
			{
				if((*Current_Sib)->Inside(isect->IPoint, threaddata))
				{
					if((*Current_Sib)->Type & IS_COMPOUND_OBJECT)
						(*Current_Sib)->Determine_Textures(isect, hitinside, textures, threaddata);
					else if((*Current_Sib)->Texture != NULL)
						textures.push_back(WeightedTexture(1.0, isect->Object->Texture));
				}
			}

			COLC weight = 1.0f / min(COLC(textures.size() - firstinserted), 1.0f);

			for(size_t i = firstinserted; i < textures.size(); i++)
				textures[i].weight = weight;
		}
	}	
}

}
