/*******************************************************************************
 * pvmem.cpp
 *
 * This module implements windows-specific memory handling routines.
 *
 * Author: Christopher J. Cason.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/windows/pvmem.cpp $
 * $Revision: #9 $
 * $Change: 4537 $
 * $DateTime: 2008/02/09 01:25:49 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#define POVWIN_FILE
#define _WIN32_IE COMMONCTRL_VERSION

#define WIN32_LEAN_AND_MEAN

#include <windows.h>
#include <crtdbg.h>
#include <errno.h>
#include <assert.h>

#include "pvengine.h"

#define WIN_MEM_TRACKING          1

#ifdef _DEBUG
  #define EXTRA_VALIDATION        1
#else
  #define EXTRA_VALIDATION        0
#endif

#include "pvmem.h"

// this must be the last file included
#include "syspovdebug.h"

extern "C"
{
  int __cdecl WinMainCRTStartup();
}

#undef new
#undef delete

using namespace pov_base;

namespace povwin
{

static WinHeap *gHeap = NULL;
WinHeap *WinMemBlock::m_Heap = NULL;
static WinMemStats gMemStats;
__declspec(thread) static WinMemStats *lMemStats = NULL;

#if POV_MEM_STATS == 0 && WIN_MEM_TRACKING == 0

int __cdecl POVWINStartup()
{
  return WinMainCRTStartup();
}

void *win_malloc(size_t size)
{
  return malloc(size);
}

void *win_malloc(size_t size, const void *data, int line)
{
  return malloc(size);
}

void *win_calloc(size_t nelem, size_t size)
{
  return calloc(nelem, size);
}

void *win_calloc(size_t nelem, size_t size, const void *data, int line)
{
  return calloc(nelem, size);
}

void *win_realloc(void *p, size_t size)
{
  return realloc(p, size);
}

void *win_realloc(void *p, size_t size, const void *data, int line)
{
  return realloc(p, size);
}

void win_free(void *p)
{
  free(p);
}

void win_free(void *p, const void *data, int line)
{
  free(p);
}

char *win_strdup(const char *s)
{
  return _strdup(s);
}

char *win_strdup(const char *s, const void *data, int line)
{
  return _strdup(s);
}

void WinMemStage(bool BeginRender, void *cookie)
{
}

#else //

// This function is the first called when POVWIN starts (even before the CRT startup code).
// Therefore we can't count on very much being set up for us.
int __cdecl POVWINStartup()
{
  // note that WinHeap has its own operator new.
  gHeap = new WinHeap;
  if (gHeap == NULL || gHeap->Validate() == false)
  {
    // not safe to call MessageBox here due to some other threads getting fired up
    // MessageBox(GetDesktopWindow(), "Failed to set up private heap - POV-Ray will now exit", "POV-Ray for Windows", MB_OK | MB_ICONERROR);
    return 1;
  }
  WinMemBlock::SetAllocator(gHeap);

  // we never delete gHeap since WinMainCRTStartup() often exits via means other than a return.
  return WinMainCRTStartup();
}

void WinMemThreadStartup()
{
  lMemStats = new WinMemStats();
}

void WinMemThreadCleanup()
{
  delete lMemStats;
}

bool WinMemReport(bool global, uint64& allocs, uint64& frees, int64& current, uint64& peak, uint64 &smallest, uint64 &largest)
{
  HeapLock lock(gHeap);

  allocs = frees = current = peak = smallest = largest = 0;
  if (global)
  {
    gMemStats.Report(allocs, frees, current, peak, smallest, largest);
    return true;
  }
  if (lMemStats == NULL)
    return false;
  lMemStats->Report(allocs, frees, current, peak, smallest, largest);
  return true;
}

void WinMemStage(bool BeginRender, void *cookie)
{
  HeapLock lock(gHeap);

  if (BeginRender)
    gMemStats.Clear();
}

#ifdef _DEBUG

void *win_malloc(size_t size, const void *data, int line)
{
  HeapLock lock(gHeap);

  // note that WinMemBlock has its own operator new
  WinMemBlock *block = new (size) WinMemBlock(data, line);
  if (block == NULL)
    return NULL;
  gMemStats.RecordAlloc(block->Size(size));
  return block->GetMem();
}

void *win_realloc(void *p, size_t newSize, const void *data, int line)
{
  if (p == NULL)
    return win_malloc(newSize, data, line);

  if (newSize != 0)
  {
    HeapLock lock(gHeap);
    size_t oldSize;
    WinMemBlock *block = WinMemBlock::GetBlock(p);
    if ((block == NULL) || ((oldSize = block->Size()) == (size_t) -1))
      throw POV_EXCEPTION(kCannotHandleDataErr, "Bad memory block in realloc()");
    if ((block = block->Realloc(newSize, data, line)) == NULL)
      return NULL;
    gMemStats.RecordFree(oldSize);
    gMemStats.RecordAlloc(block->Size(newSize));
    return block->GetMem();
  }
  else
  {
    win_free(p, data, line);
    return NULL;
  }
}

void win_free(void *p, const void *data, int line)
{
  if (p == NULL)
    return;
  HeapLock lock(gHeap);
  size_t size;
  WinMemBlock *block = WinMemBlock::GetBlock(p);
  if ((block == NULL) || ((size = block->Size()) == (size_t) -1))
    throw POV_EXCEPTION(kCannotHandleDataErr, "Bad memory block in free()");
  gMemStats.RecordFree(size);
  // note that WinMemBlock has a private delete operator
  delete block;
}

void *win_calloc(size_t nelem, size_t size, const void *data, int line)
{
  size_t count = size * nelem;
  void *p = win_malloc(count, data, line);
  if (p == NULL)
    return NULL;
  memset(p, 0, count);
  return p;
}

char *win_strdup(const char *s, const void *data, int line)
{
  char *p = (char *) win_malloc(strlen(s) + 1, data, line);
  strcpy(p, s);
  return p;
}

#else // _DEBUG

// release-mode functions

void *win_malloc(size_t size)
{
  HeapLock    lock(gHeap);

  void *p = gHeap->Alloc(size);
  if (p != NULL)
    gMemStats.RecordAlloc(size);
  return p;
}

void *win_realloc(void *p, size_t size)
{
  if (p == NULL)
    return win_malloc(size);
  if (size == 0)
  {
    win_free(p);
    return NULL;
  }
  HeapLock lock(gHeap);
  size_t currentSize = gHeap->BlockSize(p);
  if (currentSize == (size_t) -1)
    throw POV_EXCEPTION(kCannotHandleDataErr, "Bad memory block in realloc()");
  void *newBlock = gHeap->ReAlloc(p, size);
  if (newBlock == NULL)
  {
    // this can happen with the low-fragmentation heap, even if memory is available
    // as long as it's a shrink, we return the original pointer
    if (size < currentSize)
      return p;
    return NULL;
  }
  gMemStats.RecordFree(currentSize);
  gMemStats.RecordAlloc(size);
  return newBlock;
}

void win_free(void *p)
{
  if (p == NULL)
    return;
  HeapLock lock(gHeap);
  size_t len = gHeap->BlockSize(p);
  if (len == (size_t) -1)
    throw POV_EXCEPTION(kCannotHandleDataErr, "Bad memory block in free()");
  gMemStats.RecordFree(len);
  if (!gHeap->Free(p))
    throw POV_EXCEPTION(kCannotHandleDataErr, "Bad memory block in free()");
}

void *win_calloc(size_t nelem, size_t size)
{
  size_t count = size * nelem;
  void *p = win_malloc(count);
  if (p == NULL)
    return NULL;
  memset(p, 0, count);
  return p;
}

char *win_strdup(const char *s)
{
  int len = strlen(s) + 1;
  char *p = (char *) win_malloc(len);
  memcpy(p, s, len);
  return p;
}

#endif // else portion of '#ifdef _DEBUG'

#endif // else portion of '#if POV_MEM_STATS == 0 && WIN_MEM_TRACKING == 0'
} // end of namespace povwin

#if WIN_MEM_TRACKING

using namespace povwin;

#ifdef _DEBUG
  #define DEBUGPARAMS   , data, line
#else
  #define DEBUGPARAMS
#endif

void *operator new (size_t size, const char *func, const void *data, int line)
{
  void *res;

  if ((res = win_malloc(size DEBUGPARAMS)) != NULL)
    return res;
  throw std::bad_alloc () ;
}

void *operator new[] (size_t size, const char *func, const void *data, int line)
{
  void *res;

  if ((res = win_malloc(size DEBUGPARAMS)) != NULL)
    return res;
  throw std::bad_alloc () ;
}

void operator delete (void *p, const char *func, const void *data, int line)
{
  win_free(p DEBUGPARAMS);
}

void operator delete[] (void *p, const char *func, const void *data, int line)
{
  win_free(p DEBUGPARAMS);
}

#ifdef _DEBUG
  #undef DEBUGPARAMS
  #define DEBUGPARAMS   , ReturnAddress(), -1
#endif

void *operator new (size_t size)
{
  void *res;

  if ((res = win_malloc(size DEBUGPARAMS)) != NULL)
    return res;
  throw std::bad_alloc () ;
}

void *operator new[] (size_t size)
{
  void *res;

  if ((res = win_malloc(size DEBUGPARAMS)) != NULL)
    return res;
  throw std::bad_alloc () ;
}

void operator delete (void *p) throw ()
{
  win_free(p DEBUGPARAMS);
}

void operator delete[] (void *p) throw ()
{
  win_free(p DEBUGPARAMS);
}

#endif // #if WIN_MEM_TRACKING
