/*******************************************************************************
 * vfe.cpp
 *
 * This module contains the C++ implementation for the virtual frontend.
 *
 * Author: Christopher J. Cason
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 2005-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/vfe/vfe.cpp $
 * $Revision: #23 $
 * $Change: 4546 $
 * $DateTime: 2008/02/09 22:23:03 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#ifdef _MSC_VER
#pragma warning(disable : 4244)
#pragma warning(disable : 4267)
#endif

#include "frame.h"
#include "povray.h"
#include "vfe.h"

// this must be the last file included
#include "base/povdebug.h"

namespace vfe
{

using namespace std;
using namespace pov_base;

////////////////////////////////////////////////////////////////////////////////////////
//
// class POVMSMessageDetails
//
////////////////////////////////////////////////////////////////////////////////////////

class POVMSMessageDetails
{
  public:
    POVMSMessageDetails (POVMS_Object &Obj);
    virtual ~POVMSMessageDetails () {} ;
    string GetContext (int NumLines) ;

  protected:
    string File ;
    UCS2String UCS2File;
    string URL ;
    string Message ;
    POVMSInt Line ;
    POVMSInt Col ;
    POVMSLong Offset ;
} ;

POVMSMessageDetails::POVMSMessageDetails (POVMS_Object& Obj)
{
  char                  buffer [2048] = "";
  UCS2                  ubuffer [2048];
  POVMSInt              l = sizeof (ubuffer);
  POVMSLong             ll ;
  POVMSObject           msgobj (Obj());
  POVMSObjectPtr        msg = &msgobj;

  Line = Col = 0 ;
  Offset = -1 ;
  ubuffer[0] = 0 ;

  if (POVMSUtil_GetUCS2String (msg, kPOVAttrib_FileName, ubuffer, &l) == kNoErr)
  {
    UCS2File = ubuffer ;
    File = UCS2toASCIIString (UCS2File);
  }
  if (POVMSUtil_GetLong (msg, kPOVAttrib_Line, &ll) == kNoErr)
    Line = POVMSInt(ll) ;
  if (POVMSUtil_GetLong (msg, kPOVAttrib_Column, &ll) == kNoErr)
    Col = POVMSInt(ll + 1) ;
  if(POVMSUtil_GetLong(msg, kPOVAttrib_FilePosition, &ll) == kNoErr)
    Offset = ll ;
  l = sizeof (buffer) ;
  if (POVMSUtil_GetString (msg, kPOVAttrib_EnglishText, buffer, &l) == kNoErr)
    Message = buffer ;

  POVMSObject_Delete(msg);
}

string POVMSMessageDetails::GetContext (int NumLines)
{
  return ("") ;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class ParseErrorDetails, ParseWarningDetails
//
////////////////////////////////////////////////////////////////////////////////////////

class ParseWarningDetails : public POVMSMessageDetails
{
  public:
    ParseWarningDetails (POVMS_Object &Obj) : POVMSMessageDetails (Obj) {} ;
    virtual ~ParseWarningDetails () {} ;

  public:
    POVMSMessageDetails::File ;
    POVMSMessageDetails::UCS2File ;
    POVMSMessageDetails::Message ;
    POVMSMessageDetails::Line ;
    POVMSMessageDetails::Col ;
    POVMSMessageDetails::Offset ;
} ;

class ParseErrorDetails : public POVMSMessageDetails
{
  public:
    ParseErrorDetails (POVMS_Object &Obj) : POVMSMessageDetails (Obj) {} ;
    virtual ~ParseErrorDetails () {} ;

  public:
    POVMSMessageDetails::File ;
    POVMSMessageDetails::UCS2File ;
    POVMSMessageDetails::Message ;
    POVMSMessageDetails::Line ;
    POVMSMessageDetails::Col ;
    POVMSMessageDetails::Offset ;
} ;

////////////////////////////////////////////////////////////////////////////////////////
//
// class vfeConsole
//
////////////////////////////////////////////////////////////////////////////////////////

vfeConsole::vfeConsole(vfeSession *session, int width) : Console(width == -1 ? session->GetConsoleWidth() : width)
{
  m_Session = session;
  Initialise();
}

vfeConsole::~vfeConsole()
{
}

void vfeConsole::Initialise()
{
  rawBuffer [0] = '\0' ;
}

void vfeConsole::BufferOutput(const char *str, unsigned int chars, vfeSession::MessageType mType)
{
  char                  *s ;

  // HACK FIXME - this is to prevent duplicate messages
  if (m_Session->HadErrorMessage() && strncmp (str, "Fatal error in parser: ", 23) == 0)
    return ;

  if (str [0] == '\n' && str [1] == '\0')
  {
    m_Session->AppendStreamMessage (mType, rawBuffer) ;
    rawBuffer [0] = '\0' ;
    return ;
  }

  size_t sLen = chars ;
  if (sLen == 0)
    sLen = strlen (str) ;
  size_t bLen = strlen (rawBuffer) ;
  if (sLen > sizeof (rawBuffer) - bLen - 1)
    sLen = sizeof (rawBuffer) - bLen - 1 ;
  strncat (rawBuffer, str, sLen) ;
  if ((s = strrchr (rawBuffer, '\n')) != NULL)
  {
    *s++ = '\0' ;
    m_Session->AppendStreamMessage (mType, rawBuffer) ;
    strcpy (rawBuffer, s) ;
  }
}

void vfeConsole::Output(const char *str, vfeSession::MessageType mType)
{
  BufferOutput (str, (unsigned int) strlen (str), mType) ;
  BufferOutput ("\n", 1, mType) ;
}

void vfeConsole::Output(const string& str, vfeSession::MessageType mType)
{
  Output (str.c_str(), mType) ;
}

void vfeConsole::Output(const string& str)
{
  Output (str.c_str()) ;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class vfePlatformBase
//
////////////////////////////////////////////////////////////////////////////////////////

vfePlatformBase::vfePlatformBase(vfeSession& session) : m_Session(&session), PlatformBase()
{
}

vfePlatformBase::~vfePlatformBase()
{
}

IStream *vfePlatformBase::CreateIStream(const unsigned int stype)
{
  return (new IStream (stype)) ;
}

OStream *vfePlatformBase::CreateOStream(const unsigned int stype)
{
  return (new OStream (stype)) ;
}

UCS2String vfePlatformBase::GetTemporaryPath(void)
{
  return m_Session->GetTemporaryPath();
}

UCS2String vfePlatformBase::CreateTemporaryFile(void)
{
  return m_Session->CreateTemporaryFile();
}

void vfePlatformBase::DeleteTemporaryFile(const UCS2String& filename)
{
  m_Session->DeleteTemporaryFile(filename);
}

bool vfePlatformBase::ReadFileFromURL(OStream *file, const UCS2String& url, const UCS2String& referrer)
{
  return false;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class vfeParserMessageHandler
//
////////////////////////////////////////////////////////////////////////////////////////

vfeParserMessageHandler::vfeParserMessageHandler() : ParserMessageHandler()
{
  m_Session = vfeSession::GetSessionFromThreadID();
}

vfeParserMessageHandler::~vfeParserMessageHandler()
{
}

void vfeParserMessageHandler::Options(Console *Con, POVMS_Object& Obj, bool conout)
{
  if (Obj.TryGetBool (kPOVAttrib_OutputAlpha, false))
    m_Session->SetUsingAlpha();
  if (Obj.TryGetBool (kPOVAttrib_ClocklessAnimation, false))
    m_Session->SetClocklessAnimation();
  if (Obj.TryGetBool (kPOVAttrib_RealTimeRaytracing, false))
    m_Session->SetRealTimeRaytracing();
  ParserMessageHandler::Options (Con, Obj, conout) ;
}

void vfeParserMessageHandler::Statistics(Console *Con, POVMS_Object& Obj, bool conout)
{
  ParserMessageHandler::Statistics (Con, Obj, conout) ;
}

void vfeParserMessageHandler::Progress(Console *Con, POVMS_Object& Obj, bool verbose)
{
  switch(Obj.GetType(kPOVMSObjectClassID))
  {
    case kPOVObjectClass_ParserProgress:
    {
      m_Session->AppendStatusMessage (format ("Parsing %uK tokens") % (Obj.GetLong (kPOVAttrib_CurrentTokenCount) / 1000));
      break;
    }
    case kPOVObjectClass_BoundingProgress:
    {
      m_Session->AppendStatusMessage (format ("Constructed %uK BSP nodes") % (Obj.GetLong (kPOVAttrib_CurrentNodeCount) / 1000));
      break;
    }
  }
}

void vfeParserMessageHandler::Warning(Console *Con, POVMS_Object& Obj, bool conout)
{
  ParseWarningDetails   d (Obj) ;

  if (d.Message == "")
    return ;

  if (!d.File.empty() && (d.Line > 0))
  {
    format f = format ("File '%s' line %d: %s") % d.File % d.Line % d.Message ;
    if (m_Session->m_OptimizeForConsoleOutput == false)
      m_Session->AppendStatusMessage (f) ;
    if (conout)
    {
      if (m_Session->m_OptimizeForConsoleOutput == true)
        Con->puts (f.str().c_str()) ;
      else
        Con->Output (f.str()) ;
    }
  }
  else
    if (conout)
      Con->Output (d.Message) ;
}

void vfeParserMessageHandler::Error(Console *Con, POVMS_Object& Obj, bool conout)
{
  ParseErrorDetails     d (Obj) ;

  if (d.Message == "" && (d.File == "" || d.Line <= 0))
    return ;

  // as we provide special treatment for parser errors here if we're not
  // optimized for console output, we don't duplicate them to the console
  // regardless of whether or not conout is set.
  if (m_Session->m_OptimizeForConsoleOutput == false)
    m_Session->AppendErrorMessage (d.Message, d.UCS2File, d.Line, d.Col) ;

  if (!d.Message.empty())
  {
    if (!d.File.empty() && (d.Line > 0))
    {
      format f = format ("File '%s' line %d: %s") % d.File % d.Line % d.Message ;
      if (m_Session->m_OptimizeForConsoleOutput == false)
        m_Session->AppendStatusMessage (f) ;
      if (conout)
        if (m_Session->m_OptimizeForConsoleOutput == true)
          Con->puts (f.str().c_str()) ;
    }
    else
    {
      if (m_Session->m_OptimizeForConsoleOutput == false)
        m_Session->AppendStatusMessage (d.Message) ;
      if (conout)
        if (m_Session->m_OptimizeForConsoleOutput == true)
          Con->puts (d.Message.c_str()) ;
    }
  }
  else
  {
    format f = format ("Parse error in file '%s' at line %d") % d.File % d.Line ;
    if (m_Session->m_OptimizeForConsoleOutput == false)
      m_Session->AppendStatusMessage (f) ;
    if (conout)
      if (m_Session->m_OptimizeForConsoleOutput == true)
        Con->puts (f.str().c_str()) ;
  }
}

void vfeParserMessageHandler::FatalError(Console *Con, POVMS_Object& Obj, bool conout)
{
  m_Session->SetFailed();
  Error (Con, Obj, conout) ;
}

void vfeParserMessageHandler::DebugInfo(Console *Con, POVMS_Object& Obj, bool conout)
{
  string str(Obj.GetString(kPOVAttrib_EnglishText));
  if (m_Session->m_OptimizeForConsoleOutput == true)
  {
    if (conout)
        Con->puts (str.c_str()) ;
  }
  else
    m_Session->AppendStreamMessage (vfeSession::mDebug, str.c_str()) ;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class vfeRenderMessageHandler
//
////////////////////////////////////////////////////////////////////////////////////////

vfeRenderMessageHandler::vfeRenderMessageHandler() : RenderMessageHandler()
{
  m_Session = vfeSession::GetSessionFromThreadID();
}

vfeRenderMessageHandler::~vfeRenderMessageHandler()
{
}

void vfeRenderMessageHandler::Options(Console *Con, POVMS_Object& Obj, bool conout)
{
  RenderMessageHandler::Options (Con, Obj, conout) ;
}

void vfeRenderMessageHandler::Statistics(Console *Con, POVMS_Object& Obj, bool conout)
{
  RenderMessageHandler::Statistics (Con, Obj, conout) ;
}

void vfeRenderMessageHandler::Progress(Console *Con, POVMS_Object& Obj, bool verbose)
{
  switch (Obj.GetType(kPOVMSObjectClassID))
  {
    case kPOVObjectClass_PhotonProgress:
    {
      int cpc (Obj.GetInt (kPOVAttrib_CurrentPhotonCount)) ;
      m_Session->AppendStatusMessage (format ("Photon count %u") % cpc, 250) ;
      break;
    }
    case kPOVObjectClass_RadiosityProgress:
    {
      int pc (Obj.GetInt (kPOVAttrib_Pixels)) ;
      int cc (Obj.GetInt (kPOVAttrib_PixelsCompleted)) ;
      m_Session->SetPixelsRendered(cc, pc);
      int percent = pc > 0 ? int ((cc * 100.0) / pc) : 0 ;
      m_Session->SetPercentComplete (percent);
      m_Session->AppendStatusMessage (format ("Performing radiosity pretrace: %d of %d pixels (%d%%)") % cc % pc % percent, 250) ;
      break;
    }
    case kPOVObjectClass_RenderProgress:
    {
      int pc (Obj.GetInt (kPOVAttrib_Pixels)) ;
      int cc (Obj.GetInt (kPOVAttrib_PixelsCompleted)) ;

      if (m_Session->GetRealTimeRaytracing() == false)
      {
        m_Session->SetPixelsRendered(cc, pc);
        int percent = pc > 0 ? (int) ((cc * 100.0) / pc) : 0 ;
        m_Session->SetPercentComplete (percent);
        m_Session->AppendStatusMessage (format ("Rendered %u of %u pixels (%d%%)") % cc % pc % percent, 250) ;
      }
      else
      {
        m_Session->SetPixelsRendered(cc % pc, pc);
        float elapsed = m_Session->GetElapsedTime() / 1000.0f;
        float frames = (float) cc / pc;
        float fps = frames / elapsed;
        m_Session->AppendStatusMessage (format ("Rendered %g frames over %g seconds (%g FPS)") % frames % elapsed % fps, 250) ;
      }
      break;
    }
  }
}

void vfeRenderMessageHandler::Warning(Console *Con, POVMS_Object& Obj, bool conout)
{
  RenderMessageHandler::Warning (Con, Obj, conout) ;
}

void vfeRenderMessageHandler::Error(Console *Con, POVMS_Object& Obj, bool conout)
{
  m_Session->SetFailed();
  RenderMessageHandler::Error (Con, Obj, conout) ;
}

void vfeRenderMessageHandler::FatalError(Console *Con, POVMS_Object& Obj, bool conout)
{
  m_Session->SetFailed();
  RenderMessageHandler::FatalError (Con, Obj, conout) ;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class vfeProcessRenderOptions
//
////////////////////////////////////////////////////////////////////////////////////////

vfeProcessRenderOptions::vfeProcessRenderOptions(vfeSession *Session) : ProcessRenderOptions(), m_Session(Session)
{
}

vfeProcessRenderOptions::~vfeProcessRenderOptions()
{
}

int vfeProcessRenderOptions::ReadSpecialOptionHandler(INI_Parser_Table *Table, char *Param, POVMSObjectPtr Obj)
{
  return ProcessRenderOptions::ReadSpecialOptionHandler (Table, Param, Obj);
}

int vfeProcessRenderOptions::ReadSpecialSwitchHandler(Cmd_Parser_Table *Table, char *Param, POVMSObjectPtr Obj, bool On)
{
  return ProcessRenderOptions::ReadSpecialSwitchHandler (Table, Param, Obj, On);
}

int vfeProcessRenderOptions::WriteSpecialOptionHandler(INI_Parser_Table *Table, POVMSObjectPtr Obj, OTextStream *S)
{
  return ProcessRenderOptions::WriteSpecialOptionHandler (Table, Obj, S);
}

bool vfeProcessRenderOptions::WriteOptionFilter(INI_Parser_Table *Table)
{
  return ProcessRenderOptions::WriteOptionFilter (Table);
}

int vfeProcessRenderOptions::ProcessUnknownString(char *String, POVMSObjectPtr Obj)
{
  return ProcessRenderOptions::ProcessUnknownString (String, Obj);
}

ITextStream *vfeProcessRenderOptions::OpenFileForRead(const char *Name, POVMSObjectPtr Obj)
{
  return (ProcessRenderOptions::OpenFileForRead (Name, Obj)) ;
}

OTextStream *vfeProcessRenderOptions::OpenFileForWrite(const char *Name, POVMSObjectPtr Obj)
{
  return (ProcessRenderOptions::OpenFileForWrite (Name, Obj)) ;
}

void vfeProcessRenderOptions::ParseError(const char *format, ...)
{
  char str[1024];
  va_list marker;

  va_start(marker, format);
  vsnprintf(str, sizeof(str)-2, format, marker);
  va_end(marker);

  m_Session->AppendStatusMessage (str);
  m_Session->AppendErrorMessage (str) ;
  m_Session->SetFailed();
}

void vfeProcessRenderOptions::ParseErrorAt(ITextStream *file, const char *format, ...)
{
  char str[1024];
  va_list marker;

  va_start(marker, format);
  vsnprintf(str, sizeof(str)-2, format, marker);
  va_end(marker);

  m_Session->AppendStatusMessage (str);
  m_Session->AppendErrorMessage (str, file->name(), file->line(), 0) ;
  m_Session->SetFailed();
}

void vfeProcessRenderOptions::WriteError(const char *format, ...)
{
  char str[1024];
  va_list marker;

  va_start(marker, format);
  vsnprintf(str, sizeof(str)-2, format, marker);
  va_end(marker);

  m_Session->AppendStatusMessage (str);
  m_Session->AppendErrorMessage (str) ;
  m_Session->SetFailed();
}

////////////////////////////////////////////////////////////////////////////////////////
//
// class VirtualFrontEnd
//
////////////////////////////////////////////////////////////////////////////////////////

VirtualFrontEnd::VirtualFrontEnd(vfeSession& session, POVMSContext ctx, POVMSAddress addr, POVMS_Object& msg, POVMS_Object *result, shared_ptr<Console>& console) :
  m_Session(&session), m_PlatformBase(session), renderFrontend (ctx)
{
  backendAddress = addr ;
  state = kReady ;
  consoleResult = NULL ;
  displayResult = NULL ;
  m_PausedAfterFrame = false;
  renderFrontend.ConnectToBackend(backendAddress, msg, result, console);
}

VirtualFrontEnd::~VirtualFrontEnd()
{
  renderFrontend.DisconnectFromBackend(backendAddress);
  state = kUnknown;
}

bool VirtualFrontEnd::Start(POVMS_Object& opts)
{
  boost::xtime t;

  if(state != kReady)
    return false;

  m_Session->Clear();
  animationProcessing.reset() ;
  m_PausedAfterFrame = false;

  POVMS_List declares;
  if(opts.Exist(kPOVAttrib_Declare) == true)
    opts.Get(kPOVAttrib_Declare, declares);

  POVMS_Object image_width(kPOVMSType_WildCard);
  image_width.SetString(kPOVAttrib_Identifier, "image_width");
  image_width.SetFloat(kPOVAttrib_Value, opts.TryGetInt(kPOVAttrib_Width, 160));
  declares.Append(image_width);

  POVMS_Object image_height(kPOVMSType_WildCard);
  image_height.SetString(kPOVAttrib_Identifier, "image_height");
  image_height.SetFloat(kPOVAttrib_Value, opts.TryGetInt(kPOVAttrib_Height, 120));
  declares.Append(image_height);

  int initialFrame = opts.TryGetInt (kPOVAttrib_InitialFrame, 0) ;
  int finalFrame = opts.TryGetInt (kPOVAttrib_FinalFrame, 0) ;
  if ((initialFrame == 0 && finalFrame == 0) || (initialFrame == 1 && finalFrame == 1))
  {
    POVMS_Object clock_delta(kPOVMSType_WildCard);
    clock_delta.SetString(kPOVAttrib_Identifier, "clock_delta");
    clock_delta.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(clock_delta);

    POVMS_Object final_clock(kPOVMSType_WildCard);
    final_clock.SetString(kPOVAttrib_Identifier, "final_clock");
    final_clock.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(final_clock);

    POVMS_Object final_frame(kPOVMSType_WildCard);
    final_frame.SetString(kPOVAttrib_Identifier, "final_frame");
    final_frame.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(final_frame);

    POVMS_Object frame_number(kPOVMSType_WildCard);
    frame_number.SetString(kPOVAttrib_Identifier, "frame_number");
    frame_number.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(frame_number);

    POVMS_Object initial_clock(kPOVMSType_WildCard);
    initial_clock.SetString(kPOVAttrib_Identifier, "initial_clock");
    initial_clock.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(initial_clock);

    POVMS_Object initial_frame(kPOVMSType_WildCard);
    initial_frame.SetString(kPOVAttrib_Identifier, "initial_frame");
    initial_frame.SetFloat(kPOVAttrib_Value, 0.0f);
    declares.Append(initial_frame);

    opts.Set(kPOVAttrib_Declare, declares);
    imageProcessing = shared_ptr<ImageProcessing> (new ImageProcessing (opts)) ;
    options = opts;

    if (m_Session->OutputToFileSet())
    {
      UCS2String filename = imageProcessing->GetOutputFilename (opts, 0, 0);
      options.SetUCS2String (kPOVAttrib_OutputFile, filename.c_str());

      // test access permission now to avoid surprise later after waiting for
      // the render to complete.
      if (m_Session->TestAccessAllowed(filename, true) == false)
      {
        string str ("IO Restrictions prohibit write access to '") ;
        str += UCS2toASCIIString(filename);
        str += "'";
        throw POV_EXCEPTION(kCannotOpenFileErr, str);
      }
      m_Session->AdviseOutputFilename (filename);
    }
  }
  else
  {
    // the output filename is set in Process()
    m_Session->SetRenderingAnimation();
    opts.Set(kPOVAttrib_Declare, declares);
    imageProcessing = shared_ptr<ImageProcessing> (new ImageProcessing (opts)) ;
    animationProcessing = shared_ptr<AnimationProcessing> (new AnimationProcessing (opts)) ;
    options = animationProcessing->GetFrameRenderOptions () ;
  }

  // this code may not be removed or modified unless the compilation is for personal
  // use and the binary is not distributed to anyone else.
  xtime_get(&t, TIME_UTC);
  if (t.sec > 1230760800)
    throw POV_EXCEPTION(kCannotHandleRequestErr, "Please obtain an updated version of this application.");

  state = kStarting;

  return true;
}

bool VirtualFrontEnd::Stop()
{
  switch(state)
  {
    case kStarting:
      state = kStopped;
      m_Session->SetFailed();
      return true;

    case kParsing:
    case kPausedParsing:
      renderFrontend.StopParser(sceneId);
      m_Session->SetFailed();
      state = kStopping;
      return true;

    case kRendering:
    case kPausedRendering:
      m_Session->SetFailed();
      if (m_PausedAfterFrame == true)
      {
        m_PausedAfterFrame = false;
        state = kStopped;
      }
      else
      {
        renderFrontend.StopRender(viewId);
        state = kStopping;
      }
      return true;
  }

  return false;
}

bool VirtualFrontEnd::Pause()
{
  try
  {
    switch(state)
    {
      case kParsing:
        renderFrontend.PauseParser(sceneId);
        state = kPausedParsing;
        return true;

      case kRendering:
        renderFrontend.PauseRender(viewId);
        state = kPausedRendering;
        return true;
    }
  }
  catch (pov_base::Exception&)
  {
    return false;
  }
  return false;
}

bool VirtualFrontEnd::Resume()
{
  try
  {
    switch(state)
    {
      case kPausedParsing:
        renderFrontend.ResumeParser(sceneId);
        state = kParsing;
        return true;

      case kPausedRendering:
        if (m_PausedAfterFrame)
        {
          state = kStarting;
          m_PausedAfterFrame = false;
          return true;
        }
        renderFrontend.ResumeRender(viewId);
        state = kRendering;
        return true;
    }
  }
  catch (pov_base::Exception&)
  {
    return (false) ;
  }
  return false;
}

State VirtualFrontEnd::Process()
{
  if (state == kReady)
    return kReady;

  switch(state)
  {
    case kStarting:
      try
      {
        m_Session->SetSucceeded (false);
        if (animationProcessing != NULL)
        {
          UCS2String filename;
          int frame = animationProcessing->GetFrameNumber() - animationProcessing->GetStartFrame() ;
          options = animationProcessing->GetFrameRenderOptions ();
          if (m_Session->OutputToFileSet())
          {
            filename = imageProcessing->GetOutputFilename (options, animationProcessing->GetFrameNumber(), animationProcessing->GetFrameNumberDigits());
            options.SetUCS2String (kPOVAttrib_OutputFile, filename.c_str());

            // test access permission now to avoid surprise later after waiting for
            // the render to complete.
            if (m_Session->TestAccessAllowed(filename, true) == false)
            {
              string str ("IO Restrictions prohibit write access to '") ;
              str += UCS2toASCIIString(filename);
              str += "'";
              throw POV_EXCEPTION(kCannotOpenFileErr, str);
            }
            m_Session->AdviseOutputFilename (filename);
          }
          m_Session->AppendAnimationStatus (frame + 1, animationProcessing->GetTotalFrames(), filename) ;
        }
      }
      catch(pov_base::Exception& e)
      {
        state = kFailed;
        m_Session->SetFailed();
        m_Session->AppendErrorMessage (e.what()) ;
        m_Session->AppendStatusMessage (e.what()) ;
        return kFailed;
      }
      try { sceneId = renderFrontend.CreateScene(backendAddress, options, boost::bind(&vfe::VirtualFrontEnd::CreateConsole, this)); }
      catch(pov_base::Exception& e)
      {
        state = kFailed;
        m_Session->SetFailed();
        m_Session->AppendErrorMessage (e.what()) ;
        m_Session->AppendStatusMessage (e.what()) ;
        return kFailed;
      }
      try { renderFrontend.StartParser(sceneId, options); }
      catch(pov_base::Exception& e)
      {
        state = kFailed;
        m_Session->SetFailed();
        m_Session->AppendErrorMessage (e.what()) ;
        m_Session->AppendStatusMessage (e.what()) ;
        return kFailed;
      }
      state = kParsing;
      return kParsing;

    case kParsing:
      switch(renderFrontend.GetSceneState(sceneId))
      {
        case SceneData::Scene_Failed:
          state = kStopped;
          m_Session->SetFailed();
          return kStopped;

        case SceneData::Scene_Stopping:
          state = kStopping;
          return kStopping;

        case SceneData::Scene_Ready:
          try { viewId = renderFrontend.CreateView(sceneId, options, imageProcessing, boost::bind(&vfe::VirtualFrontEnd::CreateDisplay, this, _1, _2, _3)); }
          catch(pov_base::Exception& e)
          {
            state = kFailed;
            m_Session->SetFailed();
            m_Session->AppendErrorMessage (e.what()) ;
            m_Session->AppendStatusMessage (e.what()) ;
            return kFailed;
          }
          try { renderFrontend.StartRender(viewId, options); }
          catch(pov_base::Exception& e)
          {
            m_Session->ClearStatusMessages();
            if (e.codevalid() && e.code() == kImageAlreadyRenderedErr)
            {
              // this is not a failure; continue has been requested and
              // the file has already been rendered, so we skip it.
              m_Session->AppendStatusMessage ("File already rendered and continue requested; skipping.") ;
              m_Session->AppendStreamMessage (vfeSession::mInformation, "File already rendered and continue requested; skipping.") ;
              if ((animationProcessing != NULL) && (animationProcessing->MoreFrames() == true))
              {
                animationProcessing->ComputeNextFrame();
                m_Session->SetPixelsRendered(0, m_Session->GetTotalPixels());
                m_Session->SetPercentComplete(0);
                state = kStarting;
                return kStarting;
              }
              else
              {
                state = kDone;
                m_Session->SetSucceeded (true);
                return kDone;
              }
            }

            state = kFailed;
            m_Session->SetFailed();
            m_Session->AppendErrorMessage (e.what()) ;
            m_Session->AppendStatusMessage (e.what()) ;
            return kFailed;
          }
          // now we display the render window, if enabled
          shared_ptr<Display> display(GetDisplay());
          if (display != NULL)
          {
            vfeDisplay *disp = dynamic_cast<vfeDisplay *>(display.get());
            if (disp != NULL)
              disp->Show () ;
          }
          state = kRendering;
          return kRendering;
      }
      return kParsing;

    case kRendering:
      switch(renderFrontend.GetViewState(viewId))
      {
        case ViewData::View_Failed:
          state = kStopped;
          m_Session->SetFailed();
          return kStopped;

        case ViewData::View_Stopping:
          state = kStopping;
          return kStopping;

        case ViewData::View_Rendered:
          try
          {
            if(animationProcessing != NULL)
            {
              m_Session->AdviseOutputFilename (imageProcessing->WriteImage(options, animationProcessing->GetFrameNumber(), animationProcessing->GetFrameNumberDigits()));
              m_Session->AdviseFrameCompleted();
            }
            else
              m_Session->AdviseOutputFilename (imageProcessing->WriteImage(options));
          }
          catch (pov_base::Exception& e)
          {
            state = kFailed;
            m_Session->SetFailed();
            m_Session->AppendErrorMessage (e.what()) ;
            m_Session->AppendStatusMessage (e.what()) ;
            // TODO: perhaps we should allow them to pause the queue/insert render
            //       here if need be.
            return kFailed;
          }
          if ((animationProcessing != NULL) && (animationProcessing->MoreFrames() == true))
          {
            try { renderFrontend.CloseView(viewId); }
            catch (pov_base::Exception&) { /* Ignore any error here! */ }
            try { renderFrontend.CloseScene(sceneId); }
            catch (pov_base::Exception&) { /* Ignore any error here! */ }
            animationProcessing->ComputeNextFrame();
            if (m_Session->GetPauseWhenDone())
            {
              // wait for a manual continue
              m_PausedAfterFrame = true;
              state = kPausedRendering;
              return kPausedRendering;
            }
            else
            {
              state = kStarting;
              return kStarting;
            }
          }
          else
          {
            state = kDone;
            m_Session->SetSucceeded (true);
            return kDone;
          }
      }
      return kRendering;

    case kStopping:
      if(renderFrontend.GetSceneState(sceneId) == SceneData::Scene_Ready || renderFrontend.GetSceneState(sceneId) == SceneData::Scene_Failed)
      {
        state = kStopped;
        return kStopped;
      }
      else if(renderFrontend.GetViewState(viewId) == ViewData::View_Rendered || renderFrontend.GetViewState(viewId) == ViewData::View_Failed)
      {
        state = kStopped;
        return kStopped;
      }
      return kStopping;

    case kFailed:
      m_Session->SetFailed();
      // FALL THROUGH

    case kStopped:
    case kDone:
      try { renderFrontend.CloseView(viewId); }
      catch (pov_base::Exception&) { /* Ignore any error here! */ }
      try { renderFrontend.CloseScene(sceneId); }
      catch (pov_base::Exception&) { /* Ignore any error here! */ }
      animationProcessing.reset () ;
      state = kReady;
      return kReady;
  }

  return state;
}

void VirtualFrontEnd::SetResultPointers(Console **cr, Image **ir, Display **dr)
{
  consoleResult = cr;
  displayResult = dr;
}

bool VirtualFrontEnd::IsPausable (void)
{
  switch (GetState ())
  {
    case kParsing :
    case kPausedParsing :
    case kRendering :
    case kPausedRendering :
         return (true) ;

    default :
         return (false) ;
  }
}

bool VirtualFrontEnd::Paused (void)
{
  switch (GetState ())
  {
    case kPausedParsing :
    case kPausedRendering :
         return (true) ;

    default :
         return (false) ;
  }
}

////////////////////////////////////////////////////////////////////////////////////////
//
// helper funtions
//
////////////////////////////////////////////////////////////////////////////////////////

int Allow_File_Write (const char *Filename, const unsigned int FileType)
{
  return (vfeSession::GetSessionFromThreadID()->TestAccessAllowed(Filename, true));
}

int Allow_File_Write (const unsigned short *Filename, const unsigned int FileType)
{
  return (vfeSession::GetSessionFromThreadID()->TestAccessAllowed(Filename, true));
}

int Allow_File_Read (const char *Filename, const unsigned int FileType)
{
  return (vfeSession::GetSessionFromThreadID()->TestAccessAllowed(Filename, false));
}

int Allow_File_Read (const unsigned short *Filename, const unsigned int FileType)
{
  return (vfeSession::GetSessionFromThreadID()->TestAccessAllowed(Filename, false));
}

FILE *vfeFOpen (const std::basic_string<unsigned short>& name, const char *mode)
{
  return (fopen (UCS2toASCIIString (name).c_str(), mode)) ;
}

bool vfeRemove(const UCS2String& Filename)
{
  return (DELETE_FILE (UCS2toASCIIString (Filename).c_str()) == 0);
}

}

