/*******************************************************************************
 * renderfrontend.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/frontend/renderfrontend.cpp $
 * $Revision: #71 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#include "base/types.h"
#include "base/povms.h"
#include "base/povmscpp.h"
#include "base/povmsgid.h"
#include "base/processoptions.h"
#include "base/platformbase.h"
#include "base/fileinputoutput.h"

#include "frontend/configfrontend.h"
#include "frontend/renderfrontend.h"

#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>

// this must be the last file included
#include "base/povdebug.h"

namespace pov_frontend
{

using namespace std;
using namespace boost;

const int gStreamTypeUtilDataCount = 6;

const POVMSType gStreamTypeUtilData[gStreamTypeUtilDataCount] =
{
	kPOVAttrib_DebugFile,
	kPOVAttrib_FatalFile,
	kPOVAttrib_RenderFile,
	kPOVAttrib_StatisticsFile,
	kPOVAttrib_WarningFile,
	kPOVAttrib_AllFile
};

const char *gStreamDefaultFile[gStreamTypeUtilDataCount] =
{
	"debug.out",
	"fatal.out",
	"render.out",
	"stats.out",
	"warning.out",
	"alltext.out"
};

const int gStreamNumber[gStreamTypeUtilDataCount] =
{
	DEBUG_STREAM,
	FATAL_STREAM,
	RENDER_STREAM,
	STATISTIC_STREAM,
	WARNING_STREAM,
	ALL_STREAM
};

namespace Message2Console
{

void InitInfo(POVMS_Object& cppmsg, TextStreamBuffer *tsb);

void FileMessage(TextStreamBuffer *tsb, int stream, POVMSObjectPtr msg);
const char *GetOptionSwitchString(POVMSObjectPtr msg, POVMSType key, bool defaultstate);

}

class FileTextStreamBuffer : public TextStreamBuffer
{
	public:
		FileTextStreamBuffer(const UCS2 *filename, bool append) : stream(filename, POV_File_Text_Stream, append) { }
		virtual ~FileTextStreamBuffer() { flush(); stream.flush(); }
	protected:
		virtual void lineoutput(const char *str, unsigned int chars) { stream.printf("%s\n", string(str, chars).c_str()); stream.flush(); }
	private:
		OTextStream stream;
};

RenderFrontendBase::RenderFrontendBase(POVMSContext ctx) :
	POVMS_MessageReceiver(ctx),
	context(ctx)
{
	InstallFront(kPOVMsgClass_BackendControl, kPOVMsgIdent_Failed, this, &RenderFrontendBase::HandleMessage);

	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_ParserStatistics, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Warning, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Error, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_FatalError, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Debug, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Progress, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Done, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_SceneOutput, kPOVMsgIdent_Failed, this, &RenderFrontendBase::HandleMessage);

	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_RenderStatistics, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Warning, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Error, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_FatalError, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Debug, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Progress, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Done, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewOutput, kPOVMsgIdent_Failed, this, &RenderFrontendBase::HandleMessage);

	InstallFront(kPOVMsgClass_ViewImage, kPOVMsgIdent_PixelSet, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewImage, kPOVMsgIdent_PixelBlockSet, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewImage, kPOVMsgIdent_PixelRowSet, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewImage, kPOVMsgIdent_RectangleFrameSet, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_ViewImage, kPOVMsgIdent_FilledRectangleSet, this, &RenderFrontendBase::HandleMessage);

	InstallFront(kPOVMsgClass_FileAccess, kPOVMsgIdent_FindFile, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_FileAccess, kPOVMsgIdent_ReadFile, this, &RenderFrontendBase::HandleMessage);
	InstallFront(kPOVMsgClass_FileAccess, kPOVMsgIdent_CreatedFile, this, &RenderFrontendBase::HandleMessage);
}

RenderFrontendBase::~RenderFrontendBase()
{
	// nothing to do
}

void RenderFrontendBase::ConnectToBackend(POVMSAddress backendaddress, POVMS_Object& obj, POVMS_Object *resultobj, shared_ptr<Console>& console)
{
	POVMS_Message msg(obj, kPOVMsgClass_BackendControl, kPOVMsgIdent_InitInfo);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(backendaddress);

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
		throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));

	backendaddresses.insert(backendaddress);

	if(resultobj != NULL)
		*resultobj = result;

	if(console != NULL)
		Message2Console::InitInfo(result, console.get());
}

void RenderFrontendBase::DisconnectFromBackend(POVMSAddress backendaddress)
{
	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_BackendControl, kPOVMsgIdent_Done);
	POVMS_Message result(kPOVObjectClass_ResultData);

	// Do not check here, the backend will check if the request to disconnect is valid! [trf]
	backendaddresses.erase(backendaddress);

	msg.SetDestinationAddress(backendaddress);

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
		throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));
}

RenderFrontendBase::SceneId RenderFrontendBase::CreateScene(SceneData& shd, POVMSAddress backendaddress, POVMS_Object& obj)
{
	if(backendaddresses.find(backendaddress) == backendaddresses.end())
		throw POV_EXCEPTION_STRING("TODO"); // TODO FIXME

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_BackendControl, kPOVMsgIdent_CreateScene);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(backendaddress);

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() == kPOVMsgIdent_Done)
	{
		shd.scenepath = Path(obj.TryGetUCS2String(kPOVAttrib_InputFile, ""));
		shd.scenepath.SetFile("");

		shd.outputpath = Path(obj.TryGetUCS2String(kPOVAttrib_OutputPath, ""));
		// TODO FIXME BEGIN - this should not be needed, determine reason and fix [trf]
		if (shd.outputpath.Empty() == true)
		{
			shd.outputpath = Path(obj.TryGetUCS2String(kPOVAttrib_OutputFile, ""));
			shd.outputpath.SetFile("");
			if (shd.outputpath.Empty() == false)
				obj.SetUCS2String(kPOVAttrib_OutputPath, shd.outputpath().c_str());
		}
		// TODO FIXME END

		shd.verbose = obj.TryGetBool(kPOVAttrib_Verbose, true);

		for(size_t i = 0; i < MAX_STREAMS; i++)
		{
			shd.consoleoutput[i] = true;
			shd.streams[i].reset();
		}

		shd.consoleoutput[ALL_STREAM] = false;

		if(obj.Exist(kPOVAttrib_AllConsole))
		{
			bool b = obj.GetBool(kPOVAttrib_AllConsole);
			// NEVERE disable copyright banner BANNER_STREAM stream! [trf]
			shd.consoleoutput[DEBUG_STREAM] = b;
			shd.consoleoutput[FATAL_STREAM] = b;
			shd.consoleoutput[RENDER_STREAM] = b;
			shd.consoleoutput[STATISTIC_STREAM] = b;
			shd.consoleoutput[WARNING_STREAM] = b;
		}
		else
		{
			if(obj.Exist(kPOVAttrib_DebugConsole))
				shd.consoleoutput[DEBUG_STREAM] = obj.GetBool(kPOVAttrib_DebugConsole);
			if(obj.Exist(kPOVAttrib_FatalConsole))
				shd.consoleoutput[FATAL_STREAM] = obj.GetBool(kPOVAttrib_FatalConsole);
			if(obj.Exist(kPOVAttrib_RenderConsole))
				shd.consoleoutput[RENDER_STREAM] = obj.GetBool(kPOVAttrib_RenderConsole);
			if(obj.Exist(kPOVAttrib_StatisticsConsole))
				shd.consoleoutput[STATISTIC_STREAM] = obj.GetBool(kPOVAttrib_StatisticsConsole);
			if(obj.Exist(kPOVAttrib_DebugConsole))
				shd.consoleoutput[WARNING_STREAM] = obj.GetBool(kPOVAttrib_WarningConsole);
		}

		for(size_t i = 0; i < gStreamTypeUtilDataCount; i++)
		{
			if(obj.Exist(gStreamTypeUtilData[i]) == true)
			{
				shd.streamnames[gStreamNumber[i]] = obj.GetUCS2String(gStreamTypeUtilData[i]);
				if(ProcessOptions::IsTrue(UCS2toASCIIString(shd.streamnames[gStreamNumber[i]]).c_str()) == true)
					shd.streamnames[gStreamNumber[i]] = ASCIItoUCS2String(gStreamDefaultFile[i]);
			}
		}

		for(size_t i = 0; i < MAX_STREAMS; i++)
		{
			if(shd.streamnames[i].empty() == false)
			{
				Path path(shd.outputpath);
				path.SetFile(shd.streamnames[i]);
				shd.streams[i].reset(new FileTextStreamBuffer(path().c_str(), obj.TryGetBool(kPOVAttrib_ContinueTrace, false)));
			}
		}

		if(obj.Exist(kPOVAttrib_LibraryPath) == true)
		{
			POVMS_List lps;

			obj.Get(kPOVAttrib_LibraryPath, lps);
			for(int i = 1; i <= lps.GetListSize(); i++)
			{
				POVMS_Attribute lp;

				lps.GetNth(i, lp);
				UCS2String str = lp.GetUCS2String();
				if (str.empty() == true)
					continue;
				if (*str.rbegin() != POV_FILE_SEPARATOR)
					str += POV_FILE_SEPARATOR;
				shd.searchpaths.push_back(Path(str));
			}
		}

		return SceneId(backendaddress, result.GetInt(kPOVAttrib_SceneId));
	}
	else
		throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));
}

void RenderFrontendBase::CloseScene(SceneData& shd, SceneId sid)
{
	if(sid == Id())
		return;

	try
	{
		POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_BackendControl, kPOVMsgIdent_CloseScene);
		POVMS_Message result(kPOVObjectClass_ResultData);

		msg.SetDestinationAddress(sid.GetAddress());
		msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

		POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

		if(result.GetIdentifier() != kPOVMsgIdent_Done)
			throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));
	}
	catch(pov_base::Exception&)
	{
		shd.state = SceneData::Scene_Invalid; // Cannot recover from error! - Current state of backend scene invalid! [trf]
		throw;
	}
}

void RenderFrontendBase::StartParser(SceneData& shd, SceneId sid, POVMS_Object& obj)
{
	if(shd.state != SceneData::Scene_Created)
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(obj, kPOVMsgClass_SceneControl, kPOVMsgIdent_StartParser);

	msg.SetDestinationAddress(sid.GetAddress());
	msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

	POVMS_SendMessage(context, msg, NULL, kPOVMSSendMode_NoReply);

	shd.state = SceneData::Scene_Parsing;
}

void RenderFrontendBase::PauseParser(SceneData& shd, SceneId sid)
{
	if(shd.state != SceneData::Scene_Parsing)
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_SceneControl, kPOVMsgIdent_PauseParser);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(sid.GetAddress());
	msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}

	shd.state = SceneData::Scene_Paused;
}

void RenderFrontendBase::ResumeParser(SceneData& shd, SceneId sid)
{
	if(shd.state != SceneData::Scene_Paused)
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_SceneControl, kPOVMsgIdent_ResumeParser);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(sid.GetAddress());
	msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}

	shd.state = SceneData::Scene_Parsing;
}

void RenderFrontendBase::StopParser(SceneData& shd, SceneId sid)
{
	if((shd.state != SceneData::Scene_Parsing) && (shd.state != SceneData::Scene_Paused))
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_SceneControl, kPOVMsgIdent_StopParser);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(sid.GetAddress());
	msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

	shd.state = SceneData::Scene_Stopping;

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}
}

RenderFrontendBase::ViewId RenderFrontendBase::CreateView(SceneData& shd, ViewData& vhd, SceneId sid, POVMS_Object& obj)
{
	POVMS_Message msg(obj, kPOVMsgClass_SceneControl, kPOVMsgIdent_CreateView);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(sid.GetAddress());
	msg.SetInt(kPOVAttrib_SceneId, sid.GetIdentifier());

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() == kPOVMsgIdent_Done)
		return ViewId(sid.GetAddress(), result.GetInt(kPOVAttrib_ViewId));
	else
		throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));
}

void RenderFrontendBase::CloseView(ViewData& vhd, ViewId vid)
{
	if(vid == Id())
		return;

	try
	{
		POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_SceneControl, kPOVMsgIdent_CloseView);
		POVMS_Message result(kPOVObjectClass_ResultData);

		msg.SetDestinationAddress(vid.GetAddress());
		msg.SetInt(kPOVAttrib_ViewId, vid.GetIdentifier());

		POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

		if(result.GetIdentifier() != kPOVMsgIdent_Done)
			throw POV_EXCEPTION_CODE(result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr));
	}
	catch(pov_base::Exception&)
	{
		vhd.state = ViewData::View_Invalid; // Cannot recover from error! - Current state of backend view invalid! [trf]
		throw;
	}
}

void RenderFrontendBase::StartRender(ViewData& vhd, ViewId vid, POVMS_Object& obj)
{
	if((vhd.state != ViewData::View_Created) && (vhd.state != ViewData::View_Rendered))
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(obj, kPOVMsgClass_ViewControl, kPOVMsgIdent_StartRender);

	msg.SetDestinationAddress(vid.GetAddress());
	msg.SetInt(kPOVAttrib_ViewId, vid.GetIdentifier());

	POVMS_SendMessage(context, msg, NULL, kPOVMSSendMode_NoReply);

	vhd.state = ViewData::View_Rendering;
}

void RenderFrontendBase::PauseRender(ViewData& vhd, ViewId vid)
{
	if(vhd.state != ViewData::View_Rendering)
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_ViewControl, kPOVMsgIdent_PauseRender);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(vid.GetAddress());
	msg.SetInt(kPOVAttrib_ViewId, vid.GetIdentifier());

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}

	vhd.state = ViewData::View_Paused;
}

void RenderFrontendBase::ResumeRender(ViewData& vhd, ViewId vid)
{
	if(vhd.state != ViewData::View_Paused)
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_ViewControl, kPOVMsgIdent_ResumeRender);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(vid.GetAddress());
	msg.SetInt(kPOVAttrib_ViewId, vid.GetIdentifier());

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}

	vhd.state = ViewData::View_Rendering;
}

void RenderFrontendBase::StopRender(ViewData& vhd, ViewId vid)
{
	if((vhd.state != ViewData::View_Rendering) && (vhd.state != ViewData::View_Paused))
		throw POV_EXCEPTION_CODE(kNotNowErr);

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_ViewControl, kPOVMsgIdent_StopRender);
	POVMS_Message result(kPOVObjectClass_ResultData);

	msg.SetDestinationAddress(vid.GetAddress());
	msg.SetInt(kPOVAttrib_ViewId, vid.GetIdentifier());

	vhd.state = ViewData::View_Stopping;

	POVMS_SendMessage(context, msg, &result, kPOVMSSendMode_WaitReply);

	if(result.GetIdentifier() != kPOVMsgIdent_Done)
	{
		int err = result.TryGetInt(kPOVAttrib_ErrorNumber, kNoErr);

		if((err != kNoErr) && (err != kNotNowErr))
			throw POV_EXCEPTION_CODE(err);
	}
}

void RenderFrontendBase::HandleMessage(POVMS_Message& msg, POVMS_Message& result, int)
{
	POVMSType ident = msg.GetIdentifier();

	switch(msg.GetClass())
	{
		case kPOVMsgClass_BackendControl:
			if(msg.Exist(kPOVAttrib_ErrorNumber) == true)
				OutputFatalError(msg.TryGetString(kPOVAttrib_EnglishText, "Unknown failure in backend!"), msg.GetInt(kPOVAttrib_ErrorNumber));
			else
				OutputFatalError(msg.TryGetString(kPOVAttrib_EnglishText, "Unknown failure in backend!"), 0);
			break;
		case kPOVMsgClass_SceneOutput:
			HandleParserMessage(SceneId(msg.GetSourceAddress(), msg.GetInt(kPOVAttrib_SceneId)), ident, msg);
			break;
		case kPOVMsgClass_ViewOutput:
			HandleRenderMessage(ViewId(msg.GetSourceAddress(), msg.GetInt(kPOVAttrib_ViewId)), ident, msg);
			break;
		case kPOVMsgClass_ViewImage:
			HandleImageMessage(ViewId(msg.GetSourceAddress(), msg.GetInt(kPOVAttrib_ViewId)), ident, msg);
			break;
		case kPOVMsgClass_FileAccess:
			HandleFileMessage(ViewId(msg.GetSourceAddress(), msg.GetInt(kPOVAttrib_SceneId)), ident, msg, result);
			break;
	}
}

void RenderFrontendBase::MakeBackupPath(POVMS_Object& ropts, ViewData& vd, const Path& outputpath)
{
	vd.imageBackupFile = outputpath;
	vd.imageBackupFile.SetFile((Path(ropts.TryGetUCS2String(kPOVAttrib_OutputFile, ""))).GetFile());
	if(vd.imageBackupFile.GetFile().empty() == true)
		vd.imageBackupFile.SetFile(ropts.TryGetUCS2String(kPOVAttrib_InputFile, "object.pov"));

	vd.imageBackupFile.SetFile(GetFileName(Path(vd.imageBackupFile.GetFile())) + ASCIItoUCS2String(".pov-state"));
}

void RenderFrontendBase::NewBackup(POVMS_Object& ropts, ViewData& vd, const Path& outputpath)
{
	vd.imageBackup.reset();

	MakeBackupPath(ropts, vd, outputpath);
	if(POV_ALLOW_FILE_WRITE(vd.imageBackupFile().c_str(), POV_File_Data_Backup) == false)
		throw POV_EXCEPTION(kCannotOpenFileErr, "Permission denied to create render state output file.");
	vd.imageBackup = shared_ptr<OStream>(POV_PLATFORM_BASE.CreateOStream(POV_File_Data_Backup));
	if(vd.imageBackup != NULL)
	{
		Backup_File_Header hdr;

		if(vd.imageBackup->open(vd.imageBackupFile().c_str()) == false)
			throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot create render state output file.");
		memcpy(hdr.sig, RENDER_STATE_SIG, sizeof(hdr.sig));
		memcpy(hdr.ver, RENDER_STATE_VER, sizeof(hdr.ver));
		if(vd.imageBackup->write(&hdr, sizeof(hdr)) == false)
			throw POV_EXCEPTION(kFileDataErr, "Cannot write header to render state output file.");
		vd.imageBackup->flush();

		// we remove the file now since if the render doesn't complete and the file
		// already exists, an attempt to do a continue later on will skip the render.
		UCS2String filename = ropts.TryGetUCS2String(kPOVAttrib_OutputFile, "");
		if(filename.length() > 0)
		{
			// we do this test even if the file doesn't exist as we need to write there
			// eventually anyhow. might as well test if before the render starts ...
			if(CheckIfFileExists(filename.c_str()))
				POV_UCS2_REMOVE(filename.c_str());
		}
	}
	else
		throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot create render state output file.");
}

void RenderFrontendBase::ContinueBackup(POVMS_Object& ropts, ViewData& vd, ViewId vid, POVMSInt& serial, vector<POVMSInt>& skip, const Path& outputpath)
{
	bool outputToFile = ropts.TryGetBool(kPOVAttrib_OutputToFile, true);

	// Note: due to the fact that tellg() only returns a 32-bit int on some platforms,
	// currently this code will only work properly with a state file that is < 4gb in
	// size, which works out to a render of roughly 16k*16k pixels.

	serial = 0;
	vd.imageBackup.reset();
	MakeBackupPath(ropts, vd, outputpath);

	scoped_ptr<IStream> inbuffer(POV_PLATFORM_BASE.CreateIStream(POV_File_Data_Backup));

	size_t pos = sizeof(Backup_File_Header);

	if(inbuffer != NULL)
	{
		Backup_File_Header hdr;

		if(inbuffer->open(vd.imageBackupFile().c_str()) == true)
		{
			// IOBase::eof() only is based on feof() and will only return
			// true if we have attempted to read past the end of the file.
			// therefore msg.Read() will throw an exception when we try to
			// read from the end of the file, which isn't harmful per se but
			// makes debugging more difficult since it is caught by the VC++
			// IDE, and we don't want to disable catching pov_base::Exception
			// since they are generally useful. therefore we explicitly check
			// for the end of the file.
			inbuffer->seekg (0, IOBase::seek_end);
			POV_LONG end = inbuffer->tellg();
			inbuffer->seekg (0, IOBase::seek_set);

			if (inbuffer->read (&hdr, sizeof (hdr)) == false)
				throw POV_EXCEPTION(kFileDataErr, "Cannot read header from render state file.");
			if (memcmp (hdr.sig, RENDER_STATE_SIG, sizeof (hdr.sig)) != 0)
				throw POV_EXCEPTION(kFileDataErr, "Render state file header appears to be invalid.");
			if (memcmp (hdr.ver, RENDER_STATE_VER, sizeof (hdr.ver)) != 0)
				throw POV_EXCEPTION(kFileDataErr, "Render state file was written by another version of POV-Ray.");

			while(pos < end && inbuffer->eof() == false)
			{
				POVMS_Message msg;

				try
				{
					msg.Read(*(inbuffer.get()));

					// do not render complete blocks again
					if(msg.Exist(kPOVAttrib_PixelId) == true)
					{
						POVMSInt pid = msg.GetInt(kPOVAttrib_PixelId);

						if(pid > (serial + 1))
							skip.push_back(pid);
						else
							serial++;
					}

					HandleImageMessage(vid, msg.GetIdentifier(), msg);
				}
				catch(pov_base::Exception&)
				{
					// ignore all problems, just assume file is broken from last message on
					break;
				}
				pos = inbuffer->tellg();
			}
		}
		else
		{
			// file doesn't exist, we create it via NewBackup
			if (outputToFile == true)
				NewBackup(ropts, vd, outputpath);
			return;
		}
	}
	else
		throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot open state file from previous render.");

	// make sure the input file is closed since we're about to write to it
	inbuffer.reset();

	// if there isn't going to be an output file, we don't write to the state file
	if(outputToFile == true)
	{
		vd.imageBackup = shared_ptr<OStream>(POV_PLATFORM_BASE.CreateOStream(POV_File_Data_Backup));
		if(vd.imageBackup != NULL)
		{
			if(vd.imageBackup->open(vd.imageBackupFile().c_str(), IOBase::append) == false)
				throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot append to state output file.");

			vd.imageBackup->seekg(0, IOBase::seek_end);
			vd.imageBackup->seekg(min((POV_LONG)pos, vd.imageBackup->tellg()), IOBase::seek_set);
		}
		else
			throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot create state output file stream.");
	}
}

namespace Message2Console
{

void InitInfo(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	const int NUMBER_OF_AUTHORS_ACROSS = 4;
	POVMSAttributeList attrlist;
	POVMSAttribute item;
	char charbuf[1024];
	int h, i, j;
	int cnt;
	int l;

	l = 1024;
	charbuf[0] = 0;
	if(POVMSUtil_GetString(msg, kPOVAttrib_CoreVersion, charbuf, &l) == kNoErr)
		tsb->printf("%s\n", charbuf);

	l = 1024;
	charbuf[0] = 0;
	if(POVMSUtil_GetString(msg, kPOVAttrib_EnglishText, charbuf, &l) == kNoErr)
		tsb->printf("%s\n", charbuf);

	tsb->printf("\n");

	tsb->printf("Primary POV-Ray 3.6/3.7 Developers: (Alphabetically)\n");

	if(POVMSObject_Get(msg, &attrlist, kPOVAttrib_PrimaryDevs) == kNoErr)
	{
		cnt = 0;

		if(POVMSAttrList_Count(&attrlist, &cnt) == kNoErr)
		{
			for(i = 0, h = 1; h <= cnt; i++)
			{
				for(j = 0; (j < NUMBER_OF_AUTHORS_ACROSS) && (h <= cnt); j++, h++)
				{
					if(POVMSAttrList_GetNth(&attrlist, h, &item) == kNoErr)
					{
						l = 1023;
						charbuf[0] = 0;
						if(POVMSAttr_Get(&item, kPOVMSType_CString, charbuf, &l) == kNoErr)
							tsb->printf("  %-18s", charbuf);

						(void)POVMSAttr_Delete(&item);
					}
				}
				tsb->printf("\n");
			}
		}

		(void)POVMSAttrList_Delete(&attrlist);
	}

	tsb->printf("\n");
	tsb->printf("Contributing Authors: (Alphabetically)\n");

	if(POVMSObject_Get(msg, &attrlist, kPOVAttrib_ContributingDevs) == kNoErr)
	{
		cnt = 0;

		if(POVMSAttrList_Count(&attrlist, &cnt) == kNoErr)
		{
			for(i = 0, h = 1; h <= cnt; i++)
			{
				for(j = 0; (j < NUMBER_OF_AUTHORS_ACROSS) && (h <= cnt); j++, h++)
				{
					if(POVMSAttrList_GetNth(&attrlist, h, &item) == kNoErr)
					{
						l = 1023;
						charbuf[0] = 0;
						if(POVMSAttr_Get(&item, kPOVMSType_CString, charbuf, &l) == kNoErr)
							tsb->printf("  %-18s", charbuf);

						(void)POVMSAttr_Delete(&item);
					}
				}
				tsb->printf("\n");
			}
		}

		(void)POVMSAttrList_Delete(&attrlist);
	}

	tsb->printf("\n");
	tsb->printf("Other contributors are listed in the documentation.\n");
	tsb->printf("\n");

	if(POVMSObject_Get(msg, &attrlist, kPOVAttrib_ImageLibVersions) == kNoErr)
	{
		cnt = 0;

		if(POVMSAttrList_Count(&attrlist, &cnt) == kNoErr)
		{
			if(cnt > 0)
			{
				tsb->printf("Support libraries used by POV-Ray:\n");

				for(i = 1; i <= cnt; i++)
				{
					if(POVMSAttrList_GetNth(&attrlist, i, &item) == kNoErr)
					{
						l = 1023;
						charbuf[0] = 0;
						if(POVMSAttr_Get(&item, kPOVMSType_CString, charbuf, &l) == kNoErr)
							tsb->printf("  %s\n", charbuf);

						(void)POVMSAttr_Delete(&item);
					}
				}
			}
		}

		(void)POVMSAttrList_Delete(&attrlist);
	}

	POVMSObject_Delete(msg);
}

void ParserOptions(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	POVMSAttribute attr;
	POVMSInt i, i2;
	POVMSFloat f, f2, f3, f4;
	POVMSBool b;
	UCS2 ucs2buf[1024];
	char *t;
	int outputQuality;
	int outputCompression;
	int l;
	char outputFormat;

	tsb->printf("Parser Options\n");

	f = 0.0;
	l = sizeof (ucs2buf);
	ucs2buf[0] = 0;
	(void)POVMSUtil_GetUCS2String(msg, kPOVAttrib_InputFile, ucs2buf, &l);
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_Version, &f) != kNoErr)
		tsb->printf("  Input file: %s\n", UCS2toASCIIString(ucs2buf).c_str());
	else
		tsb->printf("  Input file: %s (compatible to version %1.2f)\n", UCS2toASCIIString(ucs2buf).c_str(), (double)f);
	tsb->printf("  Remove bounds.......%s\n  Split unions........%s\n",
	              GetOptionSwitchString(msg, kPOVAttrib_RemoveBounds, true),
	              GetOptionSwitchString(msg, kPOVAttrib_SplitUnions, false));

	tsb->printf("  Library paths:\n");
	if(POVMSObject_Get(msg, &attr, kPOVAttrib_LibraryPath) == kNoErr)
	{
		int cnt = 0;

		if(POVMSAttrList_Count(&attr, &cnt) == kNoErr)
		{
			POVMSAttribute item;
			int ii;

			for(ii = 1; ii <= cnt; ii++)
			{
				if(POVMSAttrList_GetNth(&attr, ii, &item) == kNoErr)
				{
					l = sizeof(ucs2buf);
					ucs2buf[0] = 0;
					(void)POVMSAttr_Get(&item, kPOVMSType_UCS2String, ucs2buf, &l);
					tsb->printf("    %s\n", UCS2toASCIIString(ucs2buf).c_str());

					(void)POVMSAttr_Delete(&item);
				}
			}
		}

		(void)POVMSAttr_Delete(&attr);
	}

	POVMSObject_Delete(msg);
}

void RenderOptions(POVMS_Object& obj, TextStreamBuffer *tsb)
{
	tsb->printf("----------------------------------------------------------------------------\n");
	tsb->printf("Render Options\n");

	tsb->printf("  Quality: %2d\n", clip(obj.TryGetInt(kPOVAttrib_Quality, 9), 0, 9));

	if(obj.TryGetBool (kPOVAttrib_Bounding, true))
		tsb->printf("  Bounding boxes.......On   Bounding threshold: %d\n", obj.TryGetInt (kPOVAttrib_BoundingThreshold, 3));
	else
		tsb->printf("  Bounding boxes.......Off\n");

	/*
	tsb->printf("  Light Buffer........%s\n", GetOptionSwitchString(msg, kPOVAttrib_LightBuffer, true));
	tsb->printf("  Vista Buffer........%-3s", GetOptionSwitchString(msg, kPOVAttrib_VistaBuffer, true));
	b = false;
	(void)POVMSUtil_GetBool(msg, kPOVAttrib_VistaBuffer, &b);
	if(b == true)
		tsb->printf("  Draw Vista Buffer...%s", GetOptionSwitchString(msg, kPOVAttrib_DrawVistas, false));
	tsb->printf("\n");
*/

	if(obj.TryGetBool(kPOVAttrib_Antialias, false) == true)
	{
		int method = 0;
		if(obj.TryGetBool(kPOVAttrib_Antialias, false) == true)
			method = clip(obj.TryGetInt(kPOVAttrib_SamplingMethod, 1), 0, 2);
		int depth = clip(obj.TryGetInt(kPOVAttrib_AntialiasDepth, 3), 1, 9);
		float threshold = clip(obj.TryGetFloat(kPOVAttrib_AntialiasThreshold, 0.3f), 0.0f, 1.0f);
		float jitter = 0.0f;
		if(obj.TryGetBool(kPOVAttrib_Jitter, true))
			jitter = clip(obj.TryGetFloat(kPOVAttrib_JitterAmount, 1.0f), 0.0f, 1.0f);
		if(jitter > 0.0f)
			tsb->printf("  Antialiasing.........On  (Method %d, Threshold %.3f, Depth %d, Jitter %.2f)\n",
			               method, threshold, depth, jitter);
		else
			tsb->printf("  Antialiasing.........On  (Method %d, Threshold %.3f, Depth %d, Jitter Off)\n",
			               method, threshold, depth);
	}
	else
		tsb->printf("  Antialiasing.........Off\n");
}

void OutputOptions(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	POVMSAttribute attr;
	POVMSInt i, i2;
	POVMSFloat f, f2, f3, f4;
	POVMSBool b;
	char charbuf[1024];
	char *t;
	int outputQuality;
	int outputCompression;
	int l;
	char outputFormat;

	tsb->printf("Image  Output Options\n");

	(void)POVMSUtil_GetInt(msg, kPOVAttrib_Width, &i);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_Height, &i2);
	(void)POVMSUtil_GetFloat(msg, kPOVAttrib_StartRow, &f);
	(void)POVMSUtil_GetFloat(msg, kPOVAttrib_EndRow, &f2);
	(void)POVMSUtil_GetFloat(msg, kPOVAttrib_StartColumn, &f3);
	(void)POVMSUtil_GetFloat(msg, kPOVAttrib_EndColumn, &f4);
	tsb->printf("  Image resolution %d by %d (rows %d to %d, columns %d to %d).\n",
	              (int)i, (int)i2, (int)(f + 1), (int)f2, (int)(f3 + 1), (int)f4);

	if(POVMSUtil_GetInt(msg, kPOVAttrib_OutputFileType, &i) == kNoErr)
		outputFormat = (char)tolower(i);
	if(POVMSUtil_GetInt(msg, kPOVAttrib_BitsPerColor, &i) == kNoErr)
		outputQuality = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_Compression, &i) == kNoErr)
	{
		if(toupper(outputFormat) == 'J')
		{
			outputQuality = i;
			outputQuality = max(0, outputQuality);
			outputQuality = min(100, outputQuality);
		}
	}

	b = false;
	(void)POVMSUtil_GetBool(msg, kPOVAttrib_OutputToFile, &b);
	if(b == true)
	{
		char charbuf2[1024];
		char *al = "";

		l = 1023;
		charbuf2[0] = 0;
// TODO FIXME		(void)POVMSUtil_GetUCS2String(msg, kPOVAttrib_OutputPath, charbuf2, &l);
		l = 1023;
		charbuf[0] = 0;
// TODO FIXME		(void)POVMSUtil_GetUCS2String(msg, kPOVAttrib_OutputFile, charbuf, &l);
		b = false;
		(void)POVMSUtil_GetBool(msg, kPOVAttrib_OutputAlpha, &b);

		if(toupper(outputFormat) == 'J')
		{
			outputCompression = outputQuality;
			outputQuality = 8;
		}

		if(b == true)
		{
			outputQuality *= 4;
			al = " with alpha";
		}
		else
			outputQuality *= 3;

		switch(toupper(outputFormat))
		{
			case 'C': t = "RLE Targa";       break;
			case 'N': t = "PNG";             break;
			case 'J': t = "JPEG";            break;
			case 'P': t = "PPM";             break;
			case 'S': t = "(system format)"; break;
			case 'T': t = "Targa";           break;
			case 'B': t = "BMP";             break;
			case 'E': t = "EXR";             break;
			case 'H': t = "HDR";             break;
			default:  t = "(none)";          break;
		}

		if(toupper(outputFormat) == 'J')
			tsb->printf("  Output file: %s%s, %d bpp, quality %d%s%s %s\n", charbuf2, charbuf, outputQuality, outputCompression, "%", al, t);
		else
			tsb->printf("  Output file: %s%s, %d bpp%s %s\n", charbuf2, charbuf, outputQuality, al, t);
	}

	b = false;
	(void)POVMSUtil_GetBool(msg, kPOVAttrib_Display, &b);
	if(b == true)
	{
		f = 0.0;
		(void)POVMSUtil_GetFloat(msg, kPOVAttrib_DisplayGamma, &f);
		tsb->printf("  Graphic display......On  (gamma: %g)\n", (float)f);
	}
	else
		tsb->printf("  Graphic display......Off\n");

	i = 0;
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_PreviewStartSize, &i);
	if(i > 1)
	{
		i2 = 0;
		(void)POVMSUtil_GetInt(msg, kPOVAttrib_PreviewEndSize, &i2);
		tsb->printf("  Mosaic preview.......On  (pixel sizes %d to %d)\n", (int)i, (int)i2);
	}
	else
		tsb->printf("  Mosaic preview.......Off\n");

	b = false;
	(void)POVMSUtil_GetBool(msg, kPOVAttrib_CreateHistogram, &b);
	if(b == true)
	{
		if(POVMSUtil_GetInt(msg, kPOVAttrib_HistogramFileType, &i) == kNoErr)
		{
			switch(i)
			{
				case 'C':
				case 'c':
					t = "CSV";
					break;
				case 'T':
				case 't':
					t = "TGA";
					break;
				case 'N':
				case 'n':
					t = "PNG";
					break;
				case 'P' :
				case 'p' :
					t = "PPM";
					break;
				case 'B' :
				case 'b' :
					t = "BMP";
					break;
				case 'E' :
				case 'e' :
					t = "EXR";
					break;
				case 'H' :
				case 'h' :
					t = "HDR";
					break;
				case 'S':
				case 's':
					t = "(system format)";
					break;
				default:    
					t = "(none)";
					break;
			}

			i = 0;
			i2 = 0;
			l = 1023;
			charbuf[0] = 0;
			(void)POVMSUtil_GetInt(msg, kPOVAttrib_HistogramGridSizeX, &i);
			(void)POVMSUtil_GetInt(msg, kPOVAttrib_HistogramGridSizeY, &i2);
// TODO FIXME			(void)POVMSUtil_GetUCS2String(msg, kPOVAttrib_HistogramFile, charbuf, &l);
			tsb->printf("  CPU usage histogram..On  (name: %s type: %s, grid: %dx%d)\n",
			              charbuf, t, (int)i, (int)i2);
		}
	}
	else
		tsb->printf("  CPU usage histogram..Off\n");

	tsb->printf("  Continued trace.....%s\n", GetOptionSwitchString(msg, kPOVAttrib_ContinueTrace, false));

	tsb->printf("Information Output Options\n");

	tsb->printf("  All Streams to console.........%s", GetOptionSwitchString(msg, kPOVAttrib_AllConsole, true));
//	if(streamnames[ALL_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[ALL_STREAM]);
//	else
		tsb->printf("\n");

	tsb->printf("  Debug Stream to console........%s", GetOptionSwitchString(msg, kPOVAttrib_DebugConsole, true));
//	if(streamnames[DEBUG_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[DEBUG_STREAM]);
//	else
		tsb->printf("\n");

	tsb->printf("  Fatal Stream to console........%s", GetOptionSwitchString(msg, kPOVAttrib_FatalConsole, true));
//	if(streamnames[FATAL_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[FATAL_STREAM]);
//	else
		tsb->printf("\n");

	tsb->printf("  Render Stream to console.......%s", GetOptionSwitchString(msg, kPOVAttrib_RenderConsole, true));
//	if(streamnames[RENDER_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[RENDER_STREAM]);
//	else
		tsb->printf("\n");

	tsb->printf("  Statistics Stream to console...%s", GetOptionSwitchString(msg, kPOVAttrib_StatisticsConsole, true));
//	if(streamnames[STATISTIC_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[STATISTIC_STREAM]);
//	else
		tsb->printf("\n");

	tsb->printf("  Warning Stream to console......%s", GetOptionSwitchString(msg, kPOVAttrib_WarningConsole, true));
//	if(streamnames[WARNING_STREAM] != NULL)
//		tsb->printf("  and file %s\n", streamnames[WARNING_STREAM]);
//	else
		tsb->printf("\n");

	POVMSObject_Delete(msg);
}

void AnimationOptions(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	POVMSAttribute attr;
	POVMSInt i, i2;
	POVMSFloat f, f2, f3, f4;
	POVMSBool b;
	char charbuf[1024];
	char *t;
	int outputQuality;
	int outputCompression;
	int l;
	char outputFormat;

	i = 1;
	i2 = 1;
	f = 0.0;
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_InitialFrame, &i);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_FinalFrame, &i2);
	(void)POVMSUtil_GetFloat(msg, kPOVAttrib_Clock, &f);
	if((i != 1) || (i2 != 1) || (i != i2) || (f != 0.0))
	{
		tsb->printf("Animation Options\n");
		tsb->printf("  Initial Frame: %8d  Final Frame: %8d\n", (int)i, (int)i2);
		f = 0.0;
		f2 = 0.0;
		(void)POVMSUtil_GetFloat(msg, kPOVAttrib_InitialClock, &f);
		(void)POVMSUtil_GetFloat(msg, kPOVAttrib_FinalClock, &f2);
		tsb->printf("  Initial Clock: %8.3f  Final Clock: %8.3f\n", (float)f, (float)f2);
		tsb->printf("  Cyclic Animation....%s  Field render........%s  Odd lines/frames....%s",
		              GetOptionSwitchString(msg, kPOVAttrib_CyclicAnimation, false),
		              GetOptionSwitchString(msg, kPOVAttrib_FieldRender, false),
		              GetOptionSwitchString(msg, kPOVAttrib_OddField, false));
	}
	else
		tsb->printf("  Clock value: %8.3f  (Animation off)", (float)f);
	tsb->printf("\n");

	POVMSObject_Delete(msg);
}

void ParserStatistics(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSLong ll = 0;
	int l = 0;
	int s = 0;
	int i = 0;

	tsb->printf("----------------------------------------------------------------------------\n");
	tsb->printf("Parser Statistics\n");
	tsb->printf("----------------------------------------------------------------------------\n");

	s = cppmsg.TryGetInt(kPOVAttrib_FiniteObjects, 0);
	i = cppmsg.TryGetInt(kPOVAttrib_InfiniteObjects, 0);
	l = cppmsg.TryGetInt(kPOVAttrib_LightSources, 0);

	tsb->printf("Finite Objects:   %10d\n", s);
	tsb->printf("Infinite Objects: %10d\n", i);
	tsb->printf("Light Sources:    %10d\n", l);
	tsb->printf("Total:            %10d\n", s + i + l);

	if(cppmsg.Exist(kPOVAttrib_BSPNodes) == true)
	{
		tsb->printf("----------------------------------------------------------------------------\n");
		tsb->printf("BSP Split Nodes:  %10d\n", cppmsg.TryGetInt(kPOVAttrib_BSPSplitNodes, 0));
		tsb->printf("BSP Object Nodes: %10d\n", cppmsg.TryGetInt(kPOVAttrib_BSPObjectNodes, 0));
		tsb->printf("BSP Empty Nodes:  %10d\n", cppmsg.TryGetInt(kPOVAttrib_BSPEmptyNodes, 0));
		tsb->printf("BSP Total Nodes:  %10d\n", cppmsg.TryGetInt(kPOVAttrib_BSPNodes, 0));
		tsb->printf("----------------------------------------------------------------------------\n");
		tsb->printf("BSP Objects/Node Average:       %8.2f          Maximum:      %10d\n",
		            cppmsg.TryGetFloat(kPOVAttrib_BSPAverageObjects, 0.0f), cppmsg.TryGetInt(kPOVAttrib_BSPMaxObjects, 0));
		tsb->printf("BSP Tree Depth Average:         %8.2f          Maximum:      %10d\n",
		            cppmsg.TryGetFloat(kPOVAttrib_BSPAverageDepth, 0.0f), cppmsg.TryGetInt(kPOVAttrib_BSPMaxDepth, 0));
		tsb->printf("BSP Max Depth Stopped Nodes:  %10d (%3.1f%%)   Objects/Node:   %8.2f\n",
		            cppmsg.TryGetInt(kPOVAttrib_BSPAborts, 0), cppmsg.TryGetFloat(kPOVAttrib_BSPAverageAborts, 0.0f) * 100.0f,
		            cppmsg.TryGetFloat(kPOVAttrib_BSPAverageAbortObjects, 0.0f));
	}

	tsb->printf("----------------------------------------------------------------------------\n");
}

void RenderStatistics(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	POVMSAttribute attr;
	POVMSLong l, l2;
	POVMSFloat f, f2;
	POV_LONG Pixels_In_Image;
	int i, i2;

	tsb->printf("----------------------------------------------------------------------------\n");

	(void)POVMSUtil_GetInt(msg, kPOVAttrib_Width, &i);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_Height, &i2);
	Pixels_In_Image = (POV_LONG)i * (POV_LONG)i2;

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_Pixels, &l);
	if(Pixels_In_Image > POVMSLongToCDouble(l))
		tsb->printf("Render Statistics (Partial Image Rendered)\n");
	else
		tsb->printf("Render Statistics\n");

	tsb->printf("Image Resolution %d x %d\n", i, i2);

	tsb->printf("----------------------------------------------------------------------------\n");

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_Pixels, &l);
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_PixelSamples, &l2);
	if(POVMSLongToCDouble(l) > 0.5)
		tsb->printf("Pixels:  %15.0f   Samples: %15.0f   Smpls/Pxl: %.2f\n",
		            POVMSLongToCDouble(l), POVMSLongToCDouble(l2), POVMSLongToCDouble(l2) / POVMSLongToCDouble(l));
	else
		tsb->printf("Pixels:  %15.0f   Samples: %15.0f   Smpls/Pxl: -\n",
		            POVMSLongToCDouble(l), POVMSLongToCDouble(l2));

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_Rays, &l);
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_RaysSaved, &l2);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_TraceLevel, &i);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_MaxTraceLevel, &i2);
	tsb->printf("Rays:    %15.0f   Saved:   %15.0f   Max Level: %d/%d\n",
	            POVMSLongToCDouble(l), POVMSLongToCDouble(l2), i, i2);

	tsb->printf("----------------------------------------------------------------------------\n");
	tsb->printf("Ray->Shape Intersection          Tests       Succeeded  Percentage\n");
	tsb->printf("----------------------------------------------------------------------------\n");

	if(POVMSObject_Get(msg, &attr, kPOVAttrib_ObjectIStats) == kNoErr)
	{
		int cnt = 0;

		if(POVMSAttrList_Count(&attr, &cnt) == kNoErr)
		{
			POVMSObject obj;
			int ii, len;
			char str[40];

			for(ii = 1; ii <= cnt; ii++)
			{
				if(POVMSAttrList_GetNth(&attr, ii, &obj) == kNoErr)
				{
					len = 40;
					str[0] = 0;
					(void)POVMSUtil_GetString(&obj, kPOVAttrib_ObjectName, str, &len);
					(void)POVMSUtil_GetLong(&obj, kPOVAttrib_ISectsTests, &l);
					(void)POVMSUtil_GetLong(&obj, kPOVAttrib_ISectsSucceeded, &l2);

					if(POVMSLongToCDouble(l) > 0.5)
					{
						tsb->printf("%-22s  %14.0f  %14.0f  %8.2f\n", str,
						              POVMSLongToCDouble(l), POVMSLongToCDouble(l2),
						              100.0 * POVMSLongToCDouble(l2) / POVMSLongToCDouble(l));
					}

					(void)POVMSAttr_Delete(&obj);
				}
			}
		}

		(void)POVMSAttr_Delete(&attr);
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_IsoFindRoot, &l);
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_FunctionVMCalls, &l2);
	if((POVMSLongToCDouble(l) > 0.5) || (POVMSLongToCDouble(l2) > 0.5))
	{
		tsb->printf("----------------------------------------------------------------------------\n");
    	if(POVMSLongToCDouble(l) > 0.5)
    		tsb->printf("Isosurface roots:   %15.0f\n", POVMSLongToCDouble(l));
    	if(POVMSLongToCDouble(l2) > 0.5)
    		tsb->printf("Function VM calls:  %15.0f\n", POVMSLongToCDouble(l2));
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_CrackleCacheTest, &l);
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_CrackleCacheTestSuc, &l2);
	if((POVMSLongToCDouble(l) > 0.5) || (POVMSLongToCDouble(l2) > 0.5))
	{
		tsb->printf("----------------------------------------------------------------------------\n");
			if(POVMSLongToCDouble(l) > 0.5)
				tsb->printf("Crackle Cache Queries: %15.0f\n", POVMSLongToCDouble(l));
			if(POVMSLongToCDouble(l2) > 0.5)
				tsb->printf("Crackle Cache Hits:    %15.0f (%3.0f percent)\n", POVMSLongToCDouble(l2), 
				            100.0 * POVMSLongToCDouble(l2) / POVMSLongToCDouble(l));
	}

	tsb->printf("----------------------------------------------------------------------------\n");

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_PolynomTest, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_RootsEliminated, &l2);
		tsb->printf("Roots tested:       %15.0f   eliminated:      %15.0f\n",
		              POVMSLongToCDouble(l), POVMSLongToCDouble(l2));
	}

  // TODO FIXME
	//(void)POVMSUtil_GetLong(msg, kPOVAttrib_CallsToNoise, &l);
	//(void)POVMSUtil_GetLong(msg, kPOVAttrib_CallsToDNoise, &l2);
	//tsb->printf("Calls to Noise:     %15.0f   Calls to DNoise: %15.0f\n",
	//              POVMSLongToCDouble(l), POVMSLongToCDouble(l2));
	// tsb->printf("----------------------------------------------------------------------------\n");

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_MediaIntervals, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_MediaSamples, &l2);
		tsb->printf("Media Intervals:    %15.0f   Media Samples:   %15.0f (%4.2f)\n",
		              POVMSLongToCDouble(l), POVMSLongToCDouble(l2), POVMSLongToCDouble(l2) / POVMSLongToCDouble(l));
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_ShadowTest, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_ShadowTestSuc, &l2);

		tsb->printf("Shadow Ray Tests:   %15.0f   Succeeded:       %15.0f\n",
		              POVMSLongToCDouble(l), POVMSLongToCDouble(l2));

		(void)POVMSUtil_GetLong(msg, kPOVAttrib_ShadowCacheHits, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Shadow Cache Hits:  %15.0f\n", POVMSLongToCDouble(l));
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_ReflectedRays, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_InnerReflectedRays, &l2);
		if(POVMSLongToCDouble(l2) > 0)
			tsb->printf("Reflected Rays:     %15.0f   Total Internal:  %15.0f\n",
			              POVMSLongToCDouble(l), POVMSLongToCDouble(l2));
	    else
	    	tsb->printf("Reflected Rays:     %15.0f\n", POVMSLongToCDouble(l));
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_RefractedRays, &l);
	if(POVMSLongToCDouble(l) > 0.5)
		tsb->printf("Refracted Rays:     %15.0f\n", POVMSLongToCDouble(l));

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_TransmittedRays, &l);
	if(POVMSLongToCDouble(l) > 0.5)
		tsb->printf("Transmitted Rays:   %15.0f\n", POVMSLongToCDouble(l));
/*
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_BoundingQueues, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		tsb->printf("Bounding Queues:    %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_BoundingQueueResets, &l);
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_BoundingQueueResizes, &l2);
		tsb->printf("Queue Resets:       %15.0f   Queue Resizes:   %15.0f\n",
		              POVMSLongToCDouble(l), POVMSLongToCDouble(l2));
	}
*/
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_RadGatherCount, &i);
	(void)POVMSUtil_GetInt(msg, kPOVAttrib_RadReuseCount, &i2);
	if((i > 0) || (i2 > 0))
	{
		tsb->printf("----------------------------------------------------------------------------\n");
		tsb->printf("Radiosity samples calculated:  %15d (%.2f %%)\n", i, 100.0 * double(i) / double(i + i2));
		tsb->printf("Radiosity samples reused:      %15d\n", i2);
	}

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_PhotonsShot, &l);
	if(POVMSLongToCDouble(l) > 0.5)
	{
		tsb->printf("----------------------------------------------------------------------------\n");
		tsb->printf("Number of photons shot: %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_PhotonsStored, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Surface photons stored: %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_MediaPhotonsStored, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Media photons stored:   %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_GlobalPhotonsStored, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Global photons stored:  %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_PhotonsPriQInsert, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Priority queue insert:  %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_PhotonsPriQRemove, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Priority queue remove:  %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_GatherPerformedCnt, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Gather function called: %15.0f\n", POVMSLongToCDouble(l));
		(void)POVMSUtil_GetLong(msg, kPOVAttrib_GatherExpandedCnt, &l);
		if(POVMSLongToCDouble(l) > 0.5)
			tsb->printf("Gather radius expanded: %15.0f\n", POVMSLongToCDouble(l));
	}

	tsb->printf("----------------------------------------------------------------------------\n");

	(void)POVMSUtil_GetLong(msg, kPOVAttrib_MinAlloc, &l);
	if (l > 0) // don't count allocs of 0 anyhow
		tsb->printf("Smallest Alloc:     %15.0f bytes\n", POVMSLongToCDouble(l));
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_MaxAlloc, &l);
	if (l > 0)
		tsb->printf("Largest  Alloc:     %15.0f bytes\n", POVMSLongToCDouble(l));

	l = 0;
	l2 = 0;
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_CallsToAlloc, &l);
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_CallsToFree, &l2);
	if(POVMSLongToCDouble(l) > 0.5)
		tsb->printf("Total Alloc calls:  %15.0f         Free calls:%15.0f\n", POVMSLongToCDouble(l), POVMSLongToCDouble(l2));

	l = 0;
	(void)POVMSUtil_GetLong(msg, kPOVAttrib_PeakMemoryUsage, &l);
	if(POVMSLongToCDouble(l) > 0.5)
		tsb->printf("Peak memory used:   %15.0f bytes\n", POVMSLongToCDouble(l));

	tsb->printf("----------------------------------------------------------------------------\n");

	POVMSObject_Delete(msg);
}

void Warning(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;

	FileMessage(tsb, WARNING_STREAM, msg);

	POVMSObject_Delete(msg);
}

void Error(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;

	FileMessage(tsb, WARNING_STREAM, msg);

	POVMSObject_Delete(msg);
}

void FatalError(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POVMSObject msgobj(cppmsg());
	POVMSObjectPtr msg = &msgobj;
	int ret = kNoErr;
	int l = 0;
	int s = 0;

	if(ret == kNoErr)
		FileMessage(tsb, FATAL_STREAM, msg);

	if(ret != kNoErr)
		throw POV_EXCEPTION_CODE(ret);

	POVMSObject_Delete(msg);
}

void FileMessage(TextStreamBuffer *tsb, int stream, POVMSObjectPtr msg)
{
	const int Num_Echo_Lines = 5; // TODO FIXME
	const int output_string_buffer_size = 1024; // TODO FIXME
	char output_string_buffer[output_string_buffer_size]; // TODO FIXME



	POVMSLong ll = 0;
	int ret = kNoErr;
	int l = 0;

	l = output_string_buffer_size;
	output_string_buffer[0] = 0;
 // TODO FIXME	if(POVMSUtil_GetUCS2String(msg, kPOVAttrib_FileName, output_string_buffer, &l) == kNoErr)
	{
		if((POVMSUtil_GetInt(msg, kPOVAttrib_Line, &l) == kNoErr) && ((stream == WARNING_STREAM) || (stream == FATAL_STREAM)))
		{
			if((strlen(output_string_buffer) > 0) && (l > 0))
				tsb->printf("File: %s  Line: %d\n", output_string_buffer, l);
		}
		if(((POVMSUtil_GetLong(msg, kPOVAttrib_FilePosition, &ll) == kNoErr) && (Num_Echo_Lines > 0)) && (stream == FATAL_STREAM))
		{
			tsb->printf("File Context (%d lines):\n", Num_Echo_Lines);
			tsb->printfile(output_string_buffer, ll, -Num_Echo_Lines);
			tsb->printf("\n");
		}
	}

	l = output_string_buffer_size;
	output_string_buffer[0] = 0;
	ret = POVMSUtil_GetString(msg, kPOVAttrib_EnglishText, output_string_buffer, &l);
	if(ret == kNoErr)
		tsb->printf("%s\n", output_string_buffer);

	if(ret != kNoErr)
		throw POV_EXCEPTION_CODE(ret);
}

const char *GetOptionSwitchString(POVMSObjectPtr msg, POVMSType key, bool defaultstate)
{
	POVMSBool b = false;

	if(POVMSUtil_GetBool(msg, key, &b) != kNoErr)
		b = defaultstate;

	if(b == true)
		return ".On ";

	return ".Off";
}

void ParserTime(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POV_LONG i = 0;
	int msec = 0;
	int sec = 0;

	tsb->printf("Parser Time\n");

	POVMS_Object parseTime;
	POVMS_Object boundingTime;

	if(cppmsg.Exist(kPOVAttrib_ParseTime))
	{
		cppmsg.Get(kPOVAttrib_ParseTime, parseTime);
		i = parseTime.TryGetLong(kPOVAttrib_ParseTime, 0);
		sec = int(i / (POV_LONG)(1000));
		msec = int(i % (POV_LONG)(1000));
		tsb->printf("  Parse Time:     %3d hours %2d minutes %2d seconds (%d.%03d seconds)\n", int(sec / 3600), int((sec / 60) % 60), int(sec % 60), sec, msec);
		if(parseTime.Exist(kPOVAttrib_CPUTime) == true)
		{
			i = parseTime.TryGetLong(kPOVAttrib_CPUTime, 0);
			sec = int(i / (POV_LONG)(1000));
			msec = int(i % (POV_LONG)(1000));
			tsb->printf("              using %d thread(s) with %d.%03d CPU-seconds total\n", int(parseTime.TryGetInt(kPOVAttrib_TimeSamples, 1)), sec, msec);
		}
		else
			tsb->printf("              using %d thread(s)\n", int(parseTime.TryGetInt(kPOVAttrib_TimeSamples, 1)));
	}
	else
		tsb->printf("  Parse Time:       No parsing\n");

	if(cppmsg.Exist(kPOVAttrib_BoundingTime))
	{
		cppmsg.Get(kPOVAttrib_BoundingTime, boundingTime);
		i = boundingTime.TryGetLong(kPOVAttrib_RealTime, 0);
		sec = int(i / (POV_LONG)(1000));
		msec = int(i % (POV_LONG)(1000));
		tsb->printf("  Bounding Time:  %3d hours %2d minutes %2d seconds (%d.%03d seconds)\n", int(sec / 3600), int((sec / 60) % 60), int(sec % 60), sec, msec);
		if(boundingTime.Exist(kPOVAttrib_CPUTime) == true)
		{
			i = boundingTime.TryGetLong(kPOVAttrib_CPUTime, 0);
			sec = int(i / (POV_LONG)(1000));
			msec = int(i % (POV_LONG)(1000));
			tsb->printf("              using %d thread(s) with %d.%03d CPU-seconds total\n", int(boundingTime.TryGetInt(kPOVAttrib_TimeSamples, 1)), sec, msec);
		}
		else
			tsb->printf("              using %d thread(s)\n", int(boundingTime.TryGetInt(kPOVAttrib_TimeSamples, 1)));
	}
	else
		tsb->printf("  Bounding Time:    No bounding\n");
}

void RenderTime(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
	POV_LONG i = 0;
	int msec = 0;
	int sec = 0;

	tsb->printf("Render Time:\n");

	POVMS_Object photonTime;
	POVMS_Object radiosityTime;
	POVMS_Object renderTime;

	if(cppmsg.Exist(kPOVAttrib_PhotonTime))
	{
		cppmsg.Get(kPOVAttrib_PhotonTime, photonTime);
		i = photonTime.TryGetLong(kPOVAttrib_RealTime, 0);
		sec = int(i / (POV_LONG)(1000));
		msec = int(i % (POV_LONG)(1000));
		tsb->printf("  Photon Time:    %3d hours %2d minutes %2d seconds (%d.%03d seconds)\n", int(sec / 3600), int((sec / 60) % 60), int(sec % 60), sec, msec);
		if(photonTime.Exist(kPOVAttrib_CPUTime) == true)
		{
			i = photonTime.TryGetLong(kPOVAttrib_CPUTime, 0);
			sec = int(i / (POV_LONG)(1000));
			msec = int(i % (POV_LONG)(1000));
			tsb->printf("              using %d thread(s) with %d.%03d CPU-seconds total\n", int(photonTime.TryGetInt(kPOVAttrib_TimeSamples, 1)), sec, msec);
		}
		else
			tsb->printf("              using %d thread(s)\n", int(photonTime.TryGetInt(kPOVAttrib_TimeSamples, 1)));
	}
	else
		tsb->printf("  Photon Time:      No photons\n");

	if(cppmsg.Exist(kPOVAttrib_RadiosityTime))
	{
		cppmsg.Get(kPOVAttrib_RadiosityTime, radiosityTime);
		i = radiosityTime.TryGetLong(kPOVAttrib_RealTime, 0);
		sec = int(i / (POV_LONG)(1000));
		msec = int(i % (POV_LONG)(1000));
		tsb->printf("  Radiosity Time: %3d hours %2d minutes %2d seconds (%d.%03d seconds)\n", int(sec / 3600), int((sec / 60) % 60), int(sec % 60), sec, msec);
		if(radiosityTime.Exist(kPOVAttrib_CPUTime) == true)
		{
			i = radiosityTime.TryGetLong(kPOVAttrib_CPUTime, 0);
			sec = int(i / (POV_LONG)(1000));
			msec = int(i % (POV_LONG)(1000));
			tsb->printf("              using %d thread(s) with %d.%03d CPU-seconds total\n", int(radiosityTime.TryGetInt(kPOVAttrib_TimeSamples, 1)), sec, msec);
		}
		else
			tsb->printf("              using %d thread(s)\n", int(radiosityTime.TryGetInt(kPOVAttrib_TimeSamples, 1)));
	}
	else
		tsb->printf("  Radiosity Time:   No radiosity\n");

	if(cppmsg.Exist(kPOVAttrib_TraceTime))
	{
		cppmsg.Get(kPOVAttrib_TraceTime, renderTime);
		i = renderTime.TryGetLong(kPOVAttrib_RealTime, 0);
		sec = int(i / (POV_LONG)(1000));
		msec = int(i % (POV_LONG)(1000));
		tsb->printf("  Trace Time:     %3d hours %2d minutes %2d seconds (%d.%03d seconds)\n", int(sec / 3600), int((sec / 60) % 60), int(sec % 60), sec, msec);
		if(renderTime.Exist(kPOVAttrib_CPUTime) == true)
		{
			i = renderTime.TryGetLong(kPOVAttrib_CPUTime, 0);
			sec = int(i / (POV_LONG)(1000));
			msec = int(i % (POV_LONG)(1000));
			tsb->printf("              using %d thread(s) with %d.%03d CPU-seconds total\n", int(renderTime.TryGetInt(kPOVAttrib_TimeSamples, 1)), sec, msec);
		}
		else
			tsb->printf("              using %d thread(s)\n", int(renderTime.TryGetInt(kPOVAttrib_TimeSamples, 1)));
	}
	else
		tsb->printf("  Trace Time:       No trace\n");
}

void DebugInfo(POVMS_Object& cppmsg, TextStreamBuffer *tsb)
{
  std::string str = cppmsg.TryGetString(kPOVAttrib_EnglishText, "<Error retrieving debug output>");
	tsb->printf("%s\n", str.c_str());
}

string GetProgressTime(POVMS_Object& obj, POVMSType key)
{
	int sec = int(obj.TryGetLong(kPOVAttrib_RealTime, 0) / (POV_LONG)(1000));
	char buffer[32];

	sprintf(buffer, "%3d:%02d:%02d", int(sec / 3600), int((sec / 60) % 60), int(sec % 60));

	return string(buffer);
}

#if 0
#pragma mark -
#endif

void RenderDone(TextStreamBuffer *tsb, POVMSObjectPtr msg)
{/*
	POVMSObject object;
	int ret = 0;
	int i = 0;

	ret = POVMSObject_Get(msg, &object, kPOVAttrib_AnimationTime);
	if(ret == kNoErr)
		tsb->printf("Total Scene Processing Times\n");

	if(ret == kNoErr)
		ret = POVMSUtil_GetInt(&object, kPOVAttrib_ParseTime, &i);
	if(ret == kNoErr)
		tsb->printf("  Parse Time:  %3d hours %2d minutes %2d seconds (%d seconds)\n", (int)(i / 3600), (int)((i / 60) % 60), (int)(i % 60), (int)i);

	if(ret == kNoErr)
		ret = POVMSUtil_GetInt(&object, kPOVAttrib_PhotonTime, &i);
	if(ret == kNoErr)
		tsb->printf("  Photon Time: %3d hours %2d minutes %2d seconds (%d seconds)\n", (int)(i / 3600), (int)((i / 60) % 60), (int)(i % 60), (int)i);

	if(ret == kNoErr)
		ret = POVMSUtil_GetInt(&object, kPOVAttrib_TraceTime, &i);
	if(ret == kNoErr)
		tsb->printf("  Render Time: %3d hours %2d minutes %2d seconds (%d seconds)\n", (int)(i / 3600), (int)((i / 60) % 60), (int)(i % 60), (int)i);

	if(ret == kNoErr)
		ret = POVMSUtil_GetInt(&object, kPOVAttrib_TotalTime, &i);
	if(ret == kNoErr)
		tsb->printf("  Total Time:  %3d hours %2d minutes %2d seconds (%d seconds)\n", (int)(i / 3600), (int)((i / 60) % 60), (int)(i % 60), (int)i);

	(void)POVMSObject_Delete(&object);

	if(ret != kNoErr)
		throw POV_EXCEPTION_CODE(ret);*/
}

void Progress(TextStreamBuffer *tsb, POVMSObjectPtr msg)
{/*
	POVMSLong ll = 0;
	POVMSBool b = false;
	int ret = kNoErr;
	int l = 0;
	int s = 0;

//	Flush(DEBUG_STREAM);

	ret = POVMSUtil_GetBool(msg, kPOVAttrib_ProgressStatus, &b);
	if(ret == kNoErr)
		ret = POVMSUtil_GetInt(msg, kPOVAttrib_TotalTime, &s);
	if(ret == kNoErr)
	{
		l = 80;

		if(b == false)
		{
			ret = POVMSUtil_GetString(msg, kPOVAttrib_EnglishText, status_string_buffer, &l);
			if(ret == kNoErr)
				tsb->printf("\n%3d:%02d:%02d %s", (int)(s / 3600), (int)((s / 60) % 60), (int)(s % 60), status_string_buffer);
		}
		else // if(opts.Options & VERBOSE) // Should this be part of verbose reporting only or not? I really don't know which way would be better... [trf]
		{
			(void)POVMSUtil_GetString(msg, kPOVAttrib_EnglishText, status_string_buffer, &l);
			tsb->printf("\r%3d:%02d:%02d %s", (int)(s / 3600), (int)((s / 60) % 60), (int)(s % 60), status_string_buffer);
		}

// FIXME		if(opts.Options & VERBOSE)
		{
			// animation frame progress
			if(POVMSUtil_GetInt(msg, kPOVAttrib_FrameCount, &l) == kNoErr)
			{
				if(POVMSUtil_GetInt(msg, kPOVAttrib_AbsoluteCurFrame, &s) == kNoErr)
					tsb->printf(" %d of %d", s, l);
			}
			// parsing progress
			else if((POVMSUtil_GetLong(msg, kPOVAttrib_CurrentTokenCount, &ll) == kNoErr) && (ll > 0))
			{
				tsb->printf(" %ldK tokens", long(((POV_LONG)(ll))/1000));
			}
			// rendering progress
			else if(POVMSUtil_GetInt(msg, kPOVAttrib_CurrentLine, &l) == kNoErr)
			{
				if(POVMSUtil_GetInt(msg, kPOVAttrib_LineCount, &s) == kNoErr)
					tsb->printf(" line %d of %d", l, s);
				if(POVMSUtil_GetInt(msg, kPOVAttrib_MosaicPreviewSize, &l) == kNoErr)
					tsb->printf(" at %dx%d", l, l);
				if(POVMSUtil_GetInt(msg, kPOVAttrib_SuperSampleCount, &l) == kNoErr)
					tsb->printf(", %d supersamples", l);
				if(POVMSUtil_GetInt(msg, kPOVAttrib_RadGatherCount, &l) == kNoErr)
					tsb->printf(", %d rad. samples", l);
			}
			// photon progress
			else if(POVMSUtil_GetInt(msg, kPOVAttrib_TotalPhotonCount, &l) == kNoErr)
			{
				// sorting
				if(POVMSUtil_GetInt(msg, kPOVAttrib_CurrentPhotonCount, &s) == kNoErr)
					tsb->printf(" %d of %d", s, l);
				// shooting
				else
				{
					tsb->printf(" Photons %d", l);
					l = 0;
					(void)POVMSUtil_GetInt(msg, kPOVAttrib_PhotonXSamples, &l);
					s = 0;
					(void)POVMSUtil_GetInt(msg, kPOVAttrib_PhotonYSamples, &s);
					tsb->printf(" (sampling %dx%d)", l, s);
				}
			}
		}
	}

	if(ret != kNoErr)
		throw POV_EXCEPTION_CODE(ret);*/
}

}

}
